# -*- coding: UTF-8 -*-
from memory_inspect.adapter.java_adapter import get_common_adapter
from memory_inspect.check_item.memory_check import (
    ArmMemoryCheckTask,
    PurelyMemoryCheckTask,
)
from memory_inspect.check_item.timed_check_item_base import (
    TimedLogBaseCheckItem,
)
from memory_inspect.utils.six_about import (
    is_support_memory_check,
    get_chip_type,
    is_bom_not_in_white_list,
)
from memory_inspect.utils.constants import CheckResult
from memory_inspect.adapter.java_adapter import get_msg

import cliUtil
import common
import os


def execute(cli):
    return MemoryCheck(py_java_env, PY_LOGGER, cli).execute()


CHECK_MEMORY_FAULT_ALARM_ID = [
            "0xF00ED0006",
            "0xF00ED0005",
            "0xF00ED000F",
            "0xF00ED000E",
        ]


class MemoryCheck(TimedLogBaseCheckItem):
    def do_check(self):
        all_cli_ret = ""
        self.logger.info("begin to execute memory inspect.")

        if not is_support_memory_check(self.dev_type, self.dev_ver):
            self.logger.info("the device not support memory check")
            return CheckResult.PASS, "the device not support memory check", ""

        if is_bom_not_in_white_list(self.sn, self.context.get("chip_type")):
            self.logger.info("bom not in white list.")
            return (
                CheckResult.PASS,
                "bom = {} not in white list".format(self.sn),
                "",
            )
        # 如果有告警 直接报错，工具不继续检查。否则继续检查
        flag, alarm_list, ret, msg = self.check_memory_alarm()
        all_cli_ret += ret
        self.logger.info("alarm check ret:{}, {}.".format(alarm_list, msg))
        if flag is not True:
            return CheckResult.NOT_PASS, all_cli_ret, msg

        if alarm_list:
            return (
                CheckResult.NOT_PASS,
                all_cli_ret,
                get_msg(
                    self.lang,
                    "log.collect.memory.alarm.exist",
                    "\n".join(alarm_list),
                ),
            )
        if self.context.get("chip_type") == "kunpeng":
            status, cli_ret, err_msg = ArmMemoryCheckTask(
                self.context
            ).execute()
        else:
            status, cli_ret, err_msg = PurelyMemoryCheckTask(
                self.context
            ).execute()
        all_cli_ret += cli_ret
        self.logger.info("end to execute memory inspect.")
        return status, all_cli_ret, msg + err_msg

    def add_extra_context(self):
        chip_type = get_chip_type(self.dev_ver)
        self.context["chip_type"] = chip_type
        if chip_type == "purely":
            config_name = "{0}configuration{0}memoryInspectConfig.cfg".format(os.sep)
        else:
            config_name = "{0}configuration{0}x10MemoryInspectConfig.cfg".format(os.sep)
        self.context["config_path"] = (
            get_common_adapter().get_inspect_work_path() + config_name
        )
        self.context["collected_files"] = self.get_collected_files()

    def check_memory_alarm(self):
        """
        先检查告警信息，如果存在告警则不继续检查。
        :return:
        """
        cli = self.context.get("cli")
        flag, ret, msg = cliUtil.enterCliModeFromSomeModel(cli, self.lang)
        if flag is not True and \
                not common.reConnectionCli(cli, common.getLogger(self.logger, __file__)):
            return flag, [], ret, msg

        cmd = "show alarm"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, self.lang)
        if cliUtil.queryResultWithNoRecord(ret):
            return flag, [], ret, msg
        ret_info_list = [
            alarm.get("Sequence")
            for alarm in cliUtil.getHorizontalCliRet(ret)
            if alarm.get("ID") in CHECK_MEMORY_FAULT_ALARM_ID
        ]

        flag, ret_info_list, ret_detail, msg = self.get_alarm_detail_info(cli, ret_info_list)
        return flag, ret_info_list, "{}\n{}".format(ret, ret_detail), msg

    def get_alarm_detail_info(self, cli, alarm_sequence_list):
        """
        获取告警详情
        :param cli: ssh连接
        :param alarm_sequence_list: 告警sequence列表
        :return:
        """
        alarm_detail_info = []
        ret_list = []
        for alarm_sequence in alarm_sequence_list:
            cmd = "show alarm sequence={}".format(alarm_sequence)
            flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, self.lang)
            ret_list.append(ret)
            if flag is not True:
                return flag, [], "\n".join(ret_list), msg
            data_dict = self.get_detail(ret)
            alarm_detail_info.append("Alarm ID:{}, Detail:{}".format(
                data_dict.get("ID"), data_dict.get("Detail")))
        return True, alarm_detail_info, "\n".join(ret_list), ""

    def get_detail(self, ret):
        """
        Detail : Interface module (controller enclosure CTE0, interface module B.IOM1) is removed.
        All connections to the ports on this module are disconnected and
                          services cannot be carried.
        处理detail有换行的场景
        :param ret: 回文
        :return: 解析后的detail
        """
        end_flag = ":/>"
        ret_list = ret.splitlines()
        data_dict = {}
        data_key = ''
        data_value = ''
        for line in ret_list:
            if not line.strip():
                continue
            if end_flag in line:
                return data_dict
            tmp_list = line.split(":")
            if len(tmp_list) >= 2:
                data_key, data_value = tmp_list[0].strip(), ":".join(
                    tmp_list[1:]).strip()
                data_dict[data_key] = data_value
                continue
            if ":" not in line and not data_dict:
                continue
            data_value = tmp_list[0].strip()
            data_dict[data_key] = "{} {}".format(data_dict.get(data_key), data_value)
        return data_dict
