# -*- coding: UTF-8 -*-
import re

import cliUtil
import common

from cbb.frame.base import product
from cbb.frame.base import baseUtil
from com.huawei.ism.tool.obase.exception import ToolException
from cbb.frame.util.common import fakeProgress
from common_utils import (
    check_in_vstore_mode_wrap,
    get_err_msg,
)

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

# 进度总剩余时间
LIMIT_TIME = 120
# 进度刷新间隔
INTERVAL = 2


@fakeProgress(py_java_env, totalSeconds=120, logger=LOGGER, interval=1)
def execute(cli):
    """
    高端共享卡端口启动器数量超规格检查：
    1 若步骤2中系统Product Model为：68XX|18XXX V5或OceanStor Dorado
    68XX|8XXX|18XXX V6，则继续检查，否则检查结果为通过;
    2 若步骤3中系统软件版本为V500R007C60SPC200 Kunpeng及之后版本，
    或6.0且低于6.0.1.SPH2，则继续检查，否则检查结果为通过;
    3 若步骤4中，“Number Of Initiators”记录数值中有大于等于509，
    则检查结果为不通过，否则检查结果为通过。
    :param cli:cli链接
    :return:
    """
    ret_list = list()
    try:
        flag, product_model, _, cli_ret, err_msg = \
            common.getProductModelAndCurSysDate(cli, LANG)
        ret_list.append(cli_ret)
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(ret_list),  err_msg

        # 若步骤2中系统Product Model为：68XX|18XXX V5或OceanStor Dorado
        # 68XX|8XXX|18XXX V6，则继续检查，否则检查结果为通过;
        if not baseUtil.isV5V6HighEnd(product_model):
            return True, "\n".join(ret_list), ""

        flag, soft_version, patch_ver, cli_ret_version, err_msg = \
            common.get_soft_and_patch_version(cli, LOGGER, LANG)
        ret_list.append(cli_ret_version)
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(ret_list), err_msg
        # 检查是否在风险版本中
        if not check_soft_version(product_model, soft_version, patch_ver):
            return True, "\n".join(ret_list), ""

        # 检查是否启动器超规格
        err_msg_list = check_fc_ports(cli, ret_list)
        if err_msg_list:
            return False, "\n".join(ret_list), "".join(err_msg_list)
        return True, "\n".join(ret_list), ""
    except common.UnCheckException as uncheck:
        return (cliUtil.RESULT_NOCHECK, "\n".join(ret_list),
                uncheck.errorMsg)
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, "\n".join(ret_list), common.getMsg(
            LANG, "query.result.abnormal")


def check_soft_version(product_model, soft_version, patch_ver):
    """
    检查是否为风险版本，若步骤3中系统软件版本为V500R007C60SPC200 Kunpeng
    及之后版本，或6.0且低于6.0.1.SPH2，则继续检查，否则检查结果为通过;
    :param product_model: 型号
    :param soft_version: 版本
    :param patch_ver: 补丁
    :return: True：是风险版本
    """
    product_version = str(py_java_env.get("devInfo").getProductVersion())
    if product.isKunpeng(
            product_version) and 'V500R007C60SPC200' <= soft_version:
        return True
    if product.isDigitalVer(product_version) and "6.0" in product_version:
        if "6.0.1" not in product_version:
            return True
        pattern_hot_patch = re.compile(r'SPH(\d+)',
                                       flags=re.IGNORECASE)
        match_hot_path = pattern_hot_patch.search(patch_ver)
        if match_hot_path and int(
                match_hot_path.group(1)) >= 2:
            return False
        return True
    return False


def check_fc_ports(cli, ret_list):
    """
        高端共享卡端口启动器数量检查：
        “Number Of Initiators”记录数值中有大于等于509，
        则检查结果为不通过，否则检查结果为通过。
    :param cli: cli链接
    :param ret_list: 回文列表
    :param err_meg_list: 异常回文列表
    :return:
    """

    err_meg_list = list()

    cmd = "show port general physical_type=FC"
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ret_list.append(ret)
    if flag is not True:
        raise common.UnCheckException(msg, ret)

    if cliUtil.queryResultWithNoRecord(ret):
        return err_meg_list

    cli_ret_lines_list = cliUtil.getHorizontalCliRet(ret)

    if len(cli_ret_lines_list) == 0:
        LOGGER.logNoPass("Cannot get information about FC port")
        raise common.UnCheckException(ret, common.getMsg(
            LANG, "query.result.abnormal"))

    for infoDict in cli_ret_lines_list:

        initiator_num = infoDict.get("Number Of Initiators", "")
        # “Number Of Initiators”记录数值中有大于等于250
        if initiator_num and int(initiator_num) >= 250:
            initiator_id = infoDict.get("ID", "")
            if initiator_id:
                err_meg_list.append(
                    get_err_msg(LANG, "number.of.initiators.not.pass",
                                (initiator_id, initiator_num)))
    return err_meg_list
