# -*- coding: UTF-8 -*-

import copy

from common import getMsg
from memory_inspect.adapter.java_adapter import get_common_adapter
from memory_inspect.check_item.timed_check_item_base import TimedLogBaseCheckItem
from memory_inspect.log import log_collector
from memory_inspect.log.log_extract import LogExtract
from memory_inspect.utils.constants import CheckResult
from memory_inspect.utils.six_about import is_support_memory_check, get_chip_type, \
    is_bom_not_in_white_list, get_time_by_delta_process


BASE_COLLECT_FILES = {
    "file": {
        'messages': '/OSM/coffer_log/log/cur_debug/messages',
        "SES_log.txt": ['/OSM/coffer_log/log/cur_debug/pangea/SES_log.txt',
                        '/OSM/coffer_log/log/cur_debug/SES_log.txt']
    },
    "dir": {
        "message_his": {
            "path": '/OSM/coffer_log/log/his_debug/message/',
            "file_pattern": "messages_\\d+_(\\d+)_.*",
            "find_pattern": "messages_.+?tgz"
        },
        "ses_his": {
            "path": '/OSM/coffer_log/log/his_debug/ses_log/',
            "file_pattern": "SES_log_(\d{8}).*",
            "find_pattern": "SES_log_.+?tgz"
        },
    }
}

PURELY_MEMORY_COLLECT_FILES = {
    "file": {
        'mcelog': '/OSM/coffer_log/log/cur_debug/mcelog',
        'bios_uart_print.tar.gz': '/OSM/coffer_log/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/bios_uart_print.tar.gz',
        "memory_info": '/OSM/coffer_log/log/cur_debug/pangea/memory_info'
    },
    "dir": {
        "mce_his": {
            "path": '/OSM/coffer_log/log/his_debug/mcelog/',
            "file_pattern": "mcelog_\\d+_(\\d+)_.*",
            "find_pattern": "mcelog_.+?tgz"
        },
        "his_tar": {
            "path": '/OSM/coffer_log/log/his_debug/log_debug/',
            "file_pattern": "his_tar_\\d+_(\\d+)_.*",
            "find_pattern": "his_tar_.+?tgz"
        }
    }
}

ARM_MEMORY_COLLECT_FILES = {
    "file": {
        'rasdaemon.log': '/OSM/coffer_log/log/cur_debug/rasdaemon.log',
        "imu_debug_log.tar.gz": "/OSM/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/imu_debug_log.tar.gz",
        "imu_debug_log_his.tar.gz": "/OSM/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/imu_debug_log_his.tar.gz",
        'bios_uart_print.tar.gz': '/OSM/coffer_log/log/cur_debug/pangea/BMC/tmp/BMCLOG/local/bios_uart_print.tar.gz'
    },
    "dir": {
        "rasdaemon_his": {
            "path": '/OSM/coffer_log/log/his_debug/rasdaemon/',
            "file_pattern": "rasdaemon_\\d+_(\\d+)_.*",
            "find_pattern": "rasdaemon_.+?tgz"
        },
        "his_tar": {
            "path": '/OSM/coffer_log/log/his_debug/log_debug/',
            "file_pattern": "his_tar_\\d+_(\\d+)_.*",
            "find_pattern": "his_tar_.+?tgz"
        }
    }
}


def execute(cli):
    return LogCollectCheck(py_java_env, PY_LOGGER, cli).execute()


class LogCollectCheck(TimedLogBaseCheckItem):
    def do_check(self):
        # 空间大小判断（空间小于1GB，则报错）
        free_space = get_common_adapter().get_free_space(get_common_adapter().get_work_path())
        self.logger.info("free space is {}".format(free_space))
        if free_space < 1024 * 1024 * 1024:
            return CheckResult.NOT_PASS, "the disk space is not enough.", \
                   getMsg(self.lang, "log.collect.space.not.enough")

        collected_logs = {}
        try:
            failed_engines = log_collector.execute(self.context, self._get_collected_log_files())
            self._clear_db_expired_data(self.context.get("sys_time"))

            # 日志解析和入库
            log_path = self.context.get("log_path")
            collected_logs = LogExtract(log_path, self.logger).execute()

            if failed_engines:
                return CheckResult.NOT_PASS, "", \
                       getMsg(self.lang, "log.collect.engine.manage.ip.can.not.reach", str(failed_engines))
            return CheckResult.PASS, "", ""
        finally:
            self.save_share_data_for_all_item({"collected_logs": collected_logs,
                                               "sys_time": self.context.get("sys_time", "")})

    def _clear_db_expired_data(self, sys_time):
        # 清理掉60天之前的过期数据
        format_time = self.format_system_time(sys_time)
        if format_time:
            begin_time = get_time_by_delta_process(format_time, days=-60)
            self.db_service.log_err_tbl.delete_rows(dict(sn=self.sn, log_time__lt=begin_time))

    def _update_check_result(self, status, detail):
        self.logger.info("sn={},status={},detail={}".format(self.sn, status, detail))
        self.db_service.task_check_result.save_check_result(self.sn, status, detail)

    def _get_collected_log_files(self):
        collected_log_files = copy.deepcopy(BASE_COLLECT_FILES)
        chip_type = get_chip_type(self.dev_ver)
        if is_support_memory_check(self.dev_type, self.dev_ver) and not is_bom_not_in_white_list(self.sn, chip_type):
            self.logger.info("need collected memory logs.")
            memory_collect_files = ARM_MEMORY_COLLECT_FILES \
                if get_chip_type(self.dev_ver) == "kunpeng" else PURELY_MEMORY_COLLECT_FILES
            for key in collected_log_files.keys():
                collected_log_files[key].update(memory_collect_files.get(key))
        return collected_log_files

    def add_extra_context(self):
        self.logger.info("no need do anything.")
