# -*- coding: UTF-8 -*-

import re

from cbb.frame.checkitem.context_adapter import InspectContext
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN

import common_utils
import common
from risk_version_config import CPU_VOLTAGE_RISK_MODEL

PY_JAVA_ENV = py_java_env
JAVA_LOGGER = PY_LOGGER
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(JAVA_LOGGER, __file__)

MIN_DATE = "2020-10-12"
MAX_DATE = "2020-12-30"


def execute(cli):
    context = InspectContext(PY_JAVA_ENV, cli)
    inspect_context = context.get_context()
    inspect_context["logger"] = JAVA_LOGGER
    inspect_context["lang"] = LANG
    check_obj = CheckItem(inspect_context, cli)
    status, err_msg = check_obj.check()
    all_ret = check_obj.get_origin_info()
    LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
    return context.get_result(status, err_msg, all_ret)


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.dev_version = ""
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = False
        self.cli = cli
        self.env = PY_JAVA_ENV
        self.local_logger = LOGGER

    def execute(self):
        flag, msg = self.init_version_and_patch()
        if flag is not True:
            return CheckStatus.NOCHECK, msg

        if not self.is_risk_version():
            return CheckStatus.PASS, ""
        return self.check_cpu_manufacture_date()

    def init_version_and_patch(self):
        """
        初始化版本、补丁信息
        :return:
        """
        self.dsl(
            "exec_cli 'show system general'",
        )
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)
        if flag is not True:
            return False, common.getMsg(
                LANG, "cannot.get.hot.patch.version.info")
        return flag, ''

    def check_cpu_manufacture_date(self):
        """
        Manufactured中出厂日期在[2020-10-12,2020-12-30]之间，
        则检查结果建议优化；否则检查通过
        :return:
        """
        ret = self.dsl(
            "exec_cli 'show controller general'",
        )
        res_list = re.compile("Manufactured=(\d{4}-\d{2}-\d{2})").findall(ret)
        for res in res_list:
            if MIN_DATE <= res <= MAX_DATE:
                return CheckStatus.WARNING, common_utils.get_err_msg(
                    LANG, "cpu.voltage.risk.not.pass")
        return CheckStatus.PASS, ''

    def is_risk_version(self):
        """
        若当前阵列软件版本为V500R007C71，且产品型号为2600 V5/5110 V5 Enhanced/
        5110F V5 Enhanced/5210 V5 Enhanced/5210F V5 Enhanced
        :return: True 风险版本, False 非风险。
        """
        product_model = common.getProductModeFromContext(self.env)
        if product_model not in CPU_VOLTAGE_RISK_MODEL:
            return False

        return True
