# -*- coding: UTF-8 -*-
import common
import cliUtil

from com.huawei.ism.exception import IsmException

from frameone.util import contextUtil
from frameone.rest import restUtil

from common import UnCheckException
from common_utils import (
    check_in_vstore_mode_wrap,
    get_err_msg,
)

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

# 进度总剩余时间
LIMIT_TIME = 120
# 进度刷新间隔
INTERVAL = 2
# quota配额类型
QU0TA_TYPE_LIST = ("2", "3")


def execute(cli):
    """
    文件系统用户配额检查
    :param cli:
    :return:
    """
    fs_quota_check = FsQuotaCheck(cli)
    flag, no_check_msg = fs_quota_check.check_fs_user_quota()
    all_ret = "\n".join(fs_quota_check.all_cli_ret_list)
    total_err_msg = (
        no_check_msg + fs_quota_check.err_msg + fs_quota_check.suggess_msg
    )
    if flag is not True:
        return cliUtil.RESULT_NOCHECK, all_ret, total_err_msg

    if fs_quota_check.err_msg:
        return False, all_ret, total_err_msg

    if fs_quota_check.suggess_msg:
        return cliUtil.RESULT_WARNING, all_ret, total_err_msg

    return True, all_ret, ""


class FsQuotaCheck:
    def __init__(self, cli):
        self.cli = cli
        # 返回框架的参数
        self.all_cli_ret_list = []
        self.err_msg = ""
        self.suggess_msg = ""

        data_dict = contextUtil.getContext(PY_JAVA_ENV)
        self.rest = contextUtil.getRest(data_dict)

        # 配额文件系统和控制器的关系
        self.all_vstore_fs_info_dict = {}

        # 每个租户下的文件系统信息
        self.all_vstore_fs_list_dict = {}

        # 非配额文件系统和控制器的关系
        self.no_vstore_fs_dict = {}

        # 文件系统和控制器对应关系
        self.all_fs_ctrl_dict = {}
        # 所有控制器
        self.controller_list = []

    def check_fs_user_quota(self):
        """
        检查主流程入口
        :return:
        """
        try:
            # 进度条刷新
            common.threadUpProcess(PY_JAVA_ENV, LIMIT_TIME, INTERVAL, LOGGER)
            # 进度开始
            common.inProcess(PY_JAVA_ENV)

            # 查询并记录所有的文件系统ID和对应租户，归属控制器信息；
            self.get_file_system_info()
            LOGGER.logInfo("fs info is:{}".format(self.all_fs_ctrl_dict))
            if not self.all_fs_ctrl_dict:
                return True, ""

            # 查询并记录quota配额类型"Quota Type"为"User"或"Group"的文件系统
            self.get_quota_file_system()
            LOGGER.logInfo(
                "vstore fs info is:{}, no vstore fs is".format(
                    self.all_vstore_fs_info_dict, self.no_vstore_fs_dict
                )
            )
            if not self.all_vstore_fs_info_dict and not self.no_vstore_fs_dict:
                return True, ""

            # 查询所有的控制器ID(Controller)
            (
                flag,
                controller_list,
                err_msg,
                cli_ret,
            ) = cliUtil.getControllerIdListWithRet(self.cli, LANG)
            self.all_cli_ret_list.append(cli_ret)
            if flag is not True:
                return flag, err_msg
            self.controller_list = controller_list
            LOGGER.logInfo("all ctrl is:{}".format(self.controller_list))
            # 若Vstore ID为"--"或不存在Vstore ID字段，则执行步骤6，
            # 首先遍历非租户的
            if self.no_vstore_fs_dict and self.is_ad_domain_joined():
                self.check_controller_domain_owner_ctrl(self.no_vstore_fs_dict)

            # 检查租户的控制器IP
            self.check_all_vstore_fs_ctrl_ip()

            return True, ""
        except UnCheckException as un:
            LOGGER.logException(un)
            return cliUtil.RESULT_NOCHECK, un.errorMsg
        except Exception as e:
            LOGGER.logException(e)
            err_key = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(LANG, err_key)
        finally:
            common.finishProcess(PY_JAVA_ENV)
            LOGGER.logInfo("finish process!")
            try:
                if self.rest:
                    self.rest.close()
                LOGGER.logInfo("conn close success, close session!")
            except Exception as exp:
                LOGGER.logError("conn close except: {}".format(str(exp)))

    def check_all_vstore_fs_ctrl_ip(self):
        """
        检查租户的控制器IP
        :return:
        """
        # 遍历租户检查
        if self.all_vstore_fs_list_dict:
            for vstore_id, info_dict in self.all_vstore_fs_list_dict.items():
                param_dict = {}
                param_dict["cli"] = self.cli
                param_dict["vstore_id"] = vstore_id
                param_dict["logger"] = LOGGER
                param_dict["lang"] = LANG
                param_dict["fs_info_dict"] = info_dict
                param_dict["cli_ret_all"] = self.all_cli_ret_list
                # 在当前租户下检查控制器IP
                self.check_vstore_domain_status(param_dict)

    def get_file_system_info(self):
        """
        获取文件系统IDshow file_system general|filterColumn include columnList=ID,
        Vstore\sID,Owner\sController
        或show file_system general|filterColumn include columnList=ID,
        Owner\sController查询并记录所有的文件系统ID和对应租户，归属控制器信息
        :return:
        """
        cmd = (
            "show file_system general|filterColumn include "
            r"columnList=ID,Vstore\sID,Owner\sController"
        )
        cmd_no_vstore = (
            "show file_system general|filterColumn include "
            r"columnList=ID,Owner\sController"
        )
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, LANG
        )
        self.all_cli_ret_list.append(cli_ret)
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd_no_vstore, True, LANG
            )
            self.all_cli_ret_list.append(cli_ret)

        # 某些型号可能不存在文件系统，如果命令不存在，则通过。
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            return

        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)

        file_system_ret_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for file_system_ret in file_system_ret_list:
            file_system_id = file_system_ret.get("ID", "")
            owner_controller = file_system_ret.get("Owner Controller", "")
            vstore_id = file_system_ret.get("Vstore ID", "")
            self.all_fs_ctrl_dict[file_system_id] = {
                "owner_controller": owner_controller,
                "vstore_id": vstore_id,
            }

    def get_quota_file_system(self):
        """
        判断文件系统quota配额类型
        :return: 配置了用户或用户组配额的文件系统
        """
        uri = "FS_QUOTA?PARENTTYPE=40&PARENTID={}&range=[0-10]&filter=QUOTATYPE::{}"
        for fs_id, value_dict in self.all_fs_ctrl_dict.items():
            for quota_type in QU0TA_TYPE_LIST:
                cmd_str = uri.format(fs_id, quota_type)
                try:
                    records = restUtil.CommonRestService.get4Big(self.rest, cmd_str)
                    records = self._filter_quota_type(records)
                    self.all_cli_ret_list.append("{}\n{}\n".format(cmd_str, str(records)))
                except IsmException as exp:
                    LOGGER.logError(u"ismException :{}".format(exp))
                    records = []
                if records:
                    self._check_quota_type(value_dict, fs_id)
                    break
            cliUtil.excuteCmdInCliMode(self.cli, "show cli configuration", True, LANG)

    @staticmethod
    def _filter_quota_type(records, quota_types=QU0TA_TYPE_LIST):
        if not records:
            return []
        return [record for record in records if str(record.get("QUOTATYPE")) in quota_types]

    def _check_quota_type(self, value_dict, fs_id):
        """
        检查是否是用户或用户组的配额
        :param value_dict: 控制器和租户信息
        :param fs_id: 文件系统ID
        :return:
        """
        vstore_id = value_dict.get("vstore_id")
        if vstore_id and vstore_id.isdigit():
            self.all_vstore_fs_info_dict[fs_id] = value_dict
            tmp_dict = self.all_vstore_fs_list_dict.get(vstore_id, {})
            tmp_dict[fs_id] = value_dict
            self.all_vstore_fs_list_dict[vstore_id] = tmp_dict
        else:
            self.no_vstore_fs_dict[fs_id] = value_dict

    def is_ad_domain_joined(self):
        """
        获取ad域信息
        :param cli: cli连接
        :return: 是否包含joined, cli_ret
        """
        cmd = "show domain ad"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, LANG
        )
        self.all_cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)

        record_list = cliUtil.getVerticalCliRet(cli_ret)
        for record in record_list:
            domain_status = record.get("Domain Status", "")
            if domain_status.lower().strip() == "joined":
                return True
        return False

    def check_controller_domain_owner_ctrl(self, fs_info_dict, vs_store_id=""):
        """
        检查每个控制器的IP是否为空
        :param fs_info_dict: 文件系统和控制器对应关系
        :param vs_store_id: 租户id可选。
        :return:
        """
        cmd = "show domain controller controller={}"
        for ctrl_id in self.controller_list:
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd.format(ctrl_id), True, LANG
            )
            self.all_cli_ret_list.append(cli_ret)
            # 如果命令不存在，则继续检查
            if not cliUtil.hasCliExecPrivilege(cli_ret):
                return

            control_list = cliUtil.getHorizontalCliRet(cli_ret)
            # 当只存在一个逻辑IP时，且另一个控制器上未配置IP，会报error，需判定为不通过
            if flag is not True and not control_list:
                self.set_check_result(
                    fs_info_dict, ctrl_id, vs_store_id
                )
            for control in control_list:
                ip_address_str = control.get("IP Address", "")
                if self.is_valid_ip_address(ip_address_str) is True:
                    continue
                self.set_check_result(
                    fs_info_dict, ctrl_id, vs_store_id
                )

    def set_check_result(self, fs_info_dict, ctrl_id, vs_store_id):
        """
        设置检查结果，提取公共方法
        :param fs_info_dict: 文件系统信息
        :param ctrl_id: 控制器ID
        :param vs_store_id: 租户ID
        :param fs_id: 文件系统ID
        :return:
        """
        for fs_id, fs_info in fs_info_dict.items():
            if fs_info.get("owner_controller") != ctrl_id:
                self.suggess_msg = self._set_err_msg(
                    self.suggess_msg, fs_id, ctrl_id, True, vs_store_id,
                )
            else:
                self.err_msg = self._set_err_msg(
                    self.err_msg, fs_id, ctrl_id, False, vs_store_id
                )

    def _set_err_msg(self, msg, fs_id, ctrl_id, sg_flag, vs_store_id):
        """
        设置错误消息
        :param msg: 当前错误消息
        :param fs_id: 文件系统id
        :param ctrl_id: 控制器id
        :param sg_flag: 存在租户时是否建议优化，提示信息不一样
        :param vs_store_id: 租户ID
        :return: 叠加后的错误消息
        """
        if not vs_store_id:
            msg_key = "soft.ware.fs.user.quota.no.vstore"
            if sg_flag:
                msg_key = "soft.ware.fs.user.quota.no.vstore.suggess"
            return msg + get_err_msg(LANG, msg_key, (fs_id, ctrl_id))

        msg_key = "soft.ware.fs.user.quota.has.vstore"
        if sg_flag:
            msg_key = "soft.ware.fs.user.quota.has.vstore.suggess"
        return msg + get_err_msg(LANG, msg_key, (vs_store_id, fs_id, ctrl_id))

    def is_valid_ip_address(self, ip_str):
        """
        检查IP是否存在
        :param ip_str:ip字符串
        :return: TRUE 存在IP，False不存在IP
        """
        ip_str = ip_str.strip()
        if not ip_str:
            LOGGER.logInfo("ipadress str is null")
            return False
        if ":" in ip_str:
            return True
        ip_list = ip_str.split(".")
        if 4 == len(ip_list):
            return True
        LOGGER.logError("not an ip address:{}".format(ip_str))
        return False

    def check_domain_ip_address(self, cmd):
        """
        :param cli: cli连接
        :param cmd: 命令
        :return: 是否ip为空，cli_ret
        """
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, LANG
        )
        self.all_cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        ldap_domain_list = cliUtil.getVerticalCliRet(cli_ret)
        for ldap_domain in ldap_domain_list:
            ip_address_str = ldap_domain.get("IP Address", "")
            if self.is_valid_ip_address(ip_address_str) is False:
                return False, cli_ret
        return True, cli_ret

    @check_in_vstore_mode_wrap
    def check_vstore_domain_status(self, param_dict, *args, **kargs):
        """
        步骤5 进入租户后，执行命令：show domain ad，查询加入AD域状态；
        步骤6 进入租户后，执行命令：show logical_port general，查询逻辑IP信息；
        步骤7 根据步骤6中查询到的逻辑端口名（Logical Port Name）xxx执行命令：
        show logical_port general logical_port_name=xxx，查询逻辑IP的详细信息
        :param param_dict: 参数字典
        :param args: 可变参数
        :param kargs:
        :return:
        """
        vstore_id = param_dict.get("vstore_id")
        fs_info_dict = param_dict.get("fs_info_dict")
        joined_flag = self.is_ad_domain_joined()
        if not joined_flag:
            return

        # 查看每个域控是否都配置了IP，如果没有配置，则不通过
        self.check_controller_domain_owner_ctrl(
            fs_info_dict, vstore_id
        )
