
# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import cliUtil
import common
 
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
# 协议类型
SNMP_PROTOCOL_CMD = "show snmp version"
# 查询usm用户
SNMP_USM_CMD = "show snmp usm"
# 不安全的鉴权算法
USM_AUTH_DICT = ["MD5", "SHA", "SHA224"]
# 不安全的加密算法
USM_PRIV_DICT = ["DES", "3DES"]


def execute(cli):
    """
    功能描述：步骤1 admin用户登录管理节点；
             步骤2 执行命令show snmp version查询SNMP配置协议；
             步骤3 执行命令show snmp usm查询USM用户信息；
    参数：cli
    返回值：(执行结果，回显结果，错误提示)
    异常描述：cli执行异常提示查询结果异常
    """
    ret_list = []
    tmp_err_list = []
    try:
        common.refreshProcess(PY_JAVA_ENV, 30, LOGGER)
        if not get_snmp_protocol(cli, ret_list, tmp_err_list):
            return False, "", common.getMsg(LANG, "query.result.abnormal")
        common.refreshProcess(PY_JAVA_ENV, 60, LOGGER)
        if not get_snmp_usm(cli, ret_list, tmp_err_list):
            return False, "", common.getMsg(LANG, "query.result.abnormal")
        common.refreshProcess(PY_JAVA_ENV, 90, LOGGER)
        all_ret = "\n".join(ret_list)
        if tmp_err_list:
            return cliUtil.RESULT_WARNING, all_ret, "".join(tmp_err_list)
        return True, all_ret, ""
    except Exception as exception:
        LOGGER.logException(exception)
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(ret_list),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    finally:
        common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)


def get_snmp_protocol(cli, ret_list, tmp_err_list):
    """
    功能描述：获取snmp协议
    参数：cli cli上下文，ret_list 回显集合，tmp_err_list错误提示集合
    返回值：返回True表示正常流程，返回False表示cli命令失败
    异常描述：NA
    """
    check_ret = cliUtil.excuteCmdInCliMode(cli, SNMP_PROTOCOL_CMD, True, LANG)
    if not check_ret[0]:
        LOGGER.logSysAbnormal()
        return False
    cli_ret = check_ret[1]
    ret_list.append(str(cli_ret))
    if "On" in cli_ret:
        tmp_err_list.append(common.getMsg(LANG, "check_snmp_protocol_result.not.pass"))
    return True


def get_snmp_usm(cli, ret_list, tmp_err_list):
    """
    功能描述：获取snmp的usm用户信息
    参数：cli cli上下文，ret_list 回显集合，tmp_err_list错误提示集合
    返回值：返回True表示正常流程，返回Flase表示执行cli失败
    异常描述：NA
    """
    check_ret = cliUtil.excuteCmdInCliMode(cli, SNMP_USM_CMD, True, LANG)
    if not check_ret[0]:
        LOGGER.logSysAbnormal()
        return False
    cli_ret = check_ret[1]
    ret_list.append(str(cli_ret))
    if cliUtil.queryResultWithNoRecord(cli_ret):
        # 如果没有回显则没有配置usm用户，返回true
        return True
    cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
    for line in cli_ret_lines_list:
        user_name = line.get("User Name")
        auth_protocol = line.get("Authenticate Protocol")
        priv_protocol = line.get("Private Protocol")
        # 有记录但是没有设置鉴权算法
        if auth_protocol == "NONE":
            tmp_err_list.append(common.getMsg(LANG, "check_snmp_usm_auth.not.pass", user_name))
        # 鉴权算法等级不够
        if auth_protocol in USM_AUTH_DICT:
            tmp_err_list.append(common.getMsg(LANG, "check_snmp_usm_auth_result.not.pass", (user_name, auth_protocol)))
        # 有记录但是没有加密算法
        if priv_protocol == "NONE":
            tmp_err_list.append(common.getMsg(LANG, "check_snmp_usm_priv.not.pass", user_name))
        # 加密算法等级不够
        if priv_protocol in USM_PRIV_DICT:
            tmp_err_list.append(common.getMsg(LANG, "check_snmp_usm_priv_result.not.pass", (user_name, priv_protocol)))
    return True