# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
"""
此模块针对不合理的历史依赖提供适配层。
需要适配的2种情形：
1、需要使用java侧的方法
2、需要使用老的python模块中的一些方法，但原方法所在的模块存在不合理的依赖
好处：
1、便于单元测试。 2、封装变化，后续架构优化可以做到上层不感知，如去Jython
"""


class DbAdapter(object):
    """
    DB相关操作的适配器，封装java侧DB相关的操作
    """
    _instance = None

    def __new__(cls, *args, **kwargs):
        if cls._instance is None:
            cls._instance = super(DbAdapter, cls).__new__(cls, *args, **kwargs)
        return cls._instance

    def __init__(self):
        from com.huawei.ism.tool.obase.db import SqliteManager as sqlite_manager
        self.sqlite = sqlite_manager()

    def get_connection(self, db_path):
        return self.sqlite.getConnection(db_path)

    def query(self, conn, sql):
        return self.sqlite.query(conn, sql)

    def query_with_params(self, conn, sql, *params):
        return self.sqlite.queryWithParams(conn, sql, *params)

    def update(self, conn, sql):
        return self.sqlite.update(conn, sql)

    def update_batch(self, conn, sql, params):
        return self.sqlite.updateBatch(conn, sql, params)

    def update_with_params(self, conn, sql, *params):
        return self.sqlite.updateWithParams(conn, sql, *params)

    def insert_or_update_query(self, conn, sql_str):
        res = self.sqlite.updateAndGetIds(conn, sql_str)
        return res[-1] if res else None

    def fetch_one(self, conn, sql_str):
        res = self.sqlite.fetchOneForValues(conn, sql_str)
        return res[0] if res else None

    def fetch_many(self, conn, sql_str):
        res = self.sqlite.fetchOneForValues(conn, sql_str)
        return res[0] if res else None


class CommonAdapter(object):
    """
    java基础接口适配器，比如工作目录、空间大小等
    """
    _instance = None

    def __new__(cls, *args, **kwargs):
        if cls._instance is None:
            cls._instance = super(CommonAdapter, cls).__new__(cls, *args, **kwargs)
        return cls._instance

    @staticmethod
    def get_free_space(path):
        from java.io import File
        return File(path).getFreeSpace()

    @staticmethod
    def get_check_period_start_time():
        from com.huawei.ism.tool.inspect.utils import InspectContext
        return InspectContext.getInstance().getCheckPeriodStartTime()

    @staticmethod
    def get_memory_task_db_path():
        from com.huawei.ism.tool.inspect.utils import InspectContext
        return InspectContext.getInstance().getMemoryTaskDbPath()

    @staticmethod
    def get_work_path():
        from com.huawei.ism.tool.inspect.utils import InspectContext
        return InspectContext.getInstance().getWorkPath()

    @staticmethod
    def get_inspect_work_path():
        from com.huawei.ism.tool.obase.utils import ApplicationContext
        return ApplicationContext.getInstance().getWorkPath()


def get_db_adapter():
    return DbAdapter()


def get_common_adapter():
    return CommonAdapter()


def get_uncheck_exception(err_msg, cli_ret):
    from common import UnCheckException
    return UnCheckException(err_msg, cli_ret)


class UnCheckException(Exception):
    def __init__(self, error_msg, cli_ret):
        self.error_msg = error_msg
        self.cli_ret = cli_ret


def get_msg(lang, msg_key, *args):
    from common import getMsg
    return getMsg(lang, msg_key, args)


def get_lang(context):
    from common import getLang
    return getLang(context)
