# -*- coding: UTF-8 -*-
import re
NOCHECK = 'NOCHECK'
CLI_RET_END_FLAG = ":/>"

# ************************************** #
# 函数名称: formatList
# 功能说明: 将list样式字符串转换成list对象
# 输入参数: descStr字符串
# 输出参数: 由多个dict对象组成的list
# ************************************** #

def formatList(descStr):
    
    # 由---和空格构成行的行号，大于等于1
    REF_LINE_NO = None                              
    # 最后需要返回的结果
    result = []
    # 存放各列数据的开始位置/结束位置
    valsIndex = []
    # 将字符串拆成单行并存放在list对象中
    lsts = descStr.splitlines()                    
    #正则表达式，以空格隔开的由多个-构成的字符串（首尾空格不限）
    pattern = re.compile("^\s*-+(\s+-+)*\s*$")    

    # 尝试找到符合正则表达式的行
    for i in xrange(len(lsts)):
        if  pattern.match(lsts[i]):
            # 确定行号
            REF_LINE_NO = i
            # 确定列取值的开始位置
            valsIndex.append(lsts[i].find("-"))
            break
        
    # 如果没有找到符合正则表达式的行号，意味着格式不符合，不再继续解析，直接返回空结果    
    if not REF_LINE_NO:
        return result

    # 取出---和空格构成的行
    strVals = lsts[REF_LINE_NO] 

    while valsIndex[-1] != -1:
        #依次取出“ -”所在的位置并添加到valsIndex，直到取到-1为止
        valsIndex.append(strVals.find(" -", valsIndex[-1] + 1))    
    #将最后一位设为字符串长度，最后一次取值直接取到末尾
    valsIndex[-1] = len(strVals)
    
    # 在各dict中作为键值的list
    keys = []   
    # 计算key值，以valsIndex中的两个相临值作为开始位置和结束位置，并去除空格
    for i in xrange(1, len(valsIndex), 1):
        key = lsts[REF_LINE_NO - 1][valsIndex[i - 1]:valsIndex[i]].strip()
        #判断是否有重复存在的列名，若存在则追加一个下标
        if keys.count(key) > 0:
            temp = 1
            while keys.count(key + "_" + str(temp)) != 0:
                temp += 1
            key = key + "_" + str(temp)
  
        keys.append(key)

    # 取出---下面的单行进行解析
    for i in xrange(REF_LINE_NO + 1, len(lsts), 1):
        
        # 排除最后一行，若遇到最后一行则退出循环
        if isEndLine(lsts[i]):            
            break
        #遇到空行时忽略
        if re.compile("^\s*$").match(lsts[i]):
            continue
        
        # 单行dict中作为value值的list        
        vals = []    
        # 计算单行的value值，以valsIndex中的两个相临值作为开始位置和结束位置，并去除空格
        for j in xrange(1, len(valsIndex), 1):
            vals.append(lsts[i][valsIndex[j - 1]:valsIndex[j]].strip())

        # 单行value计算完成后，与key值合并成dict添加到result结果集中
        result.append(dict(zip(keys, vals)))  
    
    # 所有行解析完成后返回result结果集
    return result

# **************************************
# 函数名称: formatDict
# 功能说明: 将dict样式字符串转换成list/dict对象
# 输入参数: descStr字符串
#          simpleMode:是否为简单模式，若为True则所有Value值解析成字符串，不会再对其作进一步解析
#          若为False则会对Value作进一步判断，可能会将其进一步转化为dict或list
# 输出参数: 若解析出来只有一个Dict则直接返回Dict对象，
#          若包含多个Dict则返回由多个Dict组成的List对象
# **************************************
def formatDict(descStr, simpleMode=False):
    
    # 最后需要返回的结果
    result = []
    # 将字符串拆成单行并存放在list对象中
    lsts = descStr.splitlines()
    # 用于分隔两个字典的单行，整行都由---构成
    pattern = re.compile("^\s*[-]*-\s*$")
    # 用于匹配”--------  xxx Key : ---------“这类字符串以作为Key值
    pattern_key_str = "^\s*-+\s*([a-zA-z0-9]+\s*)+:\s*-+\s*$"
    pattern_key = re.compile(pattern_key_str)
    # dict中的键集合
    keys = []
    # dict中的值集合
    vals = []                               

    #当前所读取的行号
    i = 0 
    while i < len(lsts):
        
        str_line = lsts[i] #缓存此行
        
        #遇到最后一行则退出while循环
        if isEndLine(str_line):
            break
        
        i += 1 #行号+1，准备读取下一行数据

        # 如果遇到整行都是---，将键值对添加到result中并重新初始化key和val
        if pattern.match(str_line): 
            result.append(dict(zip(keys, vals)))
            keys = []
            vals = []
            continue
        
        # 找到冒号所在的位置   
        index = str_line.find(":")        
        # 当冒号存在时，将冒号前的字符串添加到key中，冒号后的字符串添加到val中，均需要去除前后的字符串
        if index != -1:
            
            #该行中格式类似”------  key:  ------“
            from com.huawei.ism.tool.obase.utils import StringUtils
            if StringUtils.checkRegexReDos(str_line, pattern_key_str) \
                    and pattern_key.match(str_line) and not simpleMode:
                key = str_line[:index].strip('-').strip()
                val = ""
                #追加键
                keys.append(key)
                #此while语句用于拼接字符串
                while i < len(lsts):
                    str_line = lsts[i]

                    if str_line.endswith(":") or pattern_key.match(str_line) or str_line.find(":/>") != -1:
                        break
                    else:
                        val = val + "\r\n" + str_line               
                        i += 1 #行号+1，准备读取下一行数据
                val = formatStr(val) #退出while循环时需要对key值进行格式化,递归调用
                vals.append(val)             
                continue
            else:
                key = str_line[:index].strip()
                #因为有时候键后面为一堆空格，这里为特殊处理，空格不当成空处理，但保存时仍然存空（即else中的处理）
                val = str_line[index + 1:]
                #追加键
                keys.append(key)
                #若同行取到的value值为空，则认为从下一行开始才是该key的value值，在这种情况下value又将是一个Dict或List对象
                if val == "":
                    #此while语句用于拼接字符串，直到再次遇到以冒号结尾或者包含":/>"情况
                    while i < len(lsts):
                        str_line = lsts[i]
                        if str_line.endswith(":") or isEndLine(str_line):
                            break
                        else:
                            val = val + "\r\n" + str_line               
                            i += 1 #行号+1，准备读取下一行数据
                    if not simpleMode:
                        val = formatStr(val) #退出while循环时需要对key值进行格式化,递归调用
                else:
                    val = val.strip() 
                vals.append(val)               
                continue        
        
        # 若某行是[Board Properties]的集合时暂时未做处理，会忽略掉后面的行
        
    #此行已退出while循环
        
    # 退出while循环时判断result是否为空，若为空则返回字典，否则返回由字典组成的序列    
    if len(result) > 0:
        result.append(dict(zip(keys, vals)))
    else:
        result = dict(zip(keys, vals))
    return result

# **************************************
# 函数名称: formatStr
# 功能说明: 在不确定字符串样式自动判断样式并转换成List/Dict对象
# 输入参数: descStr：需要解析的字符串
#          cmdLineFlag：是否包含第一行命令的boolean值
# 输出参数: 若传入空字符串也返回空字符串
#          若解析出来只有一个Dict则直接返回Dict对象，
#          若包含多个Dict则返回由多个Dict组成的List对象
# **************************************
def formatStr(descStr):
    
    pattern = re.compile("^\s*-+(\s+-+)*\s*$") 
    lsts = descStr.strip().splitlines()
    if len(lsts) == 0:
        return ""
    #以show开头的认为是命令行，直接排除
    if lsts[0].startswith("show "):
        lsts.pop(0)
        
    for i in xrange(len(lsts)):
        if "" != lsts[i].strip():
                      
            #若字符串中带有冒号，则认为是
            if lsts[i].find(":") != -1:
                return formatDict(descStr)
            
            #当不带冒号且次行符合由多个---与空格组成的格式
            elif i + 1 < len(lsts) - 1:
                if pattern.match(lsts[i + 1].strip()):                
                    return formatList(descStr)
                
            #若无法确定格式则直接返回该字符串（去除命令行,最后的结束及前后的空白）
            else:
                if isEndLine(lsts[-1]):
                    lsts.pop()
                    
            return "\r\n".join(lsts).strip()

# **************************************
# 函数名称: isEndLine(descStr)
# 功能说明: 判断是否为结束行
# 输入参数: descStr:一行字符串
# 输出参数: Boolean值
# **************************************
def isEndLine(descStr):
    patternEndflag = re.compile(r".*:/.*>.*")
    return patternEndflag.match(descStr)

def getProductModel(cli):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, productmodel)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        productmodel：产品型号
    '''
    product_model = ""
    cliRet = cli.execCmd("show system general")
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, cliRet, product_model)
        
    return (False, cliRet, "")

def getProductFullVersion(cli):
    """
    Function name      : getUserLevel
    Function describe  : get user level
    Input              : (cli)
    Return             : (cmdExecuteSuccess, cliRet, productVersion)
    """    
    cliRet = cli.execCmd("show upgrade package")
  
    endIndex = cliRet.find('HotPatch Version')
    if endIndex == -1:
        return (False, cliRet, '')
    
    try:
        dictList = formatList(cliRet[:endIndex])
        fullVersion = dictList[0]['Current Version']
        if fullVersion.endswith('T'):
            fullVersion = fullVersion[:-1]
    except:
        return (False, cliRet, '')
    else:
        return (True, cliRet, fullVersion)

def getDiskSectorSize(cli):
    '''
    @summary: 执行show disk general |filterColumn include columnList=ID,Sector\sSize命令，获取扇区大小
    '''
    
    sectorSizeDictList = []
    errMsgDict = {"zh":"", "en":""}
    
    cmd = "show disk general |filterColumn include columnList=ID,Sector\sSize"
    cliRet = cli.execCmd(cmd)

    #兼容产品命令Bug
    if not hasCliExecPrivilege(cliRet):
        cmd = "show disk general |filterColumn include colunmList=ID,Sector\sSize"
        cliRet = cli.execCmd(cmd)
  
    if len(cliRet.splitlines()) < 5: 
        errMsgDict["zh"] +=  u"\n获取硬盘扇区大小信息失败。"
        errMsgDict["en"] += "\nFailed to obtain information about the disk sector size."
        return (False, cliRet, [], errMsgDict)
        
    sectorSizeDictList = getHorizontalCliRet(cliRet)
    return (True, cliRet, sectorSizeDictList, errMsgDict)

def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "does not have the permission" in line:
            return False
    return True

def is4KDiskWithSectorSize(diskSectorSizeDictList):
    '''
    @summary: 根据扇区大小判断是否为4K盘
    '''
    is4KDiskList = []
    errMsgDict = {"zh":"", "en":""}
    
    for diskSectorSizeDict in diskSectorSizeDictList:
        diskId = diskSectorSizeDict["ID"]
        sectorSizeStr = diskSectorSizeDict["Sector Size"]
        
        #4196B=4.0625KB，显示为4.062KB，需要处理
        if sectorSizeStr == "4.062KB":
            sectorSizeStr = "4.0625KB"
        result, sectorSizeNum = changUnit2B(sectorSizeStr)
        if not result:
            errMsgDict["zh"] +=  u"\n转换单位失败。"
            errMsgDict["en"] += "\nFailed to convert units."
            return (True, [], errMsgDict)
        
        if sectorSizeNum == 4096.0 or sectorSizeNum == 4160.0:
            is4KDiskList.append(diskId)
        
    if is4KDiskList:
        errMsgDict["zh"] +=  u"\n设备中存在扇区大小为4096B或4160B的硬盘[ID: %s]。" % ', '.join(is4KDiskList)
        errMsgDict["en"] += "\nDisks [ID: %s] whose sector size is 4096 bytes or 4160 bytes exist in the device." % ', '.join(is4KDiskList)
        return (True, is4KDiskList, errMsgDict)
    
    return(False, [], errMsgDict)

def changUnit2B(strValue):
    '''
    @summary: 根据传入的值转换单位为B
    @return: True/False, sectorSizeNum
    '''
    sectorSizeNum = 0.0
    
    try:
        if not strValue:
            return(flag, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024 * 1024 * 1024 * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip()) * 1024 * 1024 * 1024
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) * 1024 * 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) * 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip())
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)
    

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    cliExecuteCmdSuccess = "Command executed successfully"
    if re.search(cliExecuteCmdSuccess, cliRet, re.IGNORECASE):
        return True
    return False


def changUnit2GB(strValue):
    '''
    @summary: 根据传入的值转换单位为GB
    @return: True/False, floatValue
    '''
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip())
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) / 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) / 1024 / 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip()) / 1024 / 1024 / 1024 
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception:
        return (False, floatValue)

    
def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        
        if re.search("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())
    
    return dictList


def getLogicEngineOfDisk(diskId, productModel):
    """
    @summary: 获取硬盘所属的逻辑引擎
    """
    #硬盘id的第一个数字即为该硬盘所属的逻辑引擎
    firstDigit = re.search('([A-Z]+)(\d)', diskId).group(2)
    if firstDigit is None:
        return 0
    
    firstDigit = int(firstDigit)
    
    logicEngine = firstDigit
    
    #6U4C大卡环境逻辑i引擎需特殊处理
    bigCardModelList = ["6800 V3", "6900 V3", "18500 V3", "18800 V3"]
    if productModel in bigCardModelList:
        secondDigit = re.search('([A-Z]+)(\d)(\d|[A-F])', diskId).group(3)
        if secondDigit is None:
            return firstDigit
        
        if secondDigit <= "7":
            logicEngine = firstDigit * 2 + 0
        else:
            logicEngine = firstDigit * 2 + 1
    
    return logicEngine


def isBugProductVersion(currentVersion, bugProductVersionList):
    '''
    @summary: 判断产品版本是否是某个已知产品问题的产品版本
    @param: bugProductVersionList
    eg:("V300R003C00", "V300R003C00SPC300]", "V300R003C10.*"]/V300R003C00到V300R003C00SPC300（不包含V300R003C00，包含V300R003C00SPC300），V300R003C10所有版本
    '''
    for bugProductVersion in bugProductVersionList:
        if bugProductVersion.find(",") > -1:
            startVersion = bugProductVersion.split(",")[0][1:].strip()
            endVersion = bugProductVersion.split(",")[1][:-1].strip()
            isContainStartVersion = bugProductVersion.startswith("[")
            isContainEndVersion = bugProductVersion.endswith("]")
            
            if isContainStartVersion and isContainEndVersion:
                if currentVersion >= startVersion and currentVersion <= endVersion:
                    return True
            
            if isContainStartVersion and (not isContainEndVersion):
                if currentVersion >= startVersion and currentVersion < endVersion:
                    return True
            
            if (not isContainStartVersion) and isContainEndVersion:
                if currentVersion > startVersion and currentVersion <= endVersion:
                    return True
            
            if (not isContainStartVersion) and (not isContainEndVersion):
                if currentVersion > startVersion and currentVersion < endVersion:
                    return True
        else:
            if re.match(bugProductVersion, currentVersion):
                return True
    
    return False


def checkProductModelAndVersion(cli, modelAndVersionListDict):
    """
    @summary: 检查设备型号和版本是否为指定的型号和版本
    @param modelAndVersionListDict: 问题型号与对应的版本，型号与版本为正则表达式，若问题版本为某个区间，用数学表达式区间表示，
            eg:{"(?!2800 V3)" : ["V200R002C20.*", "[V200R002C30,V200R002C30SPC200]"}，表示问题型号为非2800 V3，
                                        问题版本为所有V200R002C20版本，V200R002C30到V200R002C30SPC200版本，包含问题版本为所有V200R002C20版本与V200R002C30SPC200。
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLID回显
        isPass：True/False，检查是否通过（设备型号与版本不是问题版本时检查通过）
        productModel：设备型号
        productVersion：设备版本
    """
    isSucc = True
    allCliRet = ""
    
    isPass = True
    productModel = ""
    productVersion = ""
    
    #获取产品型号
    flag, cliRet, productModel = getProductModel(cli)
    allCliRet += cliRet
    if flag != True:
        isSucc = False
        isPass = False
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    #获取产品软件版本
    flag, cliRet, productVersion = getProductFullVersion(cli)
    allCliRet += cliRet
    if flag != True:
        isSucc = False
        isPass = False
        return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    
    for model in modelAndVersionListDict:
        if not re.match(model, productModel):
            #设备型号不是问题型号
            continue
        
        versionList = modelAndVersionListDict.get(model)
        
        if isBugProductVersion(productVersion, versionList):
            #问题型号与版本均匹配
            isSucc = True
            isPass = False
            return (isSucc, allCliRet, isPass, productModel, productVersion)
        else:
            return (isSucc, allCliRet, isPass, productModel, productVersion)
    
    return (isSucc, allCliRet, isPass, productModel, productVersion)
