# -*- coding:utf-8 -*-
import time
import os
from cbb.frame.util.tar_util import decompress_tar_all_file_with_detail
from cbb.frame.base import baseUtil
import shutil
import re

import cliUtil
import common
from common_utils import get_err_msg, is_flush_through_mode
from cbb.frame.base import jsonUtil
from frameone.util import contextUtil
import java.util.HashMap as HashMap


PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
context = contextUtil.getContext(PY_JAVA_ENV)

KEY_DEDUP_ENABLED = 'Dedup Enabled'
KEY_COMPRESS_ENABLED = 'Compression Enabled'
KEY_FEATURE_NAME = 'Feature Name'
KEY_LIC_CREATE_TIME = 'Created On'
KEY_PRE_UPG_VER = "pre_spc_ver"
KEY_POST_UPG_VER = "post_spc_ver"
KEY_UPG_TIME = 'finish_time'
KEY_EVENT_TIME = 'Occurred On'
KEY_LUN_ID = 'ID'
KEY_LUN_WWN = 'WWN'
KEY_LUN_CREATE_TIME = 'TIME'
VALUE_EFFECTIVE_CAPACITY = 'Effective Capacity'
VALUE_DEDUP_COMPRESS = 'SmartDedupe & SmartCompression'
VALUE_CREATE_LUN_EVENT = 'succeeded in creating LUN ('  # '0x200F000B0285' 或 '0x200F000B0224'
VALUE_LUN_FLAG = 'Lun('
VALUE_NAMESPACE_FLAG = 'Namespace('
VALUE_DEDUP_COMPRESS_IN_FILE = 'Feature=GDCYZ'


def execute(cli):
    enclosures_check = DedupeCompressionCheck(cli)
    flag, err_msg = enclosures_check.execute_check()
    return flag, "\n".join(enclosures_check.cli_ret_list), err_msg


class DedupeCompressionCheck:
    def __init__(self, cli):
        self.cli = cli
        self.cli_ret_list = []
        self.product_version = ""
        self.product_model = ""
        self.local_path = None
        self.rest = None

    def execute_check(self):
        # 直通模式为不涉及：
        self.product_model = str(PY_JAVA_ENV.get("devInfo").getDeviceType())
        if baseUtil.is_micro_dev(self.product_model) and is_flush_through_mode(self.cli, LANG, LOGGER):
            self.cli_ret_list.append("System mode: pass-through mode.")
            return cliUtil.RESULT_NOSUPPORT, ""

        # 获取当前版本
        flag, self.product_version, cli_ret, err_msg = \
            cliUtil.getProductVersionWithCliRet(self.cli, LANG)

        if flag is not True:
            return cliUtil.RESULT_NOCHECK, err_msg

        # 初始化临时目录
        now_time = time.strftime("%Y%m%d%H%M%S", time.localtime())
        results = cliUtil.getVerticalCliRet(cli_ret)
        self.local_path = os.path.abspath(".\\temp\\" + results[0].get("SN") + "_" + now_time)

        try:
            if "6.0" in self.product_version:
                return self.check_60x()
            return self.check_61x()
        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            err_msg = common.getMsg(LANG, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, err_msg
        finally:
            # 删除临时目录所有下载的文件
            if not self.delete_temp_dir():
                LOGGER.logError("delete temp dir failed : %s" % str(self.local_path))
            try:
                if self.rest:
                    self.rest.close()
            except Exception as e:
                LOGGER.logInfo("conn close except:%s" % str(e))

    def check_60x(self):
        """
        功能描述：检查6.0.x版本的重删压缩有效性
        参数：无
        返回值：是否通过，错误描述
        异常描述：无
        """
        # 命令方式获取license信息
        has_effective_capacity, has_reduce_compression, creat_time = self.get_license_by_cmd()

        # license不存在或者包含可得容量，检查通过
        if not creat_time or has_effective_capacity:
            return True, ""

        # 通过REST获取license文件判断是否包含重删压缩
        has_reduce_compression = self.is_reduce_compression_in_license_file()
        if not has_reduce_compression:
            return True, ""

        # 获取license生效时间
        has_active_event = True
        active_time = self.get_license_active_time()
        if not active_time:
            has_active_event = False
            active_time = creat_time

        # 获取风险LUN
        lun_list = self.get_risk_lun_by_files(active_time, "", has_active_event)
        if has_active_event:
            if lun_list:
                return False, get_err_msg(LANG, "check.not.pass.60x.event.exist.has.lun",
                                          (self.product_version, self.format_lun_ids(lun_list)))
            return False, get_err_msg(LANG, "check.not.pass.60x.event.exist.has.no.lun", self.product_version)
        else:
            if lun_list:
                return False, get_err_msg(LANG, "check.not.pass.60x.event.not.exist.has.lun",
                                          (self.product_version, self.format_lun_ids(lun_list)))
            return False, get_err_msg(LANG, "check.not.pass.60x.event.not.exist.has.no.lun", self.product_version)

    def check_61x(self):
        """
        功能描述：检查6.1.0及以上版本的重删压缩有效性
        参数：无
        返回值：是否通过，错误描述
        异常描述：无
        """
        is_upgraded, upgrade_time = self.check_upgrade_60x_to_61x()
        # 非6.0.x升级，检查通过
        if not is_upgraded:
            return True, ""

        # 命令方式获取license信息
        has_effective_capacity, has_reduce_compression, creat_time = self.get_license_by_cmd()
        # license不存在或者包含可得容量或者不包含重删压缩，检查通过
        if not creat_time or has_effective_capacity or not has_reduce_compression:
            return True, ""

        # 获取license生效时间
        has_active_event = True
        active_time = self.get_license_active_time()
        if not active_time:
            has_active_event = False
            active_time = creat_time

        # license生效时间与升级时间比较
        if active_time >= upgrade_time:
            return True, ""

        lun_list = self.get_risk_lun_by_files(active_time, upgrade_time, has_active_event)
        if has_active_event and lun_list:
            # 激活事件存在且期间创建的LUN存在未开启
            return False, self.get_err_msg_event_has_risk_lun(lun_list)
        if not has_active_event and lun_list:
            return False, self.get_err_msg_event_not_exist(lun_list)
        return True, ""

    def get_err_msg_event_not_exist(self, lun_list):
        if baseUtil.is_micro_dev(self.product_model):
            return get_err_msg(LANG, "check.not.pass.micro.event.exist.has.namespace",
                               (self.product_version, self.format_lun_ids(lun_list)))
        return get_err_msg(LANG, "check.not.pass.61x.event.not.exist.has.lun",
                           (self.product_version, self.format_lun_ids(lun_list)))

    def get_err_msg_event_has_risk_lun(self, lun_list):
        if baseUtil.is_micro_dev(self.product_model):
            return get_err_msg(LANG, "check.not.pass.micro.event.exist.has.namespace",
                               (self.product_version, self.format_lun_ids(lun_list)))
        return get_err_msg(LANG, "check.not.pass.61x.event.exist.has.lun",
                           (self.product_version, self.format_lun_ids(lun_list)))

    def get_license_by_cmd(self):
        """
        功能描述：通过命令获取license信息
        参数：无
        返回值：是否包含可得容量  是否包含重删压缩  创建时间
        异常描述：无
        """
        has_effective_capacity = False
        has_reduce_compression = False
        creat_time = ""
        cmd = r"show license"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        self.cli_ret_list.append(cli_ret)

        if flag == cliUtil.RESULT_NOSUPPORT:
            return has_effective_capacity, has_reduce_compression, creat_time

        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)

        results = cliUtil.getVerticalCliRet(cli_ret)
        for item in results:
            if item.get(KEY_LIC_CREATE_TIME):
                creat_time = item.get(KEY_LIC_CREATE_TIME)
            if item.get(KEY_FEATURE_NAME) and VALUE_EFFECTIVE_CAPACITY in item.get(KEY_FEATURE_NAME):
                has_effective_capacity = True
            if item.get(KEY_FEATURE_NAME) and VALUE_DEDUP_COMPRESS in item.get(KEY_FEATURE_NAME):
                has_reduce_compression = True
        return has_effective_capacity, has_reduce_compression, creat_time

    def check_upgrade_60x_to_61x(self):
        """
        功能描述：检查是否从6.0.x升级到6.1.x以上版本
        参数：无
        返回值：是否升级 升级完成时间
        异常描述：无
        """
        is_upgrade = False
        finish_time = ""
        cmd = r"upgrade.sh -p"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInMinisystemModel(self.cli, cmd, LANG)

        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        self.cli_ret_list.append(cli_ret)
        results = cliUtil.getHorizontalCliRet(cli_ret)
        for item in results:
            if '6.0' in item.get(KEY_PRE_UPG_VER) and '6.0' not in item.get(KEY_POST_UPG_VER):
                is_upgrade = True
                finish_time = item.get(KEY_UPG_TIME)
                break

        # 切换到CLI模式
        ret = cliUtil.enterCliModeFromSomeModel(self.cli, LANG)
        if not ret[0]:
            common.reConnectionCli(self.cli, LOGGER)

        return is_upgrade, finish_time

    def get_license_active_time(self):
        """
        功能描述：获取license激活时间, 若激活事件不存在，返回空
        参数：无
        返回值：激活时间
        异常描述：无
        """
        cmd = r"show event level=informational object_type=244 " \
              r"|filterRow column=ID predict=equal_to value=0x200F00F40020"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        self.cli_ret_list.append(cli_ret)
        results = cliUtil.getHorizontalCliRet(cli_ret)
        active_time = ""
        if results:
            active_time = results[0].get(KEY_EVENT_TIME)
            active_time = active_time[:active_time.find(' ')]
            time_array = time.strptime(active_time, "%Y-%m-%d/%H:%M:%S")
            active_time = time.strftime("%Y-%m-%d %H:%M:%S", time_array)
        return active_time

    def format_lun_ids(self, lun_id_list):
        """
        功能描述：格式化lun id为字符串
        参数：lun_id_list, Lun ID的list，int类型
        返回值：连续分段的字符串，如："1~10, 28~115"
        异常描述：无
        """
        if not lun_id_list:
            return ""

        str_lun_id = ""
        lun_id_list.sort()
        start_id = lun_id_list[0]
        temp_id = lun_id_list[0]
        for i, lun_id in enumerate(lun_id_list):
            if lun_id == temp_id:
                if i == len(lun_id_list) - 1:
                    str_lun_id += str(start_id)
            elif lun_id == temp_id + 1:
                temp_id = lun_id
                if i == len(lun_id_list) - 1:
                    str_lun_id += str(start_id) + "~" + str(temp_id)
            else:
                str_lun_id += str(start_id)
                if temp_id > start_id:
                    str_lun_id += "~" + str(temp_id)
                str_lun_id += ", "
                start_id = lun_id
                temp_id = lun_id
                if i == len(lun_id_list) - 1:
                    str_lun_id += str(start_id)

        LOGGER.logInfo("risk lun id string: {}".format(str_lun_id))
        return str_lun_id

    def get_license_file(self):
        """
        功能描述：获取license文件
        参数：无
        返回值：文件路径
        异常描述：无
        """
        try:
            if not self.rest:
                self.rest = contextUtil.getRest(context)

            # 获取license文件的路径标识
            uri = self.rest.getBaseUri() + "license/exportpath"
            response = self.rest.getRest().execGet(uri)
            path_flag = jsonUtil.jsonStr2Dict(str(response.getContent()))["data"]["CMO_LICENSE_DIR_INFO_DIR"]
            if not path_flag:
                return ""

            # 下载license文件
            uri = self.rest.getBaseUri() + "file/license/Upload?path=" + path_flag
            self.download_file(uri)

            # 查找文件路径
            file_path = self.find_file("ActiveLicenseFile")
            LOGGER.logInfo("license file: {}".format(file_path))
            return file_path
        except Exception as e:
            err_args = e.args
            err_code = err_args[0] if err_args else "no code"
            LOGGER.logInfo("get_license_file throws except, errCode is : %s" % err_code)

    def get_config_file(self):
        """
        功能描述：获取运行config文件
        参数：无
        返回值：文件路径
        异常描述：无
        """
        try:
            if not self.rest:
                self.rest = contextUtil.getRest(context)

            # 获取运行数据文件的路径标识
            uri = self.rest.getBaseUri() + "exportRunningData?tag=getpath"
            response = self.rest.getRest().execGet(uri)
            path_flag = jsonUtil.jsonStr2Dict(str(response.getContent()))["data"]
            if not path_flag:
                return ""

            # 下载config文件
            uri = self.rest.getBaseUri() + "exportRunningData?tag=" + path_flag
            self.download_file(uri)

            # 查找文件路径
            file_path = self.find_file("Operating_Data")
            LOGGER.logInfo("config file: {}".format(file_path))
            return file_path
        except Exception as e:
            err_args = e.args
            err_code = err_args[0] if err_args else "no code"
            LOGGER.logInfo("get_config_file throws except, errCode is : %s" % err_code)

    def get_event_file(self):
        """
        功能描述：获取告警事件文件
        参数：无
        返回值：文件路径
        异常描述：无
        """
        try:
            if not self.rest:
                self.rest = contextUtil.getRest(context)

            # 事件打包
            uri = self.rest.getBaseUri() + "alarm/exportresource?tag=package"
            response = self.rest.getRest().execGet(uri)
            error_code = jsonUtil.jsonStr2Dict(str(response.getContent()))["error"]["code"]
            if str(error_code) != "0":
                return ""

            # 查询路径标识, 打包完成才能获取到,超时时间60秒
            uri = self.rest.getBaseUri() + "alarm/exportresource?tag=getpath"
            for i in range(60):
                time.sleep(1)
                response = self.rest.getRest().execGet(uri)
                error_code = jsonUtil.jsonStr2Dict(str(response.getContent()))["error"]["code"]
                if str(error_code) == "0":
                    path_flag = jsonUtil.jsonStr2Dict(str(response.getContent()))["data"]
                    break
            if not path_flag:
                return ""

            # 下载事件文件
            uri = self.rest.getBaseUri() + "alarm/exportresource?tag=" + path_flag
            self.download_file(uri)

            # 查找事件文件
            event_tgz_file = self.find_file("allEvent")
            file_list = self.extract_file(event_tgz_file)
            return file_list[0] if file_list else ""
        except Exception as e:
            err_args = e.args
            err_code = err_args[0] if err_args else "no code"
            LOGGER.logInfo("get_event_file throws except, errCode is : %s" % err_code)

    def download_file(self, url):
        """
        功能描述：下载文件
        参数：url，下载文件的网址
        返回值：是否执行下载
        异常描述：无
        """
        if not self.rest:
            return False

        req_param = HashMap()
        if not os.path.exists(self.local_path):
            os.makedirs(self.local_path)

        try:
            self.rest.getRest().execGetFile(url, req_param, self.local_path)
            return True
        except Exception as e:
            err_args = e.args
            err_code = err_args[0] if err_args else "no code"
            LOGGER.logInfo("download_file throws except, errCode is : %s" % err_code)

    def find_file(self, filter_str):
        """
        功能描述：遍历临时目录下的文件，查找文件名包含filter_str的文件
        参数：filter_str，文件名中的关键字符串
        返回值：文件路径
        异常描述：无
        """
        for root, dirs, files in os.walk(self.local_path):
            for file_name in files:
                if filter_str in file_name:
                    return os.path.join(root, file_name)
        return ""

    def extract_file(self, file_name):
        """
        功能描述：解压文件
        参数：压缩文件路径
        返回值：解压后的文件路径list
        异常描述：无
        """
        file_name_list = []
        ret, _, file_paths = decompress_tar_all_file_with_detail(
            file_name, self.local_path)
        if ret:
            file_name_list.extend(file_paths)
        return file_name_list

    def is_reduce_compression_in_license_file(self):
        """
        功能描述：通过license文件检查是否包含重删压缩
        参数：无
        返回值：是否包含重删压缩
        异常描述：无
        """
        file_path = self.get_license_file()
        if not file_path or not os.path.exists(file_path):
            raise common.UnCheckException(get_err_msg(LANG, "file.not.exist"), "")

        with open(file_path, "r") as read_file:
            for line in read_file:
                if VALUE_DEDUP_COMPRESS_IN_FILE in line:
                    return True
        return False

    def get_risk_lun_info_from_config(self, file_path):
        """
        功能描述：从config文件中获取风险LUN的信息
        参数：file_path，config文件的路径
        返回值：LUN的list
        异常描述：无
        """
        if not file_path or not os.path.exists(file_path):
            raise common.UnCheckException(get_err_msg(LANG, "file.not.exist"), "")

        lun_info_dict_list = []
        with open(file_path, "r") as read_file:
            line = read_file.readline()
            while line:
                if VALUE_LUN_FLAG not in line or VALUE_NAMESPACE_FLAG not in line:
                    line = read_file.readline()
                    continue

                tmp_lun_info = {}
                line = read_file.readline()
                while ":" in line:
                    split_list = line.split(":")
                    tmp_lun_info[split_list[0].strip()] = split_list[1].strip()
                    line = read_file.readline()
                if tmp_lun_info.get(KEY_DEDUP_ENABLED) == 'No' \
                        and tmp_lun_info.get(KEY_COMPRESS_ENABLED) == 'No':
                    lun_info_dict_list.append(tmp_lun_info)

        LOGGER.logInfo(str(lun_info_dict_list))
        return lun_info_dict_list

    def parse_event_lun_info(self, event_str):
        """
        功能描述：从字符串中解析LUN的信息
        参数：event_str，事件字符串
        返回值：LUN的对象，包含ID、WWN、创建时间
        异常描述：无
        """
        if VALUE_CREATE_LUN_EVENT not in event_str:
            return None

        tmp_lun_info = {}
        result = re.findall(r"\s+(.+?)\s+0x.*,\s+ID\s+(.+?),.*,\s+WWN\s+(.+?),", event_str)
        if not result:
            return None
        tmp_lun_info[KEY_LUN_CREATE_TIME] = result[0][0]
        tmp_lun_info[KEY_LUN_ID] = result[0][1]
        tmp_lun_info[KEY_LUN_WWN] = result[0][2]
        return tmp_lun_info

    def get_event_lun_from_event_file(self, file_path):
        """
        功能描述：从event文件中获取LUN的信息
        参数：file_path，event文件的路径
        返回值：LUN的list
        异常描述：无
        """
        if not file_path or not os.path.exists(file_path):
            raise common.UnCheckException(get_err_msg(LANG, "file.not.exist"), "")

        lun_info_dict_list = []
        with open(file_path, "r") as read_file:
            for line in read_file:
                tmp_lun_info = self.parse_event_lun_info(line)
                if tmp_lun_info:
                    lun_info_dict_list.append(tmp_lun_info)
        return lun_info_dict_list

    def add_create_time_to_lun(self, risk_lun_list, event_lun_list):
        """
        功能描述：为风险LUN添加创建时间
        参数：risk_lun_list，风险LUN的list
        event_lun_list，事件中的LUN的list
        返回值：无
        异常描述：无
        """
        for risk_lun in risk_lun_list:
            risk_lun[KEY_LUN_CREATE_TIME] = ""
            for event_lun in event_lun_list:
                if risk_lun.get(KEY_LUN_WWN) == event_lun.get(KEY_LUN_WWN)\
                        and risk_lun.get(KEY_LUN_ID) == event_lun.get(KEY_LUN_ID):
                    risk_lun[KEY_LUN_CREATE_TIME] = event_lun.get(KEY_LUN_CREATE_TIME)
                    break

    def get_risk_lun_id_list_by_time(self, risk_lun_list, from_time, to_time, is_contain_no_time):
        """
        功能描述：根据时间段获取风险LUN的ID
        参数：risk_lun_list，风险LUN的list
        from_time，开始时间
        to_time，结束时间
        is_contain_no_time，是否包含无创建时间的LUN
        返回值：符合条件的LunID的list
        异常描述：无
        """
        lun_id_list = []
        if not to_time:
            to_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
        LOGGER.logInfo("from_time: {}, to_time: {}".format(from_time, to_time))
        for risk_lun in risk_lun_list:
            create_time = risk_lun.get(KEY_LUN_CREATE_TIME)
            if not create_time:
                if is_contain_no_time:
                    lun_id_list.append(int(risk_lun.get(KEY_LUN_ID)))
            elif from_time <= create_time <= to_time:
                lun_id_list.append(int(risk_lun.get(KEY_LUN_ID)))
        LOGGER.logInfo(str(lun_id_list))
        return lun_id_list

    def get_risk_lun_by_files(self, active_time, upgrade_time, is_active_event_exist):
        """
        功能描述：通过下载解析config和event文件获取风险LUN的ID
        参数：active_time，license激活时间
        upgrade_time，升级时间
        is_active_event_exist，激活license的时间是否存在
        返回值：符合条件的LunID的list
        异常描述：无
        """
        config_file = self.get_config_file()
        event_file = self.get_event_file()
        if not config_file or not event_file:
            raise common.UnCheckException(get_err_msg(LANG, "file.not.exist"), "")
        risk_lun = self.get_risk_lun_info_from_config(config_file)
        event_lun = self.get_event_lun_from_event_file(event_file)
        self.add_create_time_to_lun(risk_lun, event_lun)
        # 若有激活事件，则激活之后创建的LUN都有创建时间，所以不用包含无创建时间的LUN
        return self.get_risk_lun_id_list_by_time(risk_lun, active_time,
                                                 upgrade_time, not is_active_event_exist)

    def delete_temp_dir(self):
        """
        功能描述：删除文件夹
        参数：dir_path，文件夹路径
        返回值：bool，是否删除成功
        异常描述：无
        """
        if not os.path.exists(self.local_path) or not os.path.isdir(self.local_path):
            return False

        loop = 3  # 重试次数
        while os.path.exists(self.local_path) and loop > 0:
            shutil.rmtree(self.local_path, True)
            loop = loop - 1
            time.sleep(2)   # 睡眠2秒

        return not os.path.exists(self.local_path)
