# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import time
from decimal import Decimal, ROUND_DOWN

import common
import dpa_common
from common_utils import get_dpa_rest_record
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils
from java.lang import Exception as JException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


ENVIRONMENT_FLAG = "no such user"
GB_RATIO = 1024 ** 3
TB_RATIO = 1024 ** 4
STORAGE_TYPE_LIST = [1, 3]
NO_REPOSITORY_ERROR_DESCRIPTION_LIST = {
    1: "dpa.check.distributed.backup.storage.no.backup.pool",
    3: "dpa.check.distributed.backup.storage.no.metadata.disk",
    5: "dpa.check.distributed.backup.storage.no.self.backup"
}


class Enum(set):
    def __getattr__(self, name):
        if name in self:
            return name
        raise AttributeError


class StorageType(Enum):
    # 标准备份池：type = 1
    standard_backup_pool = 1
    # 元数据卷：type = 3
    metadata_disk = 3
    # 自备份卷：type = 5
    self_backup_disk = 5


def execute(dpa_rest):
    LOGGER.logInfo("start check item dpa check backup storage planning.")
    record_list = []
    # 判断单机或者分布式，decentralized_flag为false表示单机，true表示分布式，err_msg不为空备份存储规划检查直接失败
    decentralized_flag, record, err_msg = check_decentralized_environment()
    record_list.append(record)
    if err_msg:
        LOGGER.logInfo("query decentralized flag error.")
        return False, "\n".join(record_list), err_msg
    # 获取对应节点列表
    node_info_list, record = query_node_info_list(dpa_rest)
    record_list.append(str(record))
    if not node_info_list:
        LOGGER.logInfo("query node list error.")
        err_msg += common.getMsg(LANG, "query.result.abnormal")
        return False, "\n".join(record_list), err_msg
    if not decentralized_flag:
        record, err_msg = check_standalone_repository(dpa_rest, node_info_list[0])
        record_list.append(str(record))
    else:
        records, err_msg = check_distributed_repository(dpa_rest, node_info_list)
        record_list += records
    LOGGER.logInfo(record_list)
    if not err_msg:
        return True, "\n".join(record_list), ""
    return False, "\n".join(record_list), err_msg


def check_distributed_repository(dpa_rest, node_info_list):
    record_list = []
    self_backup_pool_size_list_gb = []
    err_msg_dict = {}
    self_backup_err_msg_dict = {}
    has_self_backup = False
    ip_list = []
    for node_info in node_info_list:
        # 查询每个节点备份存储模块
        node_ip = node_info.get("nodeIp")
        ip_list.append(node_ip)
        record = query_node_storage_list(dpa_rest, node_info)
        record_list.append(str(record))
        if record.get("responseData", {}).get("totalNum") == 0:
            err_msg_dict[node_ip] = dpa_common.get_dpa_msg(
                LANG, "dpa.check.backup.storage.no.standard", (node_ip, node_ip))
            self_backup_err_msg_dict[node_ip] = dpa_common.get_dpa_msg(
                LANG, "dpa.check.distributed.backup.storage.no.self.backup", node_ip)
            continue
        backup_storage_list = record.get("responseData", {}).get("data", [{}])
        self_backup, self_backup_disk_size = check_decentralized_result(
            node_ip, backup_storage_list, err_msg_dict, self_backup_err_msg_dict)
        if self_backup:
            has_self_backup = True
            self_backup_pool_size_list_gb.append(calculate_two_digit_without_round(self_backup_disk_size, GB_RATIO))
    return record_list, concatenate_err_msg(
        has_self_backup, err_msg_dict, self_backup_err_msg_dict, self_backup_pool_size_list_gb, ip_list)


def concatenate_err_msg(has_self_backup, err_msg_dict, self_backup_err_msg_dict, self_backup_pool_size_list_gb,
                        ip_list):
    if not has_self_backup:
        return get_err_msg(err_msg_dict, self_backup_err_msg_dict, ip_list)
    if max(self_backup_pool_size_list_gb) < 30:
        return get_err_msg(err_msg_dict, self_backup_err_msg_dict, ip_list) + dpa_common.get_dpa_msg(
            LANG, "dpa.check.backup.storage.self.backup.pool.abnormal",
            str([str(size) + "GiB" for size in self_backup_pool_size_list_gb]))
    return get_err_msg(err_msg_dict, {}, ip_list)


def get_err_msg(err_msg_dict, self_backup_err_msg_dict, ip_list):
    err_msg = ""
    if not err_msg_dict and not self_backup_err_msg_dict:
        return err_msg
    for ip in ip_list:
        msg = err_msg_dict.get(ip)
        self_backup_err_msg = self_backup_err_msg_dict.get(ip)
        if msg or self_backup_err_msg:
            err_msg += dpa_common.get_dpa_msg(LANG, "dpa.check.abnormal.get.node.ip", ip)
        if msg:
            err_msg += msg
        if self_backup_err_msg:
            err_msg += self_backup_err_msg
    return err_msg


def calculate_two_digit_without_round(total_size, target_unit_ratio):
    target = Decimal(total_size) / Decimal(target_unit_ratio)
    return target.quantize(Decimal('0.01'), rounding=ROUND_DOWN)


def query_node_info_list(dpa_rest):
    node_query_uri = "/clusters/nodes?count=15&index=0"
    record = get_dpa_rest_record(dpa_rest, node_query_uri)
    node_info_list = record.get("responseData", {}).get("data", [])
    return node_info_list, record


def query_node_storage_list(dpa_rest, node_info):
    node_id = node_info.get("id")
    node_query_uri = "/storageresmgm/application/volumes?count=30&filter=&index=0&nodeId={}".format(node_id)
    record = get_dpa_rest_record(dpa_rest, node_query_uri)
    return record


def need_check_dev():
    cur_device_ip = PY_JAVA_ENV.get("devInfo").getIp()
    return [dev_node
            for dev_node in PY_JAVA_ENV.get('selectDevs')
            if dev_node.getIp() in list(PY_JAVA_ENV.get('dpaRelationshipHost').get(cur_device_ip))]


def check_decentralized_environment():
    err_msg = ""
    cmd = "id fsadmin"
    cli_ret = []
    need_check_dev_list = need_check_dev()
    for dev_node in need_check_dev_list:
        connection = None
        try:
            dev_node = EntityUtils.cloneDevNode(dev_node)
            device_sn = dev_node.getIp() + str(time.time())
            dev_node.setDeviceSerialNumber(device_sn)
            connection = PY_JAVA_ENV.get("sshManager").getSshConnection(dev_node)
            cli_ret.append(connection.execCmd(cmd))
        except JException as exp:
            err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", dev_node.getIp())
            LOGGER.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
        finally:
            PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
            LOGGER.logInfo("SSH {} connection is release".format(dev_node.getDeviceSerialNumber()))
    if ENVIRONMENT_FLAG in cli_ret[0]:
        # 回显含有“no such user”，则当前环境是单机，
        return False, "\n".join(cli_ret), err_msg
    return True, "\n".join(cli_ret), err_msg


def check_decentralized_result(node_ip, backup_storage_list, err_msg_dict, self_backup_err_msg_dict):
    err_msg = ""
    self_backup_err_msg, self_backup_disk_size = check_node_repository(
        node_ip, backup_storage_list, StorageType.self_backup_disk)
    for repository_type in STORAGE_TYPE_LIST:
        msg, disk_size = check_node_repository(
            node_ip, backup_storage_list, repository_type)
        if msg:
            err_msg += msg
            continue
        err_msg += check_backup_pool_and_metadata_disk(disk_size, repository_type)
    if err_msg:
        err_msg_dict[node_ip] = err_msg
    if self_backup_err_msg:
        self_backup_err_msg_dict[node_ip] = self_backup_err_msg
        return False, self_backup_disk_size
    return True, self_backup_disk_size


def check_backup_pool_and_metadata_disk(disk_size, repository_type):
    # 标准备份池：每个节点大于等于100TB且小于等于120TB
    if repository_type == StorageType.standard_backup_pool and (
            disk_size > 120 * TB_RATIO or disk_size < 100 * TB_RATIO):
        return dpa_common.get_dpa_msg(LANG, "dpa.check.backup.storage.standard.pool.abnormal",
                                      calculate_two_digit_without_round(disk_size, TB_RATIO))
    # 元数据卷：每个节点大于等于3TB
    if repository_type == StorageType.metadata_disk and disk_size < 3 * TB_RATIO:
        return dpa_common.get_dpa_msg(LANG, "dpa.check.backup.storage.metadata.disk.abnormal",
                                      calculate_two_digit_without_round(disk_size, GB_RATIO))
    return ""


def check_standalone_repository(dpa_rest, node_info):
    node_ip = node_info.get("nodeIp")
    record = query_node_storage_list(dpa_rest, node_info)
    if record.get("responseData", {}).get("totalNum") == 0:
        return record, dpa_common.get_dpa_msg(
            LANG, "dpa.check.standalone.backup.storage.no.self.backup", (node_ip, node_ip))
    backup_storage_list = record.get("responseData", {}).get("data", [{}])
    err_msg, self_backup_disk_size = check_node_repository(node_ip, backup_storage_list, StorageType.self_backup_disk)
    self_backup_disk_size_gb = calculate_two_digit_without_round(self_backup_disk_size, GB_RATIO)
    if err_msg:
        return record, dpa_common.get_dpa_msg(LANG, "dpa.check.abnormal.get.node.ip", node_ip) + err_msg
    if self_backup_disk_size_gb < 30:
        return record, dpa_common.get_dpa_msg(
            LANG, "dpa.check.backup.storage.self.backup.pool.abnormal", str([str(self_backup_disk_size_gb) + "GiB"]))
    return record, ""


def check_node_repository(node_ip, backup_storage_list, repository_type):
    has_repository = False
    disk_size = 0
    for backup_storage_module in backup_storage_list:
        module_type = int(backup_storage_module.get("type"))
        if module_type == repository_type:
            disk_size += int(backup_storage_module.get("totalSize"))
            has_repository = True
    if not has_repository:
        return dpa_common.get_dpa_msg(
            LANG, NO_REPOSITORY_ERROR_DESCRIPTION_LIST.get(repository_type), node_ip), disk_size
    return "", disk_size
