# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time

import cliUtil
import common
import dpa_common
from java.lang import Exception as JException
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils


PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(dpa_rest):
    check_item = CheckItem(dpa_rest, LANG, LOGGER)
    return check_item.execute()


class CheckItem:
    def __init__(self, dpa_rest, lang, logger):
        self.dpa_rest = dpa_rest
        self.lang = lang
        self.logger = logger
        self.err_msg = str()
        self.cli_ret = list()
        self.cluster_key_list = list()
        self.db_key_list = list()
        self.node_type_list = list()

    @staticmethod
    def need_check_dev():
        cur_device_ip = PY_JAVA_ENV.get("devInfo").getIp()
        return [dev_node
                for dev_node in PY_JAVA_ENV.get('selectDevs')
                if dev_node.getIp() in list(PY_JAVA_ENV.get('dpaRelationshipHost').get(cur_device_ip))]

    def execute(self):
        need_check_dev_list = CheckItem.need_check_dev()
        for dev_node in need_check_dev_list:
            connection = None
            try:
                dev_node = EntityUtils.cloneDevNode(dev_node)
                device_sn = dev_node.getIp() + str(time.time())
                dev_node.setDeviceSerialNumber(device_sn)
                connection = PY_JAVA_ENV.get("sshManager").getSshConnection(dev_node)
                self.query_command_info(connection)
            except JException as exp:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", dev_node.getIp())
                self.logger.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
            finally:
                PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
                self.logger.logInfo("SSH {} connection is release".format(dev_node.getDeviceSerialNumber()))
        self.check_result()
        if self.err_msg:
            return False, "\n".join(self.cli_ret), self.err_msg
        return True, "\n".join(self.cli_ret), ""

    def query_command_info(self, connection):
        cluster_key = ""
        db_key = ""
        node_type = ""
        cmd = 'cat /opt/CDMServer/install.cfg'
        cli_cmd_ret = connection.execCmd(cmd)
        self.cli_ret.append(cli_cmd_ret)
        cli_ret_list = cli_cmd_ret.encode("utf8").splitlines()
        command_info = cli_ret_list[1].split()
        for info in command_info:
            if "cluster-key" in info:
                cluster_key = info.split("=")[-1]
            elif "db-key" in info:
                db_key = info.split("=")[-1]
            elif "node-type" in info:
                node_type = info.split("=")[-1]
        if cluster_key == db_key:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.db.key.and.cluster.key.is.same.abnormal")
        self.cluster_key_list.append(cluster_key)
        self.db_key_list.append(db_key)
        self.node_type_list.append(node_type)

    def check_result(self):
        # 检查各节点cluster值是否相同
        if len(list(set(self.cluster_key_list))) > 1:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.cluster.key.not.same.abnormal")
        # 检查各节点db值是否相同
        if len(list(set(self.db_key_list))) > 1:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.db.key.not.same.abnormal")
        # 检查Master节点是否唯一
        if self.node_type_list.count("MASTER") > 1:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.master.node.not.only.abnormal")
