# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

import cliUtil
import common
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.common.conf.hardware_bom import ENCLOSURE_TO_CTRL
from cbb.common.conf.productConfig import VersionManager
from cbb.common.conf.productConfig import compare_dorado_version

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """ 扩控场景检查硬件版本

    :param cli:
    :return:
    """
    check_item = CheckHardwareType(cli, py_java_env, LANG, LOGGER)
    check_item.start()
    return check_item.get_result()


class CheckHardwareType:
    """硬件类型检查类
        1、原扩控的若与输入的新扩控的BOM一致，则通过；不一致则继续检查。
        2、引擎内扩控
            2.1 若原扩控的BOM属于C编码列，新扩控的BOM是其对应的C/C+编码的控制器(
                型号和内存一致)，则通过；
            2.2 若原扩控的BOM属于C+编码列，新扩控的BOM是其对应的C+编码的控制器(型
                号和内存一致)，则通过；
            2.3 其他场景不通过
        3、引擎间扩控
            3.1 若原扩控的BOM属于C编码列，新扩控的BOM是其对应的C+编码列编码，且：
                融合存储：原扩控是V5R7C70SPC200或之后版本
                Dorado：原扩控是6.1.0或之后版本，且内核版本不是1.0.0.1和1.1.0.0
                则通过；
            3.2 其他场景不通过
    """

    def __init__(self, cli, py_java_env, lang, logger):
        self.cli = cli
        self.env = py_java_env
        self.lang = lang
        self.logger = logger
        self.result_flag = True
        self.result_ret = list()
        self.result_err_msg = ""
        self.pdt_model = str(py_java_env.get("devInfo").getDeviceType())
        self.pdt_version = \
            str(py_java_env.get("devInfo").getProductVersion())
        self.ori_enc_bom = \
            str(common.get_sn_from_env(py_java_env)).upper()[2:10]
        self.logger.logInfo("pdt_model: %s, pdt_version: %s" %
                            (self.pdt_model, self.pdt_version))
        self.result_ret.append("Origin Enclosure Bom:" + self.ori_enc_bom)
        self.map_dict = ENCLOSURE_TO_CTRL.get(self.pdt_model.upper())

    def start(self):
        """ 执行检查

        :return:
        """
        try:
            if self.pdt_model in config.OCEAN_PROTECT:
                return
            # 型号不在BOM配套表时，直接通过
            if not self.map_dict:
                return
            self.get_system_info()

            check_bom = CheckBomType(self.ori_enc_bom,
                                     str(self.env.get("newCtrlBom")).upper(),
                                     str(self.env.get("newEncBom")).upper(),
                                     self.map_dict, self.logger)

            flag, result_ret, wrong_bom, need_check_version = check_bom.start()
            self.logger.logInfo("Bom Check Result{Flag:%s needCheckVersion:%s"
                                % (flag, need_check_version))
            self.result_ret.extend(result_ret)
            if not flag:
                self.result_flag = False
                self.result_err_msg = common.getMsg(self.lang,
                                                    "use.same.product.to.exp",
                                                    (",".join(wrong_bom),
                                                     self.ori_enc_bom))
                return

            if need_check_version:
                check_version = CheckVersionType(self.pdt_model,
                                                 self.pdt_version, self.cli,
                                                 self.lang)
                flag, result_ret, err_msg = check_version.start()
                self.result_flag = flag
                self.result_ret.extend(result_ret)
                self.result_err_msg = err_msg
                return
        except (JException, Exception) as e:
            self.logger.logException(e)
            self.result_flag, self.result_err_msg = \
                cliUtil.RESULT_NOCHECK, \
                common.getMsg(self.lang, "query.result.abnormal")
            return

    def get_system_info(self):
        """ 获取系统信息回显，用于界面展示

        :return:
        """
        cmd = "show system general"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                        self.lang)
        self.result_ret.append(ret)
        if flag is not True:
            raise Exception("Failed to get system info.")

    def get_result(self):
        """ 获取检查结果

        :return:
        """
        return (self.result_flag, "\n".join(self.result_ret),
                self.result_err_msg)


class CheckBomType:
    """检查Bom类
    """

    def __init__(self, ori_enc_bom, new_ctrl_bom_total, new_enc_bom_total,
                 map_dict, logger):
        self.result_ret = list()
        self.new_ctrl_bom_total = new_ctrl_bom_total
        self.new_enc_bom_total = new_enc_bom_total
        self.flag = True
        self.need_check_version = False
        self.wrong_bom = list()
        self.map_dict = map_dict
        self.ori_enc_bom = ori_enc_bom
        self.logger = logger

    def start(self):
        res1 = self.check_ctrl_bom(self.new_ctrl_bom_total)
        self.flag = self.flag and res1[0]
        self.result_ret.extend(res1[1])
        self.wrong_bom.extend(res1[2])
        self.need_check_version = self.need_check_version or res1[3]

        res2 = self.check_enc_bom(self.new_enc_bom_total)
        self.flag = self.flag and res2[0]
        self.result_ret.extend(res2[1])
        self.wrong_bom.extend(res2[2])
        self.need_check_version = self.need_check_version or res2[3]
        return self.flag, self.result_ret, self.wrong_bom, self.\
            need_check_version

    def check_ctrl_bom(self, new_ctrl_bom_total):
        """ 引擎内扩控场景，检查新扩控BOM

        :return:
        """
        # 取到的是多个Bom
        result_ret = list()
        wrong_bom = list()
        need_version_check = False
        result_ret.append("New Control Bom List:" + new_ctrl_bom_total)
        if not new_ctrl_bom_total:
            return True, result_ret, wrong_bom, need_version_check

        # 将Bom分开为一个表
        new_ctrl_bom_list = new_ctrl_bom_total.split(',')
        if new_ctrl_bom_list[-1] == '':
            # 去除尾部可能误输的,
            new_ctrl_bom_list.pop()

        res = self.check_new_ctrl_bom_list(new_ctrl_bom_list)
        wrong_bom.extend(res[1])
        need_version_check = res[2]
        return res[0], result_ret, wrong_bom, need_version_check

    def check_new_ctrl_bom_list(self, new_ctrl_bom_list):
        """ 检查一个控制器Bom列表中所有Bom是否匹配正确

        """
        flag = True
        wrong_bom = list()
        need_version_check = False
        for new_ctrl_bom in new_ctrl_bom_list:
            res = self.check_new_ctrl_bom(new_ctrl_bom)
            if not res[0]:
                flag = False
                wrong_bom.append(new_ctrl_bom)
            need_version_check = need_version_check or res[1]

        return flag, wrong_bom, need_version_check

    def check_new_ctrl_bom(self, new_ctrl_bom):
        """ 检查单个控制器Bom是否匹配

        """
        flag = True
        need_version_check = False
        # 判断是否是同型扩控
        if new_ctrl_bom == self.ori_enc_bom:
            return flag, need_version_check
        if not self.map_dict:
            return False, need_version_check
        for enc_bom, ctrl_bom in self.map_dict.items():
            # 原控为C框，新控制器BOM可以是对应的C，不需要检查版本
            if self.ori_enc_bom in enc_bom[1] and new_ctrl_bom == ctrl_bom[1]:
                return True, need_version_check
            # 原控为C框，新控制器BOM可以是对应的C+，需要检查版本
            if self.ori_enc_bom in enc_bom[1] and new_ctrl_bom == ctrl_bom[0]:
                self.logger.logInfo("NEED VERSION CHECK!")
                need_version_check = True
                return True, need_version_check
            # 原控为C+框，新控制器只能是C+,不需要检查版本
            if self.ori_enc_bom in enc_bom[0] and new_ctrl_bom == ctrl_bom[0]:
                return True, need_version_check

        return False, need_version_check

    def check_enc_bom(self, new_enc_bom_total):
        """ 引擎间扩控场景，检查新扩引擎BOM

        :return:
        """
        result_ret = list()
        wrong_bom = list()
        need_version_check = False
        result_ret.append("New Enclosure Bom List:" + new_enc_bom_total)
        if not new_enc_bom_total:
            return True, result_ret, wrong_bom, need_version_check

        new_enc_bom_list = new_enc_bom_total.split(',')
        if new_enc_bom_list[-1] == '':
            new_enc_bom_list.pop()

        res = self.check_new_enc_bom_list(new_enc_bom_list)
        wrong_bom.extend(res[1])
        need_version_check = res[2]
        return res[0], result_ret, wrong_bom, need_version_check

    def check_new_enc_bom_list(self, new_enc_bom_list):
        """ 检查一个控制框Bom列表中所有Bom是否匹配正确

        """
        flag = True
        wrong_bom = list()
        need_version_check = False
        for new_enc_bom in new_enc_bom_list:
            res = self.check_new_enc_bom(new_enc_bom)
            if not res[0]:
                flag = False
                wrong_bom.append(new_enc_bom)
            need_version_check = need_version_check or res[1]

        return flag, wrong_bom, need_version_check

    def check_new_enc_bom(self, new_enc_bom):
        """ 检查单个控制框Bom是否匹配

        """
        need_version_check = False
        # 如果是同类型框
        if new_enc_bom == self.ori_enc_bom:
            return True, need_version_check
        for enc_bom in self.map_dict:
            # 如果框型相同但是一个是双控一个是四控
            # 都是C+
            if new_enc_bom in enc_bom[0] and self.ori_enc_bom in enc_bom[0]:
                return True, need_version_check
            # 都是C
            if new_enc_bom in enc_bom[1] and self.ori_enc_bom in enc_bom[1]:
                return True, need_version_check
            # 原控为C框，新控为对应的C+框
            if new_enc_bom in enc_bom[0] and self.ori_enc_bom in enc_bom[1]:
                self.logger.logInfo("NEED VERSION CHECK!")
                need_version_check = True
                return True, need_version_check

        return False, need_version_check


class CheckVersionType:
    """检查版本类
    """

    def __init__(self, pdt_model, pdt_version, cli, lang):
        self.pdt_model = pdt_model
        self.flag = True
        self.err_msg = ""
        self.pdt_version = pdt_version
        self.result_ret = list()
        self.cli = cli
        self.lang = lang

    def start(self):

        self.check_version()
        return self.flag, self.result_ret, self.err_msg

    def check_version(self):
        """ 检查原集群版本号，C扩C+时要求

        :return:
        """
        if baseUtil.isDoradoV6Dev(self.pdt_model):
            self.check_version_dorado()
        else:
            self.check_version_hybrid()
        return

    def check_version_dorado(self):
        """ Dorado V6检查版本

        :return:
        """
        comp_res = compare_dorado_version(self.pdt_version.upper(),
                                          VersionManager.V6_SWITCH_AND_CP)
        # 输入错误检查
        if not comp_res[0]:
            self.flag = False
            self.err_msg = common.getMsg(self.lang,
                                         "cannot.get.product.version.info")
            return
        # 6.1.0之前的版本需要升级
        if not comp_res[1]:
            self.flag = False
            self.err_msg = common.getMsg(self.lang, "current.version",
                                         self.pdt_version)
        # 6.1.0及之后版本需要检查内核版本
        else:
            flag, kernel_version = self.check_kernel_version()
            if not flag:
                self.flag = False
                self.err_msg = \
                    common.getMsg(self.lang, "current.version",
                                  self.pdt_version) + \
                    common.getMsg(self.lang, "current.kernel.version",
                                  kernel_version)
        return

    def check_version_hybrid(self):
        """ 融合检查版本

        :return:
        """
        if self.pdt_version.replace("Kunpeng", "").upper() < \
                VersionManager.V5_SWITCH_AND_CP:
            self.flag = False
            self.err_msg = common.getMsg(self.lang, "current.version",
                                         self.pdt_version)
        return

    def check_kernel_version(self):
        """ V6检查原集群内核版本

        :return:
        """
        flag, ret, kernel_version = cliUtil.get_kernel_version(self.cli,
                                                               self.lang)
        self.result_ret.append(ret)
        if not flag:
            raise Exception("Failed to get kernel version.")
        return kernel_version not in ("1.0.0.1", "1.1.0.0"), kernel_version
