# -*- coding: UTF-8 -*-
import cliUtil
import common
import traceback

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

#进度总剩余时间
LIMIT_TIME = 120
#进度刷新间隔
INTERVAL = 2

"硬盘容量类型"
class DiskCapacityType():
    Type900 = 0
    Type960 = 1
    Type1000 = 2
    TypeOther = 3

    diskType960List = [0,1,2]

    @staticmethod
    def transType(diskCapacity):
        if diskCapacity % 900 == 0:
            return (DiskCapacityType.Type900, diskCapacity / 900)
        elif diskCapacity % 960 == 0:
            return (DiskCapacityType.Type960, diskCapacity / 960)
        elif diskCapacity % 1000 == 0:
            return (DiskCapacityType.Type1000, diskCapacity / 1000)
        else:
            return (diskCapacity, 1)

def execute(cli):
    '''
        扩容硬盘配套检查：扩容硬盘域场景：检查硬盘容量是否配套
    '''
    cliRetAll = ""
    errMsgAll = ""
    try:
        #无硬盘域操作，不涉及
        expDiskList = common.getExpDiskListFromContext(py_java_env)
        LOGGER.logInfo("expDiskList is %s" % expDiskList)
        if not expDiskList:
            return (cliUtil.RESULT_NOSUPPORT, cliRetAll, "")

        #进度条刷新
        common.threadUpProcess(py_java_env, LIMIT_TIME, INTERVAL, LOGGER)
        #进度开始
        common.inProcess(py_java_env)
        LOGGER.logInfo("process Thread started!")

        #新接入硬盘存在960N盘，必须满足版本和热补丁要求
        ret, exist960Disk = ifExp960NDisk(expDiskList)
        if ret != True:
            return (ret, cliRetAll, common.getMsg(LANG, "query.result.abnormal"))
        if exist960Disk:
            # 检查系统软件版本：
            # Dorado V3R1C21SPC100、Dorado V3R1C30、Dorado V3R1C30SPC100版本及之后的版本，才需要检查
            flag, cliRet, errMsg, softwareVersion, hotPatchVersion = common.getVersion(cli, LANG)
            cliRetAll += cliRet
            if flag != True or len(softwareVersion) < 11:
                return (cliUtil.RESULT_NOCHECK, cliRetAll, common.getMsg(LANG, "query.result.abnormal"))

            # 1）系统热补丁版本必须为V300R001C21SPH108及之后版本
            # 2）系统版本必须为V300R001C21SPC100之后的版本
            if hotPatchVersion:
                errHotPatchDes = hotPatchVersion
            else:
                errHotPatchDes = "Uninstalled"
            verErrMsg = common.getMsg(LANG, "exp.domain.disk.version.error", (softwareVersion, errHotPatchDes))
            if softwareVersion < "V300R001C21SPC100":
                return (False, cliRetAll, verErrMsg)
            elif softwareVersion == "V300R001C21SPC100" and hotPatchVersion < "V300R001C21SPH108":
                return (False, cliRetAll, verErrMsg)

        #新增硬盘域检查不需要检查，只需要对原硬盘域扩容进行检查
        checkExpDomainflag, cliRet, errMsg = checkExpDomain2AddDisk(py_java_env, cli)
        cliRetAll += cliRet
        errMsgAll += errMsg
        if checkExpDomainflag!= True:
            return (checkExpDomainflag, cliRetAll, errMsgAll)
        return (True, cliRetAll, "")
    except:
        LOGGER.logError(traceback.format_exc())
        return (cliUtil.RESULT_NOCHECK, cliRetAll, common.getMsg(LANG, "query.result.abnormal"))
    finally:
        common.finishProcess(py_java_env)
        LOGGER.logInfo("finish process!")

def ifExp960NDisk(expDiskDomainList):
    '''
    判断新接入的硬盘是否存在960N容量的盘
    '''
    exist960Disk = False
    for line in expDiskDomainList:
        diskCapacity = str(line.get("diskCapacity"))
        capacityUnit = str(line.get("unit"))
        ret, diskCapacity2GB = common.changUnit2GBLabelCap(diskCapacity + capacityUnit)
        if ret != True:
            return (False, exist960Disk)
        if diskCapacity2GB % 960 == 0:
            exist960Disk = True
            break
    return (True, exist960Disk)

def checkExpDomain2AddDisk(py_java_env, cli):
    '''
        硬盘域扩容硬盘检查
    '''
    cliRetAll = ""
    errMsgAll = ""

    # 扩原硬盘域：输入参数检查
    expDiskDomainList = common.getExpDiskListFromContextFilter(py_java_env)
    LOGGER.logInfo("expDiskDomainList is %s" % str(expDiskDomainList))
    if not expDiskDomainList:
        return (True, cliRetAll, "")

    #输入参数合并（以相同的DomainID进行合并）
    expDomainMap = parseExpDomainInfo(expDiskDomainList)
    if expDomainMap is None:
        return (False, cliRetAll, common.getMsg(LANG, "query.result.abnormal"))
    LOGGER.logInfo("expDomainMap is %s" % str(expDomainMap))

    checkFlags = []
    for diskDomainId, capacitys in expDomainMap.items():
        #不支持同一个硬盘域同时扩容两种容量的硬盘
        if len(capacitys)!= 1:
            checkFlags.append(False)
            capacitys = [str(int(capacity)) for capacity in capacitys]
            errMsgAll += common.getMsg(LANG, "exp.domain.disk.multi.capacity.error", (diskDomainId, ",".join(capacitys)))
            continue
        else:
            newDiskCapacity = capacitys[0]
            ret = _checkNewDiskCapacityMeet(cli, diskDomainId, newDiskCapacity)
            checkFlags.append(ret[0])
            cliRetAll += ret[1]
            errMsgAll += ret[2]

    #汇总检查结果
    LOGGER.logInfo("checkExpDomain2AddDisk check result:%s" % str(checkFlags))
    flag = _getFinalResult(checkFlags)
    return (flag, cliRetAll, errMsgAll)

def parseExpDomainInfo(expDiskDomainList):
    '''
        解析扩原硬盘域的信息（存为字典：key值为DomainId, value值为输入的标签容量的列表）
    '''
    expDomainMap = {}
    for line in expDiskDomainList:
        diskDomainId = line.get("diskDomain")
        diskCapacity = str(line.get("diskCapacity"))
        capacityUnit = str(line.get("unit"))
        ret, diskCapacity2GB = common.changUnit2GBLabelCap(diskCapacity + capacityUnit)
        if ret != True:
            return None
        if expDomainMap.has_key(diskDomainId):
            if diskCapacity2GB in expDomainMap.get(diskDomainId):
                continue
            else:
                expDomainMap[diskDomainId].append(diskCapacity2GB)
        else:
            expDomainMap[diskDomainId] = [diskCapacity2GB]

    return expDomainMap

def _checkDomainDiskCapacity(cli, diskDomainId):
    '''
        硬盘域扩容（单个硬盘域进行检查）：检查新扩容的硬盘是否满足容量要求
    '''
    cliRetAll = ""
    errMsg = ""
    flag = True
    flag, cliRet, errMsg, diskInDomainList = common.getDiskDomainInfoById(cli, diskDomainId, LANG)
    cliRetAll += cliRet
    if not flag:
        return cliUtil.RESULT_NOCHECK, cliRetAll, errMsg, []

    diskCapacityList = []  # 当前硬盘域(ID为diskDomainId)中所有硬盘的容量列表
    for disk in diskInDomainList:
        diskId = disk.get("ID")  # 获取硬盘域中的硬盘ID
        #通过硬盘标签容量
        isQrySucc, ret, errInfo = common.getDiskCapacityByDiskId(cli, diskId, LANG)
        if not isQrySucc:
            # 查询失败，返回的为命令回文
            cliRetAll += str(ret)
            flag = cliUtil.RESULT_NOCHECK
            errMsg += errInfo
            continue
        #查询成功，返回的为容量数据
        diskCapacityList.append(ret)

    diskCapacityList = list(set(diskCapacityList))
    LOGGER.logInfo("diskCapacityList is %s" % diskCapacityList)
    return flag, cliRetAll, errMsg, diskCapacityList

def _checkNewDiskCapacityMeet(cli, domainId, newDiskCapacity):
    '''
        硬盘域扩容：检查新扩容的硬盘是否满足容量要求
    '''

    cliRetAll = ""
    errMsg = ""
    #获取硬盘域已有硬盘容量信息
    ret = _checkDomainDiskCapacity(cli, domainId)
    cliRetAll += ret[1]
    if ret[0] != True:
        errMsg += ret[2]
        return ret[0], cliRetAll, errMsg

    oldDiskCapacityList = ret[3]

    #转换为整数
    newDiskCapacity = int(newDiskCapacity)
    oldDiskCapacityList = [int(diskCapacity) for diskCapacity in oldDiskCapacityList]

    #错误提示时使用
    oldDiskCapacityListDesc = [str(diskCapacity) for diskCapacity in oldDiskCapacityList]
    #1：先将容量进行类型转换
    oldDiskInfoList = [DiskCapacityType.transType(capacity) for capacity in oldDiskCapacityList]
    newDiskInfo = DiskCapacityType.transType(newDiskCapacity)
    newDiskType = newDiskInfo[0]
    newDiskMultiple = newDiskInfo[1]

    #2.如果同一个硬盘域只存在一种标签容量
    if len(oldDiskCapacityList) == 1:
        #（1）标签容量为900/960/1000GB*N，新扩容的硬盘容量必须为900/960/1000GB*M(M>=N,M和N都为整数)
        oldDiskType = oldDiskInfoList[0][0]
        oldDiskMultiple = oldDiskInfoList[0][1]
        if oldDiskType in DiskCapacityType.diskType960List:
            if oldDiskType == newDiskType and newDiskMultiple >= oldDiskMultiple:
                return True, cliRetAll, ""
            elif oldDiskType in \
                    [DiskCapacityType.Type900, DiskCapacityType.Type1000] \
                    and newDiskType == DiskCapacityType.Type960 \
                    and newDiskMultiple >= oldDiskMultiple:
                #可以扩容4T以上的960盘
                    return True, cliRetAll, ""
            else:
                return False, cliRetAll, common.getMsg(LANG, "exp.domain.disk.capacity.error",
                                            (domainId, ",".join(oldDiskCapacityListDesc), str(newDiskCapacity)))
        # （2）如果同一个硬盘域只存在一种标签容量且标签容量不为900GB*N、960GB*N、1000GB*N（N为整数），新扩容的硬盘容量必须大于等于当前硬盘域已有盘的容量则通过，否则不通过。
        else:
            if newDiskCapacity >= oldDiskCapacityList[0]:
                return True, cliRetAll, ""
            else:
                return False, cliRetAll, common.getMsg(LANG, "exp.domain.disk.capacity.error",
                                            (domainId, ",".join(oldDiskCapacityListDesc), str(newDiskCapacity)))
    elif len(oldDiskInfoList) == 2:
        oldDisk0Type = oldDiskInfoList[0][0]
        oldDisk0Multiple = oldDiskInfoList[0][1]
        oldDisk1Type = oldDiskInfoList[1][0]
        oldDisk1Multiple = oldDiskInfoList[1][1]
        #构造列表，容量从小到大排序
        oldDiskTypes = [oldDisk0Type, oldDisk1Type]
        oldDiskTypes.sort()

        #如果同一个硬盘域已经存在900GB/1000GB*M和960GB*N1的盘,#如果M和N1相同，新扩容的硬盘标签容量为960GB*N2(N2>=N1)则通过，否则不通过；
        if oldDiskTypes == [DiskCapacityType.Type900, DiskCapacityType.Type960] or \
            oldDiskTypes == [DiskCapacityType.Type960, DiskCapacityType.Type1000]:
            #如果M和N1相同，新扩容的硬盘标签容量为960GB*N2(N2>=N1)则通过，否则不通过；
            if oldDisk0Multiple == oldDisk1Multiple:
                if newDiskType == DiskCapacityType.Type960 and newDiskMultiple >= oldDisk0Multiple:
                    return True,cliRetAll,  ""
                else:
                    return False, cliRetAll, common.getMsg(LANG, "exp.domain.disk.capacity.error",
                                                (domainId, ",".join(oldDiskCapacityListDesc), str(newDiskCapacity)))
        """其他情况，全部是判断是否为最大容量"""
        # 如果同一个硬盘域已经存在900GB/1000GB*M和960GB*N1的盘, 如果M和N1不相同，新扩容的硬盘标签容量为M和N1中的较大值对应的容量则通过，否则不通过。
        # 如果同一个硬盘域已经存在900GB*N、960GB*N、1000GB*N(N为整数)其中一种及其它容量的盘，新扩容的硬盘标签容量为当前硬盘域中的大容量盘则通过，否则不通过。
        # 如果同一个硬盘域存在两种标签容量，且标签容量不为900GB*N、960GB*N、1000GB*N(N为整数)，新扩容的硬盘标签容量为当前硬盘域中的大容量盘则通过，否则不通过。
        if newDiskCapacity == max(oldDiskCapacityList):
            return True, cliRetAll, ""
        else:
            return False, cliRetAll, common.getMsg(LANG, "exp.domain.disk.capacity.error",
                                        (domainId, ",".join(oldDiskCapacityListDesc), str(newDiskCapacity)))
    else:
        return False, cliRetAll, common.getMsg(LANG, "exp.domain.disk.capacity.error",
                                                   (domainId, ",".join(oldDiskCapacityListDesc), str(newDiskCapacity)))

def _getFinalResult(checkResults):
    '''
        汇总所有检查结果，计算出最终的检查结果
    '''
    flag = True
    searchList = [False, cliUtil.RESULT_WARNING, cliUtil.RESULT_NOCHECK, True, cliUtil.RESULT_NOSUPPORT]
    for resultType in searchList:
        if resultType in checkResults:
            flag = resultType
            break
    return flag