# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.
from cbb.frame.cli import cliUtil
from cbb.business.operate.checkitems.check_required_hot_patch \
    import check_patch_version_with_no_suggestion
import common
import json

lang = common.getLang(py_java_env)
logger = common.getLogger(PY_LOGGER, __file__)

NO_PATCH = "SPH999"

PRODUCT_MODELS = [
    "OceanStor Dorado 5000 V6",
    "OceanStor Dorado 5500 V6",
    "OceanStor Dorado 5600 V6",
    "OceanStor Dorado 6000 V6",
    "OceanStor Dorado 5000",
    "OceanStor Dorado 5500",
    "OceanStor Dorado 5600",
    "OceanStor Dorado 6000"
]

REQUIRED_PATCHS = [{
    "product_version": "6.0.0",
    "required_patch": "SPH7",
    "product_series": "Dorado",
    "product_models": [],
    "emergency_patch": []
}, {
    "product_version": "6.0.RC1",
    "required_patch": NO_PATCH,
    "product_series": "Dorado",
    "product_models": [],
    "emergency_patch": []
}]

DISK_LIMIT = 50


def execute(cli):
    """扩容智能硬盘框硬盘检查
    OMRP路径：: CheckItem 资源管理/Expansion-Dorado-V6/expand_assess_risk_check

    :param cli: cli连接
    :return:
    """
    cli_ret_all = []
    err_msg_all = []

    # 获取产品型号
    flag, dev_model, err_msg, cli_ret = \
        cliUtil.get_product_model_with_ret(cli, lang)
    cli_ret_all.append(cli_ret)
    if not flag:
        return cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_all), err_msg

    # 产品型号不在列表中返回不涉及
    if dev_model not in PRODUCT_MODELS:
        return cliUtil.RESULT_NOSUPPORT, "\n".join(cli_ret_all), ""

    # 获取硬盘域信息
    flag, cli_ret, err_msg, disk_domains = common.getDiskDomainInfo(cli, lang)
    cli_ret_all.append(cli_ret)
    if not flag:
        return cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_all), err_msg
    # 没有硬盘域 返回不涉及
    if not disk_domains:
        return cliUtil.RESULT_NOSUPPORT, "\n".join(cli_ret_all), ""

    # 获取软件版本和补丁版本
    flag, dev_version, dev_patch, cli_ret = cliUtil\
        .get_system_version_with_ret(cli, lang)
    cli_ret_all.append(cli_ret)
    if not flag:
        err_msg = common.getMsg(lang, "query.result.abnormal")
        return cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_all), err_msg
    logger.logInfo(
        "dev_version is: {}; dev_patch is: {}; dev_model is: {}".format(
            dev_version, dev_patch, dev_model))

    # 检查补丁是否符合要求 保存备用
    check_patch_pass, required_patch = check_patch_version_with_no_suggestion(
        dev_version, dev_patch, dev_model, REQUIRED_PATCHS)

    pass_flag = is_pass_check_exp_smart_enc_disk(check_patch_pass, cli, cli_ret_all, err_msg_all)

    if pass_flag:
        return True, "\n".join(cli_ret_all), ""
    else:
        return False, "\n".join(cli_ret_all), "\n".join(err_msg_all)


def is_pass_check_exp_smart_enc_disk(check_patch_pass, cli, cli_ret_all, err_msg_all):
    # 获取扩容前硬盘域信息
    dd_engine_infos_before = get_dd_engine_infos_from_dev_infos()
    # 获取扩容配置的硬盘域信息
    exp_engine_infos = get_dd_engine_infos_from_exp_infos()
    # 获取扩容后硬盘域信息
    dd_engine_infos_after = add_dd_engine_infos(
        dd_engine_infos_before, exp_engine_infos)
    pass_flag = True
    for disk_domain_id in dd_engine_infos_before:
        # 硬盘域在一个控制框上的硬盘数
        engine_disks = dd_engine_infos_before.get(disk_domain_id)
        disk_per_engine = engine_disks.values()[0] if engine_disks else 0
        member_disk = sum(engine_disks.values())
        # 待扩硬盘总数
        exp_engine_disks = exp_engine_infos.get(disk_domain_id)
        if not exp_engine_disks:
            # 硬盘域不扩容
            continue
        disk_to_expand = sum(exp_engine_disks.values())
        # 扩容前硬盘域跨双引擎
        if is_across_two_engines(disk_domain_id, dd_engine_infos_before):
            ip_enclosures, cli_ret = get_ip_enclosures_for_dorado_v6(cli)
            cli_ret_all.append(cli_ret)
            contain_flag, cli_ret = contain_disk_of_ip_enclosure(
                cli, disk_domain_id, ip_enclosures)
            cli_ret_all.append(cli_ret)
            # 硬盘域中没有智能框的硬盘
            if not contain_flag:
                # 如果扩容后跨大于双引擎，则检查下一个硬盘域，否则继续原逻辑
                if not is_across_two_engines(disk_domain_id, dd_engine_infos_after):
                    continue
                exp_disk_limit = 2 * min(
                    [25 - disk_per_engine, disk_per_engine - 1])
                if disk_to_expand < exp_disk_limit:
                    pass_flag = False
                    err_msg = common.getMsg(
                        lang, "exp.smart.enclosure.disk.step.4.error",
                        (disk_domain_id, member_disk, exp_disk_limit))
                    err_msg_all.append(err_msg)
                    continue

            # 补丁版本是Dorado V6 6.0.0.SPH7以前 且 待扩盘小于50
            if not check_patch_pass and disk_to_expand < DISK_LIMIT:
                pass_flag = False
                err_msg = common.getMsg(
                    lang, "exp.smart.enclosure.disk.step.5.error",
                    disk_domain_id)
                err_msg_all.append(err_msg)
        # 扩容前硬盘域不跨引擎扩容后跨引擎 且 补丁版本是Dorado V6 6.0.0.SPH7以前
        elif is_across_engines(disk_domain_id, dd_engine_infos_after) \
                and not check_patch_pass:
            pass_flag = False
            err_msg = common.getMsg(
                lang, "exp.smart.enclosure.disk.step.6.error",
                disk_domain_id)
            err_msg_all.append(err_msg)
    return pass_flag


def is_across_engines(disk_domain_id, dd_engine_infos):
    """判断硬盘域是否跨引擎

    :param disk_domain_id: 硬盘域ID
    :param dd_engine_infos: 硬盘域在各个控制框上的分布信息
    :return: True(硬盘域跨引擎)/False(硬盘域不跨引擎)
    """
    engine_disk_numbers = dd_engine_infos.get(disk_domain_id, {})
    return len(engine_disk_numbers) > 1


def is_across_two_engines(disk_domain_id, dd_engine_infos):
    """
    判断硬盘域是否跨双引擎

    :param disk_domain_id: 硬盘域ID
    :param dd_engine_infos: 硬盘域在各个控制框上的分布信息
    :return: True(硬盘域跨双引擎)/False(硬盘域不跨双引擎)
    """
    engine_disk_numbers = dd_engine_infos.get(disk_domain_id, {})
    return len(engine_disk_numbers) == 2


def add_dd_engine_infos(one, another):
    """加总硬盘域在各个控制框上的分布信息

    :param one: 例如
    {
        "0": {"CTE0": 8, "CTE1": 8},
        "1": {"CTE0": 10, "CTE1": 10}
    }
    :param another: 例如
    {
        "0": {"CTE0": 2, "CTE1": 2},
        "1": {"CTE0": 15, "CTE1": 15}
    }
    :return: 例如
    {
        "0": {"CTE0": 10, "CTE1": 10},
        "1": {"CTE0": 25, "CTE1": 25}
    }
    """
    total = one.copy()
    for disk_domain in another:
        total[disk_domain] = add_engine_disk_numbers(
            total.get(disk_domain, {}), another.get(disk_domain))

    logger.logInfo("total dd_engine_infos is: %s" % str(total))
    return total


def get_dd_engine_infos_from_dev_infos():
    """获取设备上硬盘域在各个控制框上的分布信息

    :return: 硬盘域在各个控制框上的分布信息
    例如 硬盘域0在CTE0上有8块盘在CTE1上有8块盘 硬盘域1在CTE0上有10块盘在CTE1上有10块盘
    {
        "0": {"CTE0": 8, "CTE1": 8},
        "1": {"CTE0": 10, "CTE1": 10}
    }
    """
    dev_infos_str = py_java_env.get("expInfo").getDevEngineDiskInfoJson()
    logger.logInfo("dev_infos_str is: %s" % dev_infos_str)
    dev_infos = json.loads(dev_infos_str)

    dd_engine_infos = {}
    for dev_info in dev_infos:
        disk_domain_id = dev_info.get("diskDomainId")
        engine_ids = dev_info.get("engineIds", [])
        disks = dev_info.get("disks", [])
        dev_engine_disk_numbers = get_engine_disk_numbers(engine_ids, disks)
        tmp_engine_disk_numbers = dd_engine_infos.get(disk_domain_id, {})
        dd_engine_infos[disk_domain_id] = add_engine_disk_numbers(
            dev_engine_disk_numbers, tmp_engine_disk_numbers)

    logger.logInfo("dev dd_engine_infos is: %s" % str(dd_engine_infos))
    return dd_engine_infos


def add_engine_disk_numbers(one, another):
    """加总2个engine_disk_numbers

    :param one: 例如 {"CTE0": 8, "CTE1": 8}
    :param another: 例如 {"CTE0": 10, "CTE2": 10}
    :return: 例如 {"CTE0": 18, "CTE1": 8, "CTE2": 10}
    """
    total = one.copy()
    for engine in another:
        total[engine] = total.get(engine, 0) + another.get(engine, 0)

    logger.logInfo("total engine_disk_numbers is: %s" % str(total))
    return total


def get_engine_disk_numbers(engine_ids, disks):
    """获取控制框上的硬盘数

    :param engine_ids: 控制框列表
    :param disks: 硬盘信息列表
    :return: 控制框上的硬盘数 例如
    {"CTE0": 8, "CTE1": 8}
    """
    total_disk_count = 0
    for disk in disks:
        disk_count = disk.get("count", "0")
        total_disk_count += int(disk_count)

    engine_disk_numbers = {}
    for engine_id in engine_ids:
        engine_disk_numbers[engine_id] = total_disk_count

    return engine_disk_numbers


def get_dd_engine_infos_from_exp_infos():
    """获取扩容配置中硬盘域在各个控制框上的分布信息

    :return: 硬盘域在各个控制框上的分布信息
    例如 硬盘域0在CTE0上有8块盘在CTE1上有8块盘 硬盘域1在CTE0上有10块盘在CTE1上有10块盘
    {
        "0": {"CTE0": 8, "CTE1": 8},
        "1": {"CTE0": 10, "CTE1": 10}
    }
    """
    exp_infos = common.getExpDiskListFromContextFilter(py_java_env)
    logger.logInfo("exp_infos is: %s" % str(exp_infos))
    pool_and_dd_map = common.getpoolAndDiskDomainMap(py_java_env)

    dd_engine_infos = {}
    for exp_info in exp_infos:
        # dorado v6扩容配置信息中的diskDomain其实是pool_id 需要转换
        pool_id = exp_info.get("diskDomain")
        disk_domain_id = pool_and_dd_map.get(pool_id)
        engine = exp_info.get("eng")
        disk_count = exp_info.get("diskNum")
        exp_engine_disk_numbers = {}
        exp_engine_disk_numbers[engine] = int(disk_count)
        tmp_engine_disk_numbers = dd_engine_infos.get(disk_domain_id, {})
        dd_engine_infos[disk_domain_id] = add_engine_disk_numbers(
            exp_engine_disk_numbers, tmp_engine_disk_numbers)

    logger.logInfo("exp dd_engine_infos is: %s" % str(dd_engine_infos))
    return dd_engine_infos


def get_ip_enclosures_for_dorado_v6(cli):
    """获取dorado v6设备的智能框列表

    :param cli: cli连接
    :return: 智能框列表
    """
    cmd = "show enclosure"
    flag, cli_ret, err_msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return [], cli_ret

    enclosures = cliUtil.getHorizontalCliRet(cli_ret)
    ip_enclosures = []
    for enclosure in enclosures:
        enclosure_id = enclosure.get("ID")
        enclosure_logic_type = enclosure.get("Logic Type")
        enclosure_type = enclosure.get("Type")
        if enclosure_logic_type == "Expansion Enclosure" \
                and "smart" in enclosure_type.lower():
            ip_enclosures.append(enclosure_id)

    logger.logInfo("ip_enclosures is: %s" % str(ip_enclosures))
    return ip_enclosures, cli_ret


def contain_disk_of_ip_enclosure(cli, disk_domain_id, ip_enclosures):
    """判断硬盘域是否包含智能框的硬盘

    :param cli: cli连接
    :param disk_domain_id: 硬盘域ID
    :param ip_enclosures: 智能框列表
    :return: True(包含智能框的硬盘)/False
    """
    cmd = "show disk general"
    flag, cli_ret, err_msg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return False, cli_ret

    disks = cliUtil.getHorizontalCliRet(cli_ret)
    for disk in disks:
        disk_enclosure = disk.get("ID").split(".")[0]
        disk_domain = disk.get("Disk Domain ID")
        if disk_domain.strip() == disk_domain_id.strip() \
                and disk_enclosure in ip_enclosures:
            return True, cli_ret

    return False, cli_ret
