# -*- coding: UTF-8 -*-
import cliUtil
import common

PY_JAVA_ENV = py_java_env
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
TWO_GB_CAPACITY = 2 * 1024 * 1024 * 1024 # 一层cache里面的脏数据
CAPACITY_106_MB = 106 * 1024 * 1024 # 预留空间
MAX_CAPACITY_RATIO = 0.9
HAS_PATCH_VERSION = {
    "V500R007C10": "V500R007C10SPH039",
    "V300R006C20": "V300R006C20SPH039",
    "V500R007C30SPC100": "V500R007C30SPH138",
    "V300R006C50SPC100": "V300R006C50SPH138",
    "V500R007C61": "V500R007C61SPH030",
    "V300R006C61": "V300R006C61SPH030",
    "V500R007C71SPC100": "V500R007C71SPH130"
}
# 某些版本字段有误
WRONG_WORD_VERSION = [
    "V300R006C00SPC100",
    "V300R005C00SPC300",
    "V300R005C01",
    "V300R003C00SPC100",
    "V300R003C10SPC100",
    "V300R003C20",
    "V300R003C20SPC100",
    "V300R003C20SPC200"
]


def execute(cli):
    '''
        文件系统容量跳变检查：
        如果文件系统个数为0，则说明没有文件系统，检测通过
        如果切换开发者模式失败，则需要更高权限的账户
    '''
    flag = True
    cli_ret = ""
    err_msg = ""
    all_ret_msg = ""
    current_version = ""

    try:
        # 查看是否有admin权限
        cli_ret = cliUtil.enterDeveloperMode(cli, LANG)
        if cli_ret[0] is not True:
            err_msg = common.getMsg(LANG, "account.has.no.permission")
            return cliUtil.RESULT_NOCHECK, cli_ret[1], err_msg
        all_ret_msg += cli_ret[1]
        # 查询版本信息
        cli_ret, current_version, current_patch_version = common.getHotPatchVersionAndCurrentVersion(cli, LANG)
        all_ret_msg += cli_ret
        LOGGER.logInfo("version is: {}. patch is: {}".format(current_version, current_patch_version))
        need_patch_version = HAS_PATCH_VERSION.get(current_version)
        if need_patch_version and need_patch_version <= current_patch_version:
            return True, all_ret_msg, ""
        # 查询所有文件系统
        cmd = "show file_system general|filterColumn include columnList=ID,Work\sController,Type"
        check_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        if check_ret[0] is not True:
            LOGGER.logSysAbnormal()
            return True, check_ret[1], check_ret[2]
        cli_ret = check_ret[1]
        all_ret_msg += cli_ret
        # 所有文件系统列表
        fs_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)

        if not fs_list:
            # 如果没有文件系统，则检测通过，输出无文件系统
            LOGGER.logInfo("there is no fs")
            return flag, all_ret_msg, err_msg
        else:
            flag, res_list, err_msg, ret_msg = check_all_fs(fs_list, current_version, cli)
            all_ret_msg += ret_msg
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, res_list, err_msg
        if res_list:
            split_sign = ","
            str_fs_list = split_sign.join(res_list)
            ret_msg = common.getMsg(LANG, "fs.capacity.check.result", str_fs_list)
            if need_patch_version and need_patch_version > current_patch_version:
                ret_msg = common.getMsg(LANG, "fs.capacity.install.hotpatch", (str_fs_list, need_patch_version))
            return False, all_ret_msg, ret_msg
        return True, all_ret_msg, err_msg
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cli_ret, common.getMsg(LANG, "query.result.abnormal"))


def judge_pool_is_enough(space_info):
    """
    判断当前文件系统pool是否充足
    @param space_info:
    @return:
    """
    fs_total = int(space_info.get("File System Total", "0"))
    fs_used = int(space_info.get("File System Used", "0"))
    spa_quota_data = int(space_info.get("SPA Quota Data", "0"))
    spa_quota_meta = int(space_info.get("SPA Quota Meta", "0"))
    storage_pool_available = int(space_info.get("Storage Pool Available", "0"))
    fs_available = 0
    if fs_used + TWO_GB_CAPACITY < fs_total:
        fs_available = fs_total - fs_used - TWO_GB_CAPACITY
    fs_total_quota = spa_quota_data + spa_quota_meta
    return not (fs_total > fs_total_quota and fs_available > storage_pool_available)


def has_jumping_possibility(space_info):
    """
    直接判断，当前是否存在跳变可能行
    @param space_info:
    @return:
    """
    grain_reserve_data = int(space_info.get("Grain Reserve Data", "0"))
    grain_reserve_meta = int(space_info.get("Grain Reserve Meta", "0"))
    fs_total = int(space_info.get("File System Total", "0"))
    fs_used = int(space_info.get("File System Used", "0")) + \
             max(grain_reserve_data, grain_reserve_meta) + CAPACITY_106_MB + TWO_GB_CAPACITY
    return fs_used > fs_total * MAX_CAPACITY_RATIO


def after_version_get_fs_used(space_info, fs_id, temp_total, cli):
    """
    V5R7C50和V3R6C60之后的版本使用此方法计算fsUsed
    @param space_info:
    @param fs_id:
    @param temp_total:
    @param cli:
    @return:
    """
    LOGGER.logInfo("current version > V3R6C60 and V5R7C50")
    grain_reserve_data = int(space_info.get("Grain Reserve Data", "0"))
    grain_reserve_meta = int(space_info.get("Grain Reserve Meta", "0"))
    cli_temp = "show file_system general file_system_id={}|filterColumn include columnList=Used\sCapacity\sRatio(%)"
    cli_res = cliUtil.excuteCmdInCliMode(cli, cli_temp.format(fs_id), True, LANG)
    if not cli_res[0]:
        LOGGER.logSysAbnormal()
        LOGGER.logInfo("get ratio failed")
    used_capacity_ratio = cliUtil.getVerticalCliRet(cli_res[1])
    if used_capacity_ratio[0]:
        return (float(temp_total) * (float(used_capacity_ratio[0].get("Used Capacity Ratio(%)", "0")) / 100)
            + max(grain_reserve_data, grain_reserve_meta) + CAPACITY_106_MB + TWO_GB_CAPACITY)
    else:
        return 0


def before_version_get_fs_used(space_info, temp_total):
    """
    V5R7C30以及V3R6C50之前的版本使用此方法计算fsUsed
    @param space_info:
    @param temp_total:
    @return:
    """
    LOGGER.logInfo("current version < V3R6C60 and V5R7C50")
    grain_reserve_data = int(space_info.get("Grain Reserve Data", "0"))
    grain_reserve_meta = int(space_info.get("Grain Reserve Meta", "0"))
    fs_used = int(space_info.get("File System Used", "0"))
    fs_total = int(space_info.get("File System Total", "0"))
    spc_reserve_total = int(space_info.get("SPC Reserve Total", "0"))
    spc_reserve_used = int(space_info.get("SPC Reserve Used", "0"))
    ddp_saved_compress = int(space_info.get("DDP Saved Compress", "0"))
    ddp_saved_dedup = int(space_info.get("DDP Saved Dedup", "0"))
    snapshot_reserved_free = int(space_info.get("Snapshot Reserved Free", "0"))
    file_system_used = fs_used + max(grain_reserve_data, grain_reserve_meta) + CAPACITY_106_MB + TWO_GB_CAPACITY
    file_system_used = temp_total - min((fs_total - file_system_used), temp_total + spc_reserve_total
                                        - spc_reserve_used + ddp_saved_compress + ddp_saved_dedup
                                        + snapshot_reserved_free - file_system_used)
    return file_system_used


def check_all_fs(fs_list, current_version, cli):
    """
    遍历所有文件系统，检查其是否存在容量跳变的可能
    @param fs_list:
    @param current_version:
    @param cli:
    @return:
    """
    cli_ret, current_version, current_patch = common.getHotPatchVersionAndCurrentVersion(cli, LANG)
    special_word = "Reserved"
    if current_version in WRONG_WORD_VERSION:
        special_word = "Resvered"
    # 查询本地的sn
    all_ret_msg = ""
    dev_sn = PY_JAVA_ENV.get("devInfo").getDeviceSerialNumber()
    res_list = []
    # 开始遍历所有的文件系统,此处为并发处理，所以不需要额外查询，直接取值即可
    for item in fs_list:
        fs_id = item.get("ID")
        fs_controller = item.get("Work Controller")
        fs_type = item.get("Type")
        # 获取文件系统的统计信息
        cmd_tmp = 'show space statistic_info file_system_id={} controller={}|filterColumn include' \
                 ' columnList=File\sSystem\sTotal,File\sSystem\sUsed,SPA\sQuota\sData,SPA\sQuota\sMeta,' \
                 'Storage\sPool\sAvailable,Grain\sReserve\sData,Grain\sReserve\sMeta,SPC\sReserve\sTotal,' \
                 'SPC\sReserve\sUsed,DDP\sSaved\sDedup,Snapshot\s{}\sFree,DDP\sSaved\sCompress'
        statistic_cmd = cmd_tmp.format(fs_id, fs_controller, special_word)
        check_ret = common.getObjFromFile(py_java_env, LOGGER, dev_sn, statistic_cmd, LANG)
        LOGGER.logInfo("get info from db:{}".format(check_ret))
        if check_ret[0] is not True:
            err_msg = common.getMsg(LANG, "can.not.get.statistic.info")
            cli_ret = check_ret[1]
            return False, cli_ret, err_msg, all_ret_msg
        statistic_info = {}
        all_ret_msg += check_ret[1]
        cli_ret = cliUtil.getVerticalCliRet(check_ret[1])
        if cli_ret:
            statistic_info = cli_ret[0]
        LOGGER.logInfo(statistic_info)
        # 判断当前文件系统pool是否充足
        if fs_type == "thick" or judge_pool_is_enough(statistic_info):
            LOGGER.logInfo("this pool is enough")
            res = has_jumping_possibility(statistic_info)
            res and res_list.append(str(fs_id))
        else:
            LOGGER.logInfo("this pool is not enough")
            spa_quota_data = int(statistic_info.get("SPA Quota Data", "0"))
            spa_quota_meta = int(statistic_info.get("SPA Quota Meta", "0"))
            storage_pool_available = int(statistic_info.get("Storage Pool Available", "0"))
            fs_total = int(statistic_info.get("File System Total", "0"))
            temp_total = sum([spa_quota_data, spa_quota_meta, storage_pool_available])
            temp_total = min(temp_total, fs_total)
            file_system_used = 0
            # 判断当前版本
            if judge_current_version(current_version):
                file_system_used = after_version_get_fs_used(statistic_info, fs_id, temp_total, cli)
            else:
                file_system_used = before_version_get_fs_used(statistic_info, temp_total)
            if file_system_used > (temp_total * MAX_CAPACITY_RATIO):
                res_list.append(str(fs_id))
    return True, res_list, "", all_ret_msg


def judge_current_version(current_version):
    """
    判断当前系统版本是否大于V3R6C60或V5R7C50
    @param current_version:
    @return:
    """
    if current_version.startswith("V300"):
        if current_version >= "V300R006C60":
            return True
        else:
            return False
    else:
        if current_version >= "V500R007C50":
            return True
        else:
            return False