# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2022. All rights reserved.
from os import path

import com.huawei.ism.tool.framework.platform.runtime.SystemProperties as sysProperties
import java.util.MissingResourceException as MissingResourceException
from com.huawei.ism.tool.obase.exception import ToolException
import cliUtil
import common

PY_JAVA_ENV = py_java_env
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
HAS_PATCH_VERSION = {
    "V500R007C73SPC100": "V500R007C73SPH107",
    "V500R007C73SPC200": "V500R007C73SPH201",
    "V500R007C71SPC100": "V500R007C71SPH115",
    "V500R007C61": "V500R007C61SPH033",
    "V300R006C61": "V300R006C61SPH033"
}
MINUS_64M = 18446744073642442752
THIRTY_PB = 33776997205278720
DEFAULT_FS_MAX_NUM = 100
G_HEART_BEAT_DATA = {}
G_HEART_BEAT_CLI = []


def execute(cli):
    """
    文件系统元数据预留容量检查
    @param cli:
    @return:
    """
    conn_cli = ""
    all_ret_cli = []
    ret_flag = True
    ret_cli = ""
    ret_msg = ""
    try:
        # 18000的需要额外获取连接
        create_ret = common.createDeviceCliContFor18000(cli, PY_JAVA_ENV, LOGGER, LANG)
        if create_ret[0] is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(all_ret_cli), create_ret[2]
        conn_cli = create_ret[1]
        # 查询版本信息
        has_patch_flag, cli_ret = check_version_has_patch(conn_cli)
        all_ret_cli.append(cli_ret)
        if has_patch_flag:
            return True, "\n".join(all_ret_cli), ""
        dev_sn = PY_JAVA_ENV.get("devInfo").getDeviceSerialNumber()
        ret_flag, ret_cli, ret_msg = cliUtil.enterDeveloperMode(conn_cli, LANG)
        all_ret_cli.append(ret_cli)
        if ret_flag is not True:
            err_msg = common.getMsg(LANG, "account.has.no.permission")
            return cliUtil.RESULT_NOCHECK, "\n".join(all_ret_cli), err_msg
        # 获取所有文件系统
        ret_flag, fs_list, ret = get_fs_list(conn_cli)
        all_ret_cli.append(ret)
        # 如果当前系统不支持文件系统，则直接返回通过
        if not ret_flag:
            return True, fs_list, ret
        # 如果当前无文件系统，直接返回通过
        if not fs_list:
            return True, "\n".join(all_ret_cli), ""
        # 获取各个文件系统的硬盘域id
        fs_disk_map, cli_ret = get_disk_domain_of_fs(conn_cli, fs_list)
        all_ret_cli.append(cli_ret)
        # 获取所有的硬盘域
        disk_domain_list, cli_ret = get_all_disk_domain(conn_cli)
        all_ret_cli.append(cli_ret)
        # 获取每一个文件系统的硬盘域的各个层级raid6级别是否相等
        exec_flag, disk_domain_map, cli_ret = id_disk_domain_has_difference_raid(disk_domain_list, conn_cli)
        all_ret_cli.append(cli_ret)
        if not exec_flag:
            ret_msg = common.getMsg(LANG, "get.node.alg.size.err")
            return False, "\n".join(all_ret_cli), ret_msg
        # 判断所有硬盘域的raid6级别是否一致
        if judge_member_disk_valid(disk_domain_map):
            return True, "\n".join(all_ret_cli), ""
        # 获取每个控制器下需要执行的命令
        cmd_list = get_grain_reserve_meta(fs_list)
        # 对于控制器下数量大于最大数量的查询做截断
        ret_flag, ret_data = truncate_fs_list(cmd_list)
        if ret_flag:
            ret_msg = common.getMsg(LANG, "increase.your.max.fs.number")
            cmd_list = ret_data
        # 获取每个控制器下各个文件系统执行的结果
        ret_flag = exec_cmd_get_meta(conn_cli, PY_JAVA_ENV, LOGGER, LANG, "", cmd_list)
        global G_HEART_BEAT_DATA, G_HEART_BEAT_CLI
        all_ret_cli.extend(G_HEART_BEAT_CLI)
        if ret_flag is not True:
            LOGGER.logError("get grain info failed")
            return False, "\n".join(all_ret_cli), ""
        LOGGER.logInfo(G_HEART_BEAT_DATA)
        fs_list_result = check_all_fs(fs_list, disk_domain_map, fs_disk_map, G_HEART_BEAT_DATA)
        if ret_msg == common.getMsg(LANG, "increase.your.max.fs.number"):
            return cliUtil.RESULT_NOCHECK, "\n".join(all_ret_cli), ret_msg
        if fs_list_result:
            split_sign = ","
            ret_msg = common.getMsg(LANG, "check.reserved.grain.meta", split_sign.join(fs_list_result))
            return False, "\n".join(all_ret_cli), ret_msg
        else:
            cli_ret, current_version, current_patch = common.getHotPatchVersionAndCurrentVersion(conn_cli, LANG)
            if HAS_PATCH_VERSION.get(current_version):
                return cliUtil.RESULT_WARNING, "\n".join(all_ret_cli), ""
        return True, "\n".join(all_ret_cli), ""
    except common.UnCheckException as uncheck:
        LOGGER.logError(
            "Failed to heart beat :%s" % str(uncheck.errorMsg))
        return cliUtil.RESULT_NOCHECK, "\n".join(all_ret_cli), uncheck.errorMsg
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, "\n".join(all_ret_cli), common.getMsg(LANG, "query.result.abnormal")
    finally:
        if conn_cli != "":
            if common.is18000(PY_JAVA_ENV, conn_cli) and conn_cli is not cli:
                common.closeConnection(conn_cli, PY_JAVA_ENV, LOGGER)
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)
        common.finishProcess(PY_JAVA_ENV)
        LOGGER.logInfo("finish process!")


def judge_member_disk_valid(disk_domain_map):
    for disk_domain in disk_domain_map:
        disk_domain_result = disk_domain_map.get(disk_domain)
        if disk_domain_result[0] or disk_domain_result[1]:
            return False
    return True


def check_version_has_patch(cli):
    """
    检查当前版本是否包含补丁
    @param cli: 查询连接
    @return:
    """
    cli_ret, current_version, current_patch = common.getHotPatchVersionAndCurrentVersion(cli, LANG)
    LOGGER.logInfo("current patch %s , version : %s" % (current_patch, current_version))
    need_patch_version = HAS_PATCH_VERSION.get(current_version)
    if need_patch_version:
        return (current_patch >= need_patch_version), cli_ret
    return False, cli_ret


def get_all_pool(cli):
    """
    获取所有的pool
    @param cli: 查询连接
    @return:
    """
    cmd = "show storage_pool general"
    (flag, ret, msg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        LOGGER.logError(msg)
        return []
    pool_list = cliUtil.getHorizontalCliRet(ret)
    return pool_list, ret


def get_disk_domain_of_fs(cli, fs_list):
    """
    获取每个文件系统各个硬盘域的id
    @param cli: 查询连接
    @param fs_list: 所有文件系统
    @return:
    """
    cli_ret = ""
    pool_list, ret = get_all_pool(cli)
    cli_ret += ret
    if not pool_list:
        return {}
    fs_disk_domain_map = {}
    for fs in fs_list:
        fs_id = fs.get("ID")
        pool_id = fs.get("Pool ID")
        for pool in pool_list:
            pool_id_from_pool_list = pool.get("ID")
            disk_domain = pool.get("Disk Domain ID")
            if pool_id == pool_id_from_pool_list:
                fs_disk_domain_map[fs_id] = disk_domain
                break
    return fs_disk_domain_map, cli_ret


def get_all_disk_domain(cli):
    """
    # 获取所有的硬盘域
    @param cli: 查询连接
    @return:
    """
    cmd = "show disk_domain general"
    (flag, ret, msg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        LOGGER.logError(msg)
        return []
    disk_list = cliUtil.getHorizontalCliRet(ret)
    return disk_list, ret


def id_disk_domain_has_difference_raid(disk_domain_list, cli):
    """
    检查一个硬盘域是否存在不同层，并且不同层的raid策略是否相等
    @param disk_domain_list: 硬盘域列表
    @param cli 查询连接
    @return:
    """
    all_ret = ""
    disk_domain_map = {}
    cliUtil.enterDebugModeFromCliMode(cli, LANG)
    for disk_domain in disk_domain_list:
        disk_domain_id = disk_domain.get("ID")
        disk_domain_result = [False, False]
        disk_domain_result, cli_ret, exec_result = get_zone_info(disk_domain_id, cli)
        all_ret += cli_ret
        if exec_result is not True:
            return False, disk_domain_map, all_ret
        disk_domain_map[disk_domain_id] = disk_domain_result
    cliUtil.enterCliModeFromDebugModel(cli, LANG)
    return True, disk_domain_map, all_ret


def is_zone_has_raid6(zone_data, tier):
    """
    判断当前硬盘域是否包含raid盘
    @param zone_data: zone列表
    @param tier: 层级 0 代表SSD， 1代表 SAS， 2代表NL_SAS
    @return:
    """
    zone_list = ["0x9", "0x400b", "0x800d"]
    for item in zone_data:
        zone_id = item.get("Zone ID", "0")
        if zone_id.find(zone_list[tier]) != -1:
            return True
    return False


def get_member_disk_number(disk_domain_id, zone_id, cli):
    """

    @param disk_domain_id: 硬盘域id
    @param zone_id: zone id
    @param cli: 查询连接
    @return:
    """
    cmd_get_raid = "spa dumpobjinfo -p {} -t 4 -o {}".format(disk_domain_id, zone_id)
    LOGGER.logInfo(cmd_get_raid)
    res, cli_ret, msg = cliUtil.excuteCmdInDebugModel(cli, cmd_get_raid, LANG)
    if res is not True:
        LOGGER.logError(msg)
        return False, cli_ret, msg
    disk_list = cliUtil.getVerticalCliRet(cli_ret)
    if disk_list and disk_list[0]:
        member_disk_number = disk_list[0].get("memberDiskNum")
        return True, member_disk_number, cli_ret
    return False, cli_ret, msg


def get_zone_info(disk_domain_id, cli):
    """
    判断当前硬盘域是否存在不同raid级别
    @param disk_domain_id:
    @param cli: 查询连接
    @return: 第一个返回值代表不支持smart tier特性时的结果，第二个返回值代表支持smart tier的结果，第三个为回文, 第四个为是否异常
    """
    all_ret = ""
    zone_list = ["0x9", "0x400b", "0x800d"]
    all_raid6_disk_domain = []
    for tier in range(0, 3):
        cmd = "spa queryzoneintier -p {} -t {}".format(disk_domain_id, tier)
        LOGGER.logInfo(cmd)
        (res, cli_ret, msg) = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
        all_ret += cli_ret
        if res is not True:
            LOGGER.logError("debug cmd exec failed: {}, {}, {},".format(res, cli_ret, msg))
            return [True, True], all_ret, False
        zone_data = cliUtil.getHorizontalCliRet(cli_ret)
        raid6_disk_num = 0
        if is_zone_has_raid6(zone_data, tier):
            ret_flag, ret_data, ret_msg = get_member_disk_number(disk_domain_id, zone_list[tier], cli)
            if ret_flag is not True:
                all_ret += ret_data
                LOGGER.logError(ret_msg)
                return [True, True], all_ret, False
            raid6_disk_num = ret_data
        all_raid6_disk_domain.append(raid6_disk_num)
    all_raid6_disk_domain_no_zero = filter_zero(all_raid6_disk_domain)
    if len(all_raid6_disk_domain_no_zero) == 1:
        return [False, False], all_ret, True
    if len(all_raid6_disk_domain_no_zero) == 2:
        not_support_ret = not (all_raid6_disk_domain_no_zero[0] == all_raid6_disk_domain_no_zero[1])
        support_ret = not (all_raid6_disk_domain[1] == 0 or all_raid6_disk_domain[2] == 0 or
                           all_raid6_disk_domain[1] == all_raid6_disk_domain[2])
        return [not_support_ret, support_ret], all_ret, True
    if len(all_raid6_disk_domain) == 3:
        not_support_ret = not (all_raid6_disk_domain[0] == all_raid6_disk_domain[1] and
                               all_raid6_disk_domain[1] == all_raid6_disk_domain[2])
        support_ret = not (all_raid6_disk_domain[1] == all_raid6_disk_domain[2])
        return [not_support_ret, support_ret], all_ret, True
    return [False, False], all_ret, False


def filter_zero(arr_list):
    """
    筛选非0元素
    @param arr_list: 数组
    @return:
    """
    return [item for item in arr_list if item]


def get_grain_reserve_meta(fs_list):
    """
    查询每个文件系统的元数据预留信息指令并按照控制器进行归类
    @param fs_list: 文件系统列表
    @return:
    """
    cmd_list = {}
    for fs in fs_list:
        fs_id = fs.get("ID")
        controller_type = fs.get("Work Controller")
        tem_cmd_list = cmd_list.get(controller_type, [])
        tem_cmd_list.append("grain checkGrainRaid6Info {}".format(fs_id))
        cmd_list[controller_type] = tem_cmd_list
    return cmd_list


@common.checkAllEngineInClusterWarp
def exec_cmd_get_meta(cli, pa_java_env, logger, lang, heart_beat_cli_ret, cmd_list):
    """
    根据控制器查找文件系统的元数据信息
    @param pa_java_env: 运行环境
    @param logger: 日志打印对象
    @param lang: 语言
    @param heart_beat_cli_ret: 心跳回文
    @param cli: 查询连接
    @param cmd_list: 命令
    @return:
    """
    global G_HEART_BEAT_CLI, G_HEART_BEAT_DATA
    G_HEART_BEAT_CLI.append(heart_beat_cli_ret)
    # 获取所有的节点和引擎信息
    flag, err_msg, cli_ret, current_ctrl_node_id, engine_ctrl_node_mapping_dict, current_engine, node_num =\
        common.getEngineCtrlNodeInfo(cli, logger, lang)
    LOGGER.logInfo("current_ctrl_node_id {}".format(current_ctrl_node_id))
    if flag is not True:
        LOGGER.logError("get engine failed")
        return False
    controller_id = "0A"
    for controller in cmd_list:
        node_id = get_node_id_by_controller(engine_ctrl_node_mapping_dict, controller)
        if int(current_ctrl_node_id) == node_id:
            controller_id = controller
            break
    node_id = get_node_id_by_controller(engine_ctrl_node_mapping_dict, controller_id)
    if node_id != int(current_ctrl_node_id):
        cliUtil.enterCliModeFromSomeModel(cli, lang)
        return True
    for cmd in cmd_list.get(controller_id, []):
        flag, cli_ret, err_msg = cliUtil.excuteCmdInDebugModel(cli, cmd, lang)
        G_HEART_BEAT_CLI.append(cli_ret)
        if flag is not True:
            LOGGER.logError("get meta failed: {}".format(cmd))
            return False
        G_HEART_BEAT_DATA[cmd] = cli_ret
    cliUtil.enterCliModeFromSomeModel(cli, lang)
    return True


def get_node_id_by_controller(engine_ctrl_node_mapping_dict, controller):
    """
    根据controller id 获取slot id
    @param engine_ctrl_node_mapping_dict: 引擎列表
    @param controller: 控制器id
    @return:
    """
    engine_id = int(controller[0])
    node_id = ord(controller[1]) - 65
    sum_length = 0
    LOGGER.logInfo("engine_ctrl_node_mapping_dict {}".format(engine_ctrl_node_mapping_dict))
    for index in range(engine_id):
        str_index = str(index)
        sum_length += len(engine_ctrl_node_mapping_dict[str_index])
    return sum_length + node_id


def get_max_fs_number():
    """
    获取单次检查最大检查的文件系统个数
    获取是否设置了逃生通道，访问子工具configuration目录下的system.properties
    app.inspect.max.fs.num
    @return:
    """
    sys_path = path.abspath(common.DIR_RELATIVE_CMD)
    key = "app.max.file.system.number"
    sys_conf = sysProperties(sys_path)
    LOGGER.logInfo("config path: {}, {}".format(sys_path, sys_conf))
    try:
        value = sys_conf.getValue(key)
        LOGGER.logInfo("max fs num: {}".format(value))
        if str(value).isdigit():
            return int(value)
        return DEFAULT_FS_MAX_NUM
    except MissingResourceException as e:
        LOGGER.logError(
            "no key found. err msg:{}".format(str(e))
        )
    return DEFAULT_FS_MAX_NUM


def get_fs_list(cli):
    """
    获取所有文件系统
    @param: cli
    @return:
    """
    cmd = "show file_system general|filterColumn include columnList=" \
          "ID,Work\sController,Pool\sID,Support\sFile\sSystem\sTier"
    (flag, ret, msg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if not flag:
        return False, ret, msg
    # 所有文件系统列表
    fs_list = cliUtil.getHorizontalNostandardCliRet(ret)
    return True, fs_list, ret


def truncate_fs_list(cmd_list):
    """
    对文件系统数量大于100的控制器进行截断
    @param cmd_list: 命令列表
    @return:
    """
    # 获取最大查询的文件系统数量
    max_fs_num = get_max_fs_number()
    flag = False
    for controller in cmd_list:
        LOGGER.logInfo("current controller {}, num: {}, max: {}"
                       .format(controller, len(cmd_list.get(controller)), max_fs_num))
        if len(cmd_list.get(controller, [])) > max_fs_num:
            flag = True
            controller_cmd = cmd_list.get(controller)
            cmd_list[controller] = controller_cmd[:max_fs_num]
    return flag, cmd_list


def check_all_fs(fs_list, disk_domain_map, fs_disk_map, fs_meta_result):
    """
    遍历文件系统，检查nodeAlgSize的值
    @param fs_list: 文件系统列表
    @param disk_domain_map: 硬盘域各个层级的raid6级别
    @param fs_disk_map: 文件系统对应的硬盘域id
    @param fs_meta_result: 文件系统查询得到的元数据统计信息
    @return:
    """
    fs_list_result = []
    for fs in fs_list:
        fs_id = fs.get("ID", "0")
        is_support_tier = fs.get("Support File System Tier", "No")
        cmd = "grain checkGrainRaid6Info {}".format(fs_id)
        disk_domain_map_res = disk_domain_map.get(fs_disk_map.get(fs_id, "-1"), [])
        disk_domain_map_index = 0
        if is_support_tier == "Yes":
            disk_domain_map_index = 1
        if disk_domain_map_res[disk_domain_map_index] and fs_meta_result.get(cmd, ""):
            ret_data = fs_meta_result[cmd]
            data = cliUtil.getVerticalCliRet(ret_data)
            LOGGER.logInfo(" meta ret data:{}".format(data))
            raid_switch = data[0].get("The meta used raid6 state")
            if raid_switch and str(raid_switch)[0] == "1":
                node_alg_size_str = data[0].get("NodeAlgSize(data/meta)")
                node_alg_size_list = node_alg_size_str.split()
                node_alg_size = 0
                # 判断nodeAlgSize是否查询是否成功，正常查询出来是包括数据和元数据的，所以后面需要做切分
                if len(node_alg_size_list) >= 2:
                    node_alg_size = int(node_alg_size_list[1])
                LOGGER.logInfo("nodeAlgSize: {}".format(node_alg_size))
                if node_alg_size >= THIRTY_PB and node_alg_size <= MINUS_64M:
                    fs_list_result.append(str(fs_id))
    return fs_list_result

