# -*- coding: UTF-8 -*-
import datetime
import common
import cliUtil
from common_utils import get_err_msg
from common import UnCheckException


LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
NO_PATCH = "--"
SIX_MONTH = 6
TWELVE_MONTH = 12


def execute(cli):
    """
    控制器服务年限检查
    步骤1 以admin用户登录设备；
    步骤2 执行命令：show controller general，查询所有控制器的服务起始时间Starting Point Date和服务时长Session（单位：月）；
    步骤3 执行命令：show system general，查看系统当前时间（Time）。
    :param cli:
    :return:date_check
    """
    date_check = ControllerStartingPointDateCheck(cli, LANG, PY_JAVA_ENV,
                                                  LOGGER)
    flag, msg = date_check.execute_check()
    return flag, "\n".join(date_check.all_cli_ret), msg


class ControllerStartingPointDateCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_cli_ret = []
        self.all_err_msg = []

    def execute_check(self):

        try:
            return self.check_controller_starting_point_date()
        except UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, err_msg)

    def check_controller_starting_point_date(self):
        cli_ret, system_date = self.query_controller_list_and_system_date()
        controller_info_list = cliUtil.getVerticalCliRet(cli_ret)
        pass_flag = True
        greater_than_or_equal = True
        is_all_null = True
        err_msg = "hardware.controller.starting.point.date.error_message"
        warn_msg = "hardware.controller.starting.point.date.warning_message"
        for controller_info in controller_info_list:
            controller = controller_info.get("Controller")
            point_date = controller_info.get("Starting Point Date")
            session = controller_info.get("Session")
            if any([not point_date, not session, point_date == NO_PATCH,
                    session == NO_PATCH]):
                continue
            is_all_null = False
            service_date = self.convert_date(point_date, session)
            if service_date < system_date:
                self.all_err_msg.append(get_err_msg(self.lang, err_msg,
                                                    (controller,)))
                pass_flag = False
                continue
            if not self.is_greater_than_or_equal_six_month(
                            service_date, system_date, controller, warn_msg):
                greater_than_or_equal = False
        if is_all_null:
            return cliUtil.RESULT_NOSUPPORT, ""
        if not pass_flag:
            return False, "".join(self.all_err_msg)
        if not greater_than_or_equal:
            return cliUtil.RESULT_WARNING, "".join(self.all_err_msg)
        return True, ""

    def query_controller_list_and_system_date(self):
        cmd = "show controller general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd,
                                                            True, self.lang)
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)

        flag, system_date, date_ret = common.getCurSysDate(self.cli)
        self.all_cli_ret.append(date_ret)
        if flag is not True:
            err_msg = common.getMsg(LANG, "query.system.date.abnormal")
            raise UnCheckException(err_msg, date_ret)
        return cli_ret, system_date

    def is_greater_than_or_equal_six_month(self, service_date, system_date,
                                           controller, warn_msg):
        calculate_date = self.convert_date(system_date, SIX_MONTH)
        if service_date < calculate_date:
            self.all_err_msg.append(
                get_err_msg(self.lang, warn_msg, (controller, )))
            return False
        return True

    def convert_date(self, date, add_month):
        """
        服务起始时间加上服务时长（session）后的日期
        :return:date
        """
        year, month, day = self.query_year_month_day(date)
        new_month = month + int(add_month)
        if new_month <= TWELVE_MONTH:
            month = new_month
        else:
            if new_month % TWELVE_MONTH > 0:
                year = year + new_month // TWELVE_MONTH
                month = new_month % TWELVE_MONTH
            else:
                year = year + new_month // TWELVE_MONTH - 1
                month = TWELVE_MONTH
        month = self.convert_month_day_number(month)
        day = self.convert_month_day_number(day)
        date = "-".join([str(year), month, day])
        return date

    def query_year_month_day(self, date):
        date = datetime.datetime.strptime(date, '%Y-%m-%d')
        return date.year, date.month, date.day

    def convert_month_day_number(self, time):
        return "0{}".format(str(time)) if time <= 9 else str(time)
