# -*- coding: UTF-8 -*-
import time
import traceback

import cliUtil
import common
import hardware
from common import AsynProgress


PY_JAVA_ENV = py_java_env
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
FAN_ON_POWER_PRODUCT_MODEL = [
    "2100 V3",
    "2200 V3",
    "2600 V3",
    "2800 V3",
    "5300 V3",
    "5500 V3",
    "2600F V3",
    "5500F V3",
    "2800 V5",
    "5110 V5",
    "5210 V5",
    "5300 V5",
    "5500 V5",
    "5110F V5",
    "5210F V5",
    "5300F V5",
    "5500F V5",
    "Dorado3000 V3",
    "Dorado5000 V3",
]

UNKNOWN_BOM = "Unknown"
SLEEP_TIME = 5


def execute(cli):
    """
    风扇状态检查:
    所有风扇均在位，Health Status为Normal且Running Status为Running表示正常，其它情况均属于不正常状态。
    :param cli:
    :return:
    """
    my_thread = AsynProgress(PY_JAVA_ENV, LOGGER)
    # 进度开始
    my_thread.start_thread()
    fan_status_check = FanStatusCheck(cli)
    try:

        fan_status_check.execute_check()
        if fan_status_check.all_err_msg:
            return (
                False,
                fan_status_check.get_cli_ret(),
                fan_status_check.get_all_error_msg(),
            )
        return True, fan_status_check.get_cli_ret(), ""

    except common.UnCheckException as e:
        return (
            cliUtil.RESULT_NOCHECK,
            fan_status_check.get_cli_ret(),
            e.errorMsg,
        )
    except Exception:
        LOGGER.logNoPass(traceback.print_exc())
        return (
            cliUtil.RESULT_NOCHECK,
            fan_status_check.get_cli_ret(),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    finally:
        my_thread.setStopFlag(True)


class FanStatusCheck:
    def __init__(self, cli):
        self.cli = cli
        self.all_cli_ret = []
        self.all_err_msg = []
        self.is_fan_on_power_flag = self.is_fan_on_power()

    def get_cli_ret(self):
        return "\n".join(self.all_cli_ret)

    def get_all_error_msg(self):
        return "".join(self.all_err_msg)

    def get_fan_info(self):
        cmd = "show fan"
        LOGGER.logExecCmd(cmd)
        flag, ret, msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        self.all_cli_ret.append(ret)
        common.refreshProcess(PY_JAVA_ENV, 4, LOGGER)
        if flag is not True:
            LOGGER.logSysAbnormal()
            raise common.UnCheckException(msg, ret)

        info_list = cliUtil.getHorizontalCliRet(ret)
        if not info_list:
            err_msg = common.getMsg(LANG, "cannot.get.fan.info")
            LOGGER.logNoPass("Cannot get information about fan")
            raise common.UnCheckException(err_msg, ret)
        return info_list

    def execute_check(self):
        fan_info_list = self.get_fan_info()
        power_id_list = self.get_all_power_id()
        for fan_info in fan_info_list:
            fan_id = fan_info.get("ID", "")
            health_status = fan_info.get("Health Status")
            running_status = fan_info.get("Running Status")
            bom_code = self.hardware_config_xml_with_retry(fan_id)
            bom_code = bom_code if bom_code else UNKNOWN_BOM
            power_bom = None
            is_normal = True
            if health_status != common.STATUS_NORMAL:
                power_bom = self.check_health_status(
                    fan_id, health_status, power_id_list,
                )
                is_normal = False
            if running_status != common.STATUS_RUNNING:
                self.all_err_msg.append(
                    common.getMsg(
                        LANG,
                        "fan.running.status.abnormal",
                        (fan_id, running_status),
                    )
                )
                is_normal = False
            if not is_normal:
                bom_code = power_bom if power_bom else bom_code
                self.all_err_msg.append(
                    common.getMsg(
                        LANG, "hardware.bom.info", (fan_id, bom_code)
                    )
                )

    def check_health_status(
        self, fan_id, health_status, power_id_list,
    ):
        power_bom = None
        if fan_id.lower() in power_id_list:
            power_bom = self.get_power_bom(fan_id)
            self.all_err_msg.append(
                common.getMsg(
                    LANG,
                    "power.health.status.abnormal",
                    (fan_id, health_status),
                )
            )

        else:
            self.all_err_msg.append(
                common.getMsg(
                    LANG, "fan.health.status.abnormal", (fan_id, health_status)
                )
            )
        return power_bom

    def is_fan_on_power(self):
        _, product_model, ret, _ = cliUtil.getProductModelWithCliRet(
            self.cli, LANG
        )
        return product_model in FAN_ON_POWER_PRODUCT_MODEL

    def get_power_bom(self, fan_id):
        """
        获取电源BOM编码

        :param fan_id: 风扇ID
        :return: bom 编码
        """
        cmd = "show power_supply power_supply_id={}".format(fan_id)
        flag, ret, msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        if flag is not True:
            return UNKNOWN_BOM
        info_list = cliUtil.getVerticalCliRet(ret)
        for retDict in info_list:
            bom_number = retDict.get("Serial Number", "")
            if len(bom_number) > 10:
                return bom_number[2:10]
        return UNKNOWN_BOM

    def get_all_power_id(self):
        """
        获取电源ID列表

        :return:
        """
        cmd = "show power_supply"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)

        info_list = cliUtil.getHorizontalCliRet(ret)
        power_id_list = [info.get("ID", "").lower() for info in info_list]
        LOGGER.logNoPass("The dev power ID is : {}".format(power_id_list))
        return power_id_list

    def hardware_config_xml_with_retry(self, fan_id):
        """
        多线程出现读写文件空指针异常，使用重试+文件锁的方式暂时规避此问题
        :param fan_id: 风扇id
        :return: xml parse 结果
        """
        retry_time = 10
        while retry_time > 0:
            try:
                return hardware.getHardwareBomCode(("FAN", PY_JAVA_ENV, LOGGER), self.cli, fan_id)[1]
            except common.UnCheckException as exp:
                LOGGER.logError(str(exp))
                retry_time -= 1
                time.sleep(SLEEP_TIME)
        return ""
