# -*- coding: UTF-8 -*-
import cliUtil
import common
from common import UnCheckException
import traceback
import common_utils
import common_cache

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
localDev = common.getCurDeviceInfo(PY_JAVA_ENV)
ALL_CLI_RET = ''

SAN_DOMAIN_CMD = "show hyper_metro_domain general"
NAS_DOMAIN_CMD = "show fs_hyper_metro_domain general"


def execute(cli):
    '''
    @summary: 双活两端版本及控制器一致性检查
    检查方法
        步骤1 以admin用户登录设备；
        步骤2 执行命令show hyper_metro_domain general，获取本端设备上的双活域ID（ID）和远端设备的ID（Remote Device ID）；
        步骤3 执行命令show fs_hyper_metro_domain general，获取本端设备上的NAS双活域ID（ID）和远端设备的ID（Remote Device ID）
        步骤4 执行命令show hyper_metro_pair general，获取当前阵列配置的所有的双活pair，查询出“ID”字段值；
        步骤5 执行命令show remote_device general，获取远端设备ID（ID）和SN（SN）；
        步骤6 查看SmartKit界面获取设备对应产品型号；
        步骤7 执行命令show upgrade package，获取当前阵列配置的版本信息，分别查询Software Version和HotPatch Version下的“Current Version”字段值；
        步骤8 执行命令show controller general，获取当前阵列配置中“Running Status”为“Online”的控制器数量；
        步骤9 以admin用户登录远端设备；
        步骤10 执行步骤6-8，获取远端阵列的产品型号、详细版本信息及控制器数量。
    检查标准
        1 如果步骤4查询结果无双活pair，则检查通过；
        2 比较步骤6查询到的本端和远端的产品型号，如果两端产品型号不一致，则检查不通过，否则继续检查；
        3 比较步骤7查询到的本端和远端的版本信息、补丁信息。如果两端的版本信息、补丁信息不一致，则检查不通过，否则继续检查；
        4 比较步骤8查询到的本端和远端的控制器数量，如果两端控制器数量一致，则检查通过，否则建议优化。
    修复建议
        1、两端阵列产品型号不一致检查不通过，建议更换阵列保证两端阵列产品型号一致。
        2、两端阵列版本不一致检查不通过，请升级为一致版本。
        3、两端阵列控制器数量不一致提示建议优化，为保证性能建议扩控成一致后配置双活业务。
    '''
    global ALL_CLI_RET
    flag = True
    errMsgAll = ''
    cliRet = ''
    resultList = []
    try:
        localDevSN = str(localDev.getDeviceSerialNumber())
        ALL_CLI_RET += "ON LOCAL DEVICE(SN:%s)\n" % localDevSN

        # 检查是否有双活Domain域
        domainDict = {}
        san_domain_dict = getDomainInfo(localDevSN, SAN_DOMAIN_CMD)
        nas_domain_dict = getDomainInfo(localDevSN, NAS_DOMAIN_CMD)
        domainDict.update(nas_domain_dict)
        domainDict.update(san_domain_dict)

        LOGGER.logInfo("isRisk domainDict: %s, " % domainDict)

        # 检查是否有双活pair
        if not existshyperMetroPair(localDevSN):
            LOGGER.logInfo('***[not hyper metro]***')
            return (True, ALL_CLI_RET, '')

        # 获取远端设备SN
        domainRemoteDevDict = getRemoteDeviceInfo(localDevSN, domainDict)

        # 获取本端型号
        localProductModel = getProductModel(localDevSN)

        # 获取本端版本号和补丁版本
        localProductVersion, localHotPatchVersion = getProductModelVersion(localDevSN)

        # 获取本端控制器数量
        localCtrlNum = getControllerNum(localDevSN)

        # 获取已添加的远端设备。
        flag, addedSnList, errMsg = common.checkAddedRemoteDevSn(PY_JAVA_ENV, LANG)
        if flag != True:
            return common_utils.get_result_bureau(PY_JAVA_ENV, ALL_CLI_RET, errMsg)

        for domainId in domainRemoteDevDict:
            domain_pair_list = get_hyper_pair(domainId, localDevSN, san_domain_dict)
            if not domain_pair_list:
                LOGGER.logInfo("domain {} do not have pair. sn:{}".format(
                    domainId, localDevSN))
                continue

            remoteDevSn = domainRemoteDevDict[domainId]
            if remoteDevSn not in addedSnList:
                errMsgAll += common.getMsg(LANG, "not.add.remote.device.again", remoteDevSn)
                continue

            ALL_CLI_RET += "\n\nON REMOTE DEVICE(SN:%s)" % remoteDevSn
            remoteProductModel = getProductModel(remoteDevSn)
            remoteProductVersion, remotePatchVersion = getProductModelVersion(remoteDevSn)
            remoteCtrlNum = getControllerNum(remoteDevSn)
            if localProductModel != remoteProductModel:
                resultList.append(False)
                errMsgAll += common.getMsg(LANG, "check.hyper.array.product.model.not.pass", (localDevSN, remoteDevSn))

            if localProductVersion != remoteProductVersion or localHotPatchVersion != remotePatchVersion:
                errMsgAll += common.getMsg(LANG, "check.hyper.array.product.version.and.hot.patch.not.pass", (localDevSN, remoteDevSn))
                resultList.append(False)

            if localCtrlNum != remoteCtrlNum:
                resultList.append(cliUtil.RESULT_WARNING)
                errMsgAll += common.getMsg(LANG, "check.hyper.array.controller.num.warning", (localDevSN, remoteDevSn))

        if False in resultList:
            return False, ALL_CLI_RET, errMsgAll

        if cliUtil.RESULT_WARNING in resultList:
            return cliUtil.RESULT_WARNING, ALL_CLI_RET, errMsgAll + common.getMsg(LANG, "controller.num.warning.sugg")

        if cliUtil.RESULT_NOCHECK in resultList:
            return common_utils.get_result_bureau(PY_JAVA_ENV, ALL_CLI_RET, errMsgAll)

        return True, ALL_CLI_RET, errMsg

    except UnCheckException, unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo("UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        if not unCheckException.flag:
            return cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg

        return unCheckException.flag, unCheckException.cliRet, unCheckException.errorMsg
    except BaseException, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(LANG, "query.result.abnormal"))


def getDomainInfo(devSn, cmd):
    """
    获取domain信息
    :param devSn:
    :return:
    """
    global ALL_CLI_RET
    domainDict = {}
    flag, cliRet, errMsg = common.getObjFromFile(PY_JAVA_ENV, LOGGER, devSn,
                                                 cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag == cliUtil.RESULT_NOSUPPORT or common_utils.not_support_nas_domain(cliRet):
        return domainDict
    if flag is not True:
        LOGGER.logInfo("Failed to get information about HyperMetro domain. errMsg:%s" % errMsg)
        raise UnCheckException(
            common.getMsg(LANG, "cannot.get.info", {"zh": u"双活域", "en": "HyperMetro domain"}.get(LANG)), ALL_CLI_RET)

    hyperMetroDomainList = cliUtil.getHorizontalCliRet(cliRet)
    for domainInfo in hyperMetroDomainList:
        domainId = domainInfo.get("ID", '')
        remoteDeviceId = domainInfo.get("Remote Device ID", '')
        domainDict[domainId] = remoteDeviceId

    return domainDict


def get_hyper_pair(domain_id, sn, san_domain):
    if domain_id in san_domain:
        domain_pair_list, pair_info_list = common_utils.get_hyper_metro_pair_id_list(
            sn, domain_id, PY_JAVA_ENV, LOGGER, LANG)
        return domain_pair_list
    return get_nas_hyper_metro_pair_id_list(sn, domain_id, PY_JAVA_ENV, LOGGER, LANG)


def get_nas_hyper_metro_pair_id_list(dev_sn, domain_id, env, logger, lang):
    """
    获取nas pair
    :param dev_sn:
    :param domain_id:
    :param env:
    :param logger:
    :param lang:
    :return:
    """
    flag, ret, msg, pair_list = common_cache.get_nas_pair_from_cache(
        env, logger, dev_sn, lang)
    if flag is not True:
        logger.logInfo("Failed to get information about HyperMetro Pair")
        raise UnCheckException(msg)

    pair_id_list = []
    for pair_info in pair_list:
        if pair_info.get("Domain ID") == domain_id:
            pair_id_list.append(pair_info.get("ID"))

    return pair_id_list


def getRemoteDeviceInfo(devSn, domainDict):
    """
    获取远端设备信息
    :param devSn:
    :param domainDict:
    :return:
    """
    global ALL_CLI_RET
    remoteDeviceDict = {}
    cmd = "show remote_device general"
    flag, cliRet, errMsg = common.getObjFromFile(PY_JAVA_ENV, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about remote device. errMsg:%s" % errMsg)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", {"zh": u"远端设备", "en": "remote device"}.get(LANG)),
                               ALL_CLI_RET)

    remoteDeviceList = cliUtil.getHorizontalCliRet(cliRet)
    for remoteDev in remoteDeviceList:
        remoteDeviceId = remoteDev.get("ID", '')
        remoteDeviceSn = remoteDev.get("SN", '')
        for domainId in domainDict:
            if remoteDeviceId == domainDict[domainId]:
                remoteDeviceDict[domainId] = remoteDeviceSn

    return remoteDeviceDict


def getProductModel(devSn):
    '''
    @summary: 获取型号
    '''
    py_obj = py_java_env.get("objectForPy")
    product_model = py_obj.get("productMode_{}".format(devSn))
    return product_model


def getControllerNum(devSn):
    '''
    @summary: 获取运行的控制器
    '''
    global ALL_CLI_RET
    cmd = "show controller general"
    flag, cliRet, errMsg = common.getObjFromFile(PY_JAVA_ENV, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        raise UnCheckException(errMsg, ALL_CLI_RET)

    resDictList = cliUtil.getVerticalCliRet(cliRet)
    ctrlList = []
    for resDict in resDictList:
        if resDict.get("Running Status") == "Online":
            ctrlList.append(resDict)
    return len(ctrlList)


def existshyperMetroPair(devSn):
    '''
    @summary: check whether or not has hyper metro
    '''
    global ALL_CLI_RET
    cmd = "show hyper_metro_pair general"
    flag, ret, msg = common.getObjFromFile(
        PY_JAVA_ENV, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, ret)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about HyperMetro Pair")
        raise UnCheckException(msg)
    pair_list = cliUtil.getHorizontalNostandardCliRet(ret)
    return bool(pair_list)


def getProductModelVersion(devSn):
    '''
    @summary: get product model, product version and patch version
    '''
    global ALL_CLI_RET
    cmd = "show upgrade package"
    flag, cliRet, errMsg = common.getObjFromFile(PY_JAVA_ENV, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        raise UnCheckException(errMsg, ALL_CLI_RET)

    hotPatchIndex = cliRet.find("HotPatch Version")
    versionCliRet = cliRet[:hotPatchIndex]
    hotPatchCliRet = cliRet[hotPatchIndex:]
    versionDictList = cliUtil.getHorizontalCliRet(versionCliRet)
    hotPatchDictList = cliUtil.getHorizontalCliRet(hotPatchCliRet)

    productVersion = ""
    for versionDict in versionDictList:
        productVersion = versionDict.get("Current Version")
        if productVersion:
            break

    hotPatchVersion = ""
    for versionDict in hotPatchDictList:
        hotPatchVersion = versionDict.get("Current Version")
        if hotPatchVersion:
            break

    return productVersion, hotPatchVersion





