# -*- coding: UTF-8 -*-
import common
import cliUtil
import re

from cbb.frame.base import baseUtil
from cbb.frame.base import product
from cbb.frame.context import contextUtil
from com.huawei.ism.tool.obase.exception import ToolException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    SmartIO 2.0 FC端口做异构
    :param cli:
    :return:
    """
    ALL_CLI_RET = ""
    try:
        flag, product_model, cli_ret, err_msg = \
            cliUtil.getProductModelWithCliRet(cli, LANG)
        if flag is not True:
            return flag, cli_ret, err_msg
        flag, product_version, patch_ver, cli_ret_version, err_msg = \
            common.get_soft_and_patch_version(cli, LOGGER, LANG)
        ALL_CLI_RET = cli_ret_version

        if flag is not True:
            return flag, ALL_CLI_RET, err_msg
        # 检查是否为风险版本
        risky_version, failed_version = check_product_version(product_model,
                                                              product_version)
        if risky_version is False:
            LOGGER.logInfo("Not in risky version, pass.")
            return True, ALL_CLI_RET, ""
        if check_hot_path_version(product_version, patch_ver) is True:
            LOGGER.logInfo("check hot path pass.")
            return True, ALL_CLI_RET, ""
        # 查询链路信息：
        lwwpnList, cliRet = get_fc_link_lwwpns(cli)
        ALL_CLI_RET = common.joinLines(ALL_CLI_RET,
                                       cliRet)

        if not lwwpnList:
            LOGGER.logInfo("No heterogeneous link, pass.")
            return True, ALL_CLI_RET, ""
        # 判断是否为系统软件版本属于V500R007C60 Kunpeng或6.0.0
        if failed_version is False:
            port_id_list, cli_ret_port = get8GFCPortInLwwpnList(cli, lwwpnList)
            ALL_CLI_RET = common.joinLines(ALL_CLI_RET,
                                           cli_ret_port)
            if not port_id_list:
                LOGGER.logInfo("Max Speed not 32000")
                return True, ALL_CLI_RET, ""
            err_meg = common.getMsg(LANG,
                                    "smartio2.0.fault.logs.check.notpass",
                                    ",".join(list(set(port_id_list))))
            return False, ALL_CLI_RET, err_meg
        err_meg = common.getMsg(LANG,
                                "smartio2.0.notpass.version."
                                "logs.check.notpass", product_version)
        return False, ALL_CLI_RET, err_meg
    except common.UnCheckException as unCheckException:
        errMsg = unCheckException.errorMsg
        allCliRet = common.joinLines(ALL_CLI_RET, unCheckException.cliRet)
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(
            LANG, "query.result.abnormal")
    finally:
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s".format(ret))
        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def check_product_version(product_model, product_version):
    """
    如果步骤2中查询到当前系统版本号为
V300R006C50到V300R006C60SPC100之间的V3版本
或V500R007C30到V500R007C60SPC200（也包含带Kunpeng字段的版本）之间的V5版本
或Dorado V300R002C10到Dorado V300R002C20SPC001之间的Dorado V3版本
或Dorado V6 6.0.0版本
    :return: 第一个为风险版本，第二个为不通过
    """
    failed_version = False

    if "6.0.0" in product_version:
        failed_version = True
    if product_version >= 'V300R006C50' and \
            product_version <= 'V300R006C60SPC100':
        return True, failed_version
    if product_version >= 'V500R007C30' and \
            product_version <= 'V500R007C60SPC200':
        return True, failed_version
    if baseUtil.isDoradoDev(product_model) and not product.isDigitalVer(
            product_version) and product_version >= 'V300R002C10' \
            and product_version <= 'V300R002C20SPC002':
        return True, failed_version
    if failed_version is True:
        return True, failed_version
    return False, failed_version


def get_fc_link_lwwpns(cli):
    """
    @summary: 获取链路信息中的lwwpn。
    """
    lwwpnList = []

    cmd = "adm show link"
    flag, cliRet, errMsg = common.executeDebugCommand(cli, cmd, PY_JAVA_ENV,
                                                      LOGGER)
    if flag is not True:
        raise common.UnCheckException(errMsg, cliRet)

    for line in cliRet.split("linkid"):
        lwwpn = get_lwwpn_from_line(line)
        if lwwpn:
            lwwpnList.append(lwwpn)

    return lwwpnList, cliRet


def get_lwwpn_from_line(line):
    """
    @summary:从一行字符串中获取lwwpn。
          如果链路信息中存在“linktype(FC)”，说明设备有使用FC做异构。
    """
    if "linktype(FC)" not in line:
        return ""
    if "lwwpn(" not in line or ")" not in line:
        return ""
    if "vendor(" not in line:
        return ""

    pattern = re.compile(r'vendor\((.*?)\)',
                         flags=re.IGNORECASE)
    match = pattern.search(line)
    if not match:
        return ""
    vendor = match.group(1).upper().strip()
    if vendor == 'HUAWEI' or vendor == 'HUASY':
        return ""

    pattern_lwwpn = re.compile(r'lwwpn\((.*?)\)',
                               flags=re.IGNORECASE)
    lwwpn = pattern_lwwpn.search(line)
    if not lwwpn:
        return ""
    return lwwpn.group(1)


def get8GFCPortInLwwpnList(cli, lwwpnList):
    """
    @summary: 获取存在于链路信息中的8G FC端口。
    """
    portIdList = []

    # 获取8G FC 接口卡

    # 获取FC端口
    cmd = "show port general physical_type=FC"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        raise common.UnCheckException(errMsg, cliRet)

    if cliUtil.queryResultWithNoRecord(cliRet):
        return [], cliRet

    # 8G FC接口卡上的FC端口即为8G FC端口
    infoDictList = cliUtil.getHorizontalCliRet(cliRet)
    for infoDict in infoDictList:
        portid = infoDict.get("ID")
        wwn = infoDict.get("WWN")
        if (not portid) or (wwn not in lwwpnList):
            continue

        workRate = infoDict.get("Max Speed(Mbps)", "")
        if workRate.isdigit() and int(workRate) == 32000:
            portIdList.append(portid)

    return portIdList, cliRet


def check_hot_path_version(product_ver, hot_ver):
    """
    2020-5-30版本新增:加入版本补丁判断，减少误报
    product_ver： 系统软件版本
    hot_ver：热补丁版本
    :return: 是否不需要继续检查
    """
    # 白名单补丁
    hot_ver_dict = {'V300R006C50SPC100': 'V300R006C50SPH109',
                    'V500R007C30SPC100': 'V500R007C30SPH109',
                    'V300R002C10SPC100': 'V300R002C10SPH109',
                    'V300R002C20': 'V300R002C20SPH002'}

    # 6.0.0 版本打了补丁就算通过
    if "6.0.0" in product_ver and 'SPH' in hot_ver:
        return True
    # 判断补丁版本是否在风险版本中
    if product_ver in hot_ver_dict:
        best_hot_ver = hot_ver_dict.get(product_ver, "")
        if hot_ver >= best_hot_ver and not re.search(r"SPH\d[6-9]\d", hot_ver):
            return True
    return False
