# -*- coding: UTF-8 -*-
import decimal
import traceback
import cliUtil
import common
from common import UnCheckException


LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ALL_CLI_RET = ''


def execute(cli):
    '''
        SAN克隆空间预留检查(DoradoV3) ； 
                     检查方法：
                    步骤1 以admin用户登录设备；
                    步骤2 执行命令show clone general，查询克隆特性；
                    步骤3 执行命令show storage_pool general|filterRow column=Usage\sType predict=equal_to value=LUN，记录所有POOL的总容量（Total Capacity字段值）和空闲容量（Free Capacity字段的值）；
                    步骤4 执行show lun general pool_id=POOLID|filterColumn include columnList=ID（POOLID为空闲容量小于1%的POOL的ID），记录POOL中LUN LIST；
                    步骤5 依次检查POOL中LUN是否有克隆LUN，执行命令show lun general lun_id=LUNID|filterColumn include columnList=ID,Subscribed\sCapacity,HyperMetro\sID(s),Remote\sReplication\sID(s),Split\sClone\sID(s),Snapshot\sID(s),LUN\sMigration,Mirror\sType。如果克隆split clone字段不为--，记录下LUN ID，POOLID和Subscribed Capacity，计算相同POOL下的克隆lun容量总和，检查是否配置有其它特性并做标记
                   检查标准：
            1. 若不存在克隆配置则检查通过，否则继续步骤2；
            2. 如果所有POOL均满足：空闲容量（Free Capacity字段的值）大于POOL的总容量（Total Capacity字段值）的1%，则检查通过，否则继续步骤3；
            3. 如果只有克隆，则使用POOL下的克隆LUN使用容量总和 * 0.005与POOL的空闲容量进行比较，如果小于则检查通过，如果大于或者等于则提示用户需要对POOL进行扩容；
            4. 如果除了克隆还有其它特性，则使用POOL下的克隆LUN容量总和 * 0.01与POOL的空闲容量进行比较，如果小于则检查通过，如果大于或者等于则提示用户需要对POOL进行扩容；
    '''
    global ALL_CLI_RET
    flag = True
    errMsg = ""
    #存储池空闲容量/总容量的阀值
    poolStandard = "0.01"
    #存储池空闲容量/总克隆lun容量的阀值
    lunStandard = "0.005"
    try:
        #检查版本号是否支持
        isSucc = isVersionSupport()
        #如果查询版本为空时返回未检查
        if isSucc == False:
            return (cliUtil.RESULT_NOSUPPORT, "", "")

        #查询是否存在克隆配置
        flag, featureExist = isFeatureExist(cli, LANG)
        
        #查询是否存在克隆配置失败（下发命令失败或发生异常）
        if flag != True:          
            return (flag, ALL_CLI_RET, common.getMsg(LANG, "query.result.abnormal"))
        
        if not featureExist:
            return (True, ALL_CLI_RET, "")
        
        #检查存储池空闲容量
        flag, errMsg, freeCapDict = checkPoolFreeCapacity(cli, LANG, poolStandard, LOGGER)
        
        #查询存储池详情失败
        if flag != True:          
            raise UnCheckException(errMsg, ALL_CLI_RET, flag)

        if len(freeCapDict) == 0:
            return (True, ALL_CLI_RET, '')

        #检查增值LUN的总容量是否达标
        flag, errMsg, lunTotalCapDict = checkCloneLunToatalCapacity(cli, freeCapDict.keys(), LANG, LOGGER)       
        #查询LUN信息失败
        if not flag:
            return (cliUtil.RESULT_NOCHECK, ALL_CLI_RET, errMsg)
        
        if flag != True:
            return (flag, ALL_CLI_RET, errMsg)
        
        if len(lunTotalCapDict) == 0:
            return (True, ALL_CLI_RET, errMsg)
        #检查每个pool的空闲容量和克隆lun总容量的比与阀值比较
        for poolid in lunTotalCapDict:
            #计算lun总容量与0.005的值
            lunTotalCapRet = lunTotalCapDict.get(poolid) * decimal.Decimal(lunStandard)
            #lun总容量与0.005的值与空闲容量比较
            if lunTotalCapRet >= freeCapDict.get(poolid):
                flag = False
                lunTotalCapRet = "%.3f" % lunTotalCapRet  
                LOGGER.logInfo("lunTotalCapRet:%s" % lunTotalCapRet)             
                errMsg += common.getMsg(LANG, "clone.pool.free.capacity.lower", (poolid, lunTotalCapRet) )                
                LOGGER.logInfo("errMsg:%s" % errMsg)
        return (flag, ALL_CLI_RET, errMsg)
    
    except UnCheckException, unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo(u"UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        if unCheckException.flag == False:
            return (cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg)
        return (unCheckException.flag, unCheckException.cliRet, unCheckException.errorMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(LANG, "query.result.abnormal"))

    
def isFeatureExist(cli, lang):
    """
    @summary: 执行命令show clone general，
                                    查询克隆特性是否存在；
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        featureExist: True(存在克隆配置)
                      False(不存在克隆配置)
    """
    
    isSucc = True
    featureExist = False
    global ALL_CLI_RET
    
    cmd = "show lun_clone general"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    
    if not cliUtil.hasCliExecPrivilege(cliRet):
        errMsg = cliUtil.getMsg(lang, "has.not.cli.privilege")
        return (cliUtil.RESULT_NOSUPPORT, featureExist)
    
    if flag != True:
        if flag == False:
            errMsg =  common.getMsg(LANG, "query.result.abnormal")            
        raise UnCheckException(errMsg, ALL_CLI_RET, flag)
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, featureExist)
    
    #走到这里说明命令成功执行且回显正常，回显正常即说明有对应特性
    featureExist = True
            
    return (isSucc, featureExist)
    
    
def isVersionSupport():
    """
     @summary: 检查当前版本是否支持，只支持DoradoV3R1C21及更高版本
     
     @return: 
        isSucc：True/False，是否支持 ;NULL表示无法获取到版本
    """
    supportVersion = "V300R001C21"
    deviceInfo = common.getCurDeviceInfo(py_java_env)
    #从上下文获取不到设备信息
    if not deviceInfo:
        errMsg =  common.getMsg(LANG, "cannot.get.product.version.info")        
        raise UnCheckException(errMsg, ALL_CLI_RET, False) 
    currentVersion = deviceInfo.getProductVersion()    
    LOGGER.logInfo("currentVesion : %s" % currentVersion)
    #从上下文获取不到版本号
    if currentVersion == "":
        errMsg =  common.getMsg(LANG, "cannot.get.product.version.info")        
        raise UnCheckException(errMsg, ALL_CLI_RET, False)  
    if currentVersion >= supportVersion:
        return True
    return False
def checkPoolFreeCapacity(cli, lang, standard, logger):
    """
    @summary: 执行命令show storage_pool general,
                                    检查是否所有pool的Free Capacity与Total Capacity的比值均高于标准
    @param standard : 标准比值
    @return: flag : 方法是否正常执行
             cliRet : cli回显
             errMsg : 错误消息
             freeCapDict : 比值低于标准的ID为key、Free Capacity为值的字典
    """

    global ALL_CLI_RET
    cmd = "show storage_pool general"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)   
    if flag != True: 
        if flag == False:
            errMsg =  common.getMsg(LANG, "query.result.abnormal")        
        raise UnCheckException(errMsg, ALL_CLI_RET, flag)        

    freeCapDict = {}
    unkownPoolList = []
    linesList = cliUtil.getHorizontalCliRet(cliRet)
    for line in linesList:   
        idNum = line.get("ID").strip()
        freeCap = line.get("Free Capacity")
        totalCap = line.get("Total Capacity")
        #把获取的容量转换为GB
        freeTransRet = common.changUnit2GBDecimal(freeCap)
        totalTransRet = common.changUnit2GBDecimal(totalCap)
        
        if not freeTransRet[0] or not totalTransRet[0]:
            unkownPoolList.append(idNum)               
            flag = False
            continue
          
        freeCap = freeTransRet[1]
        totalCap = totalTransRet[1]
        if totalCap == 0:
            errMsg = common.getMsg(lang, "get.pool.capacity.failed",idNum)
            raise UnCheckException(errMsg, ALL_CLI_RET, False) 
        
        if freeCap/totalCap <= decimal.Decimal(standard):
            freeCapDict[idNum] = freeCap
            
    if len(unkownPoolList) != 0:
        errMsg = common.getMsg(lang, "get.pool.capacity.failed", ",".join(unkownPoolList))
    return (flag, errMsg, freeCapDict)    

def checkCloneLunToatalCapacity(cli, poolIDList, lang, logger): 
    """
     @summary: 获取以pool id为key、pool的克隆LUN的总容量为值的字典
    @param freeCapDict : 
    @return: flag : 方法是否正常执行
             cliRet : cli回显
             errMsg : 错误消息
             freeCapDict : 比值低于标准的ID为key、Free Capacity为值的字典
    """
    global ALL_CLI_RET
    errMsg = ''
    lunTotalCapDict = {} 
    unkownLunList = []   
    flag = True        
    cmd = "show lun general pool_id=%s|filterColumn include columnList=ID,Capacity,Is\sClone"
    for poolId in poolIDList:        
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd % poolId, True, lang)
        ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
        if flag != True:    
            if flag == False:
                errMsg =  common.getMsg(LANG, "query.result.abnormal")         
            raise UnCheckException(errMsg, ALL_CLI_RET, flag)    
        
        if cliUtil.queryResultWithNoRecord(cliRet):
            continue
        
        lunDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
        logger.logInfo("pool id :%s, lun information:%s" % (poolId, lunDictList))
        lunTotalCapacity = decimal.Decimal("0.0")
        
        for lunDict in lunDictList:
            lunid = lunDict.get("ID").strip()
            isClone = lunDict.get("Is Clone").lower()
            lunCap = lunDict.get("Capacity")
            lunTranRet = common.changUnit2GBDecimal(lunCap) 
            
            if not lunTranRet[0]:
                unkownLunList.append(lunid)               
                flag = False
                continue
            #如果是克隆lun就统计容量
            if isClone == "yes":
                lunTotalCapacity += lunTranRet[1]  
                              
        lunTotalCapDict[poolId] =  lunTotalCapacity
        
    if len(unkownLunList) != 0:
        errMsg += common.getMsg(lang, "get.LUN.capacity.failed", ",".join(unkownLunList))    
    return (flag, errMsg, lunTotalCapDict)
        