# -*- coding: UTF-8 -*-
'''
@summary: alua configuration check 
'''
import cliUtil
import common
from common import AsynProgress
from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache

LANG = common.getLang(py_java_env)
PY_JAVA_ENV = py_java_env
LOGGER = common.getLogger(PY_LOGGER, __file__)
FAILOVER_RIGHT_MODE_LOWER = ["common alua",
                             "special mode"]
FAILOVER_WRONG_MODE_LOWER = ["no alua",
                             "old alua",
                             "--"]
#initialize client echos
HOST_INITIATOR_TYPE = ["FC",
                       "ISCSI",
                       "IB"]

def execute(cli):
    '''
        @summary: NDMP recommended configuration check
        @return: (status, cliRet, errMsg)
    '''
    cliRet = ""
    errMsg = ""
    cliEchosAll = ""
    myPthread = AsynProgress(PY_JAVA_ENV, LOGGER)
    myPthread.start_thread()
    try:
        isCheckOk, isHyperMetroLunSupported, cliEchos, errMsg = \
                common.isHyperMetroLunFeatureSupported(cli, LANG, LOGGER)
        if not isCheckOk:
            
            return cliUtil.RESULT_NOCHECK, cliEchos, errMsg
        elif not isHyperMetroLunSupported:
            return cliUtil.RESULT_NOSUPPORT, cliEchos, ""
        cliEchosAll += cliEchos
        
        isXcutedSuccess, hostList, cliEchos = getDeviceMountedHosts(cli)
        cliEchosAll += cliEchos
        if not isXcutedSuccess:
            LOGGER.logInfo("try to fetch device mounted hosts but failed to be executed ")
            return cliUtil.RESULT_NOCHECK, cliEchosAll, common.getMsg(LANG, "query.host.info.failed")
        if len(hostList) < 1:
            LOGGER.logInfo("current device has not attached host!")
            return True, cliEchosAll, ""
        
        checkLunRequiredHosts = []
        failedCheckInitiatorHosts = []
        failedCheckLunHosts = []
        failedCheckLunDetailHostMap = {}
        needOptmizedHosts = []
        
        isIbSuccess, checkLunRequiredHosts, failedCheckInitiatorHosts, cliEchos = checkIbInitiators(cli, hostList)
        cliEchosAll += cliEchos
        isFcAndIscsiSucc, hostLists, cliEchos = checkISCSIandFCInitiator(cli)
        cliEchosAll += cliEchos
        checkLunRequiredHosts.extend(hostLists)
        checkLunRequiredHosts =  [i for i in set(checkLunRequiredHosts)]
        for hostId in checkLunRequiredHosts:
            isCheckSuccess, hasHyperMetroLun, failedLunIds, echos = checkHostHyperMetroLunIds(cli, hostId)
            cliEchosAll += echos
            if not isCheckSuccess:
                failedCheckLunHosts.append(hostId)
                continue
            if hasHyperMetroLun:
                LOGGER.logInfo("current host %s contains HyperMetro LUN..." % str(hostId))
                needOptmizedHosts.append(hostId)
                continue
            if  len(failedLunIds) > 0:
                LOGGER.logInfo("found failed Luns %s" % str(failedLunIds))
                failedCheckLunDetailHostMap[hostId] = ", ".join(failedLunIds)     
                
        return checkItemStatus(isIbSuccess, isFcAndIscsiSucc, needOptmizedHosts, failedCheckInitiatorHosts, 
                               failedCheckLunHosts, failedCheckLunDetailHostMap, cliEchosAll)
        
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal"))
    finally:
        myPthread.setStopFlag(True)
        
        
def checkItemStatus(isIbSuccess, isFcAndIscsiSucc, needOptmizedHosts, failedIbInitiatorHosts, 
                    failedCheckLunHosts, failedCheckLunDetailHostMap, cliEchosAll):
    '''
    @summary: summarize error Message
    '''
    LOGGER.logInfo("checking Alua configuration result...")
    returnStatus = True
    errorMsg = ""
    failedIbInitiatorHosts = [i for i in failedIbInitiatorHosts if i not in needOptmizedHosts]
    if not isFcAndIscsiSucc:
        LOGGER.logInfo("failed to get FC & ISCSI initiator information")
        returnStatus = cliUtil.RESULT_NOCHECK
        errorMsg += common.getMsg(LANG, "failed.to.query.fc.iscsi.initiators")
    if not isIbSuccess:
        LOGGER.logInfo("failed to get ib initiator information")
        returnStatus = cliUtil.RESULT_NOCHECK
        errorMsg += common.getMsg(LANG, "failed.to.query.ib.initiators")

    if  failedCheckLunHosts or failedCheckLunDetailHostMap or failedIbInitiatorHosts:
        returnStatus = cliUtil.RESULT_NOCHECK
        
        if failedIbInitiatorHosts:
            LOGGER.logInfo("current failed checking initiators hosts :" + str(failedIbInitiatorHosts))
            params = ", ".join(failedIbInitiatorHosts)
            errorMsg += common.getMsg(LANG, "failed.query.ib.initiators.about.host", params)
        if failedCheckLunHosts:
            LOGGER.logInfo("current failed checking LUNs host :" + str(failedCheckLunHosts))
            params = ", ".join(failedCheckLunHosts)
            errorMsg += common.getMsg(LANG, "failed.fetch.hosts.info", params)
        if failedCheckLunDetailHostMap:
            LOGGER.logInfo("current failed checking LUNs details lists :" + str(failedCheckLunDetailHostMap.keys()))
            for failedHostKey in failedCheckLunDetailHostMap:
                errorMsg += common.getMsg(LANG, \
                             "failed.fetch.luns.info.on.host", \
                             (failedHostKey, failedCheckLunDetailHostMap[failedHostKey]))
        
    if returnStatus != True:
        errorMsg = common.getMsg(LANG, "failed.fetch.following.hosts.initiators.luns.info", errorMsg)
    
    if needOptmizedHosts:
        returnStatus = cliUtil.RESULT_WARNING
        LOGGER.logInfo("currently needed to be optmized host lists :" + str(needOptmizedHosts))
        errorMsg = common.getMsg(LANG, \
                    "following.host.alua.config.need.opimize", ", ".join(needOptmizedHosts)) + errorMsg
    return returnStatus, cliEchosAll , errorMsg  
    
def isHostInitiatorAtRisk(initiatorList, initiatorType):
    '''
    @summary: parse peering Host Initiator configuration   
    @return: isHostInitiatoratRisk
    '''
    isRiskInitiator = False

    for initiator in initiatorList:
        multipathType = initiator.get("Multi Path Type", "") if\
                         initiatorType == HOST_INITIATOR_TYPE[2] else initiator.get("Multipath Type", "")
        failOverMode = initiator.get("Failover Mode", "")
        if  not multipathType or not failOverMode :
            LOGGER.logInfo("current initiator configuration(%s, %s) is not valid..." % (multipathType, failOverMode))
            continue
        if multipathType.lower() != "default" :
            if failOverMode.lower() not in FAILOVER_RIGHT_MODE_LOWER:
                LOGGER.logInfo("current initiator configuration(%s, %s) needs to be optimized." %\
                                                                     (multipathType, failOverMode))
                isRiskInitiator = True
        
    return isRiskInitiator

def executeCommand(cli, cmd):
    '''
    @summary: execute commands & record cli echo
    '''
    LOGGER.logExecCmd(cmd)
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        py_java_env, cli, cmd, LOGGER)
    return flag, cli_ret, err_msg


def checkHostHyperMetroLunIds(cli, hostId):
    '''
    @summary: check which LUNs designated to this host have configured as hyper-metro mode
    @return: ,isQrySuccess, hashypermetroLun, failedLunIds
    '''  
    cliRetAll = ""  
    failedLunIds = []
    cmd = "show host lun host_id=" + hostId
    echoStatus, cliRet, errMsg = executeCommand(cli, cmd)
    cliRetAll += cliRet
    if echoStatus != True or not cliRet or "error:" in cliRet.lower() or "error :" in cliRet.lower() :
        return False , False, [], cliRetAll
    if "successfully" in cliRet.lower():
        return True, False, [], cliRetAll
    
    lunInfoList = cliUtil.getHorizontalCliRet(cliRet)
    if not lunInfoList:
        LOGGER.logInfo("failed to parse LUN list about host %s" % str(hostId))
        return False, False, [], cliRetAll

    for lunInfo in lunInfoList:
        lunId = lunInfo.get("LUN ID", "")
        if not lunId or not str(lunId).isdigit():
            continue
        isSuccess, isHyperMetroLun, cliEcho = isThisHyperMetroLun(cli, lunId)
        
        if not isSuccess:
            failedLunIds.append(lunId)
        if isHyperMetroLun:
            cliRetAll += cliEcho
            return True, True, [], cliRetAll
        
    #if no hyper-metro Lun detected after the loop ,return the failed LUNS
    LOGGER.logInfo("current host %s has following LUN unchecked:%s!" % (hostId, failedLunIds))
    return True, False, failedLunIds, cliRetAll

def isThisHyperMetroLun(cli, lunId):
    '''
    @summary: checking whether the LUN is hyper metro LUN 
    @return isQrySuccess, ishypermetroLun
    '''
    cmd = "show lun general lun_id=%s" % lunId
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        PY_JAVA_ENV, cli, cmd, LOGGER)
    if flag is not True:
        return False, False, cli_ret
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, False, cli_ret
    lun_dict_list = cliUtil.getVerticalCliRet(cli_ret)
    for line in lun_dict_list:
        hypermetro_status = line.get('HyperMetro ID(s)')
        if hypermetro_status != "" and hypermetro_status != "--":
            LOGGER.logInfo(
                "current LUN %s is hyper metro LUN!" % str(lunId))
            return True, True, cli_ret
    LOGGER.logInfo("current LUN %s is not hyper metro LUN, continue..."
                   % str(lunId))
    return True, False, cli_ret
            
def getDeviceMountedHosts(cli):
    '''
    @summary: get hosts list moundted to current device
    @return: isXcutedSuccess, hostList, errorMsg, cliEchos
    '''
    hostIdList = []
    cmd = "show host general"
    echoStatus, cliRet, errMsg = executeCommand(cli, cmd)
    if echoStatus != True or not cliRet or "error:" in cliRet.lower() or "error :" in cliRet.lower() :
        return False , [], cliRet
    if "successfully" in cliRet.lower():
        return True, [], cliRet
    hostGeneralList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    if not hostGeneralList:
        LOGGER.logInfo("failed in parsing host information mounted on device...")
        return False, [] , cliRet
    
    for hostGeneral in hostGeneralList:
        hostId = hostGeneral.get("ID", "")
        if not hostId or not str(hostId).isdigit():
            continue
        hostIdList.append(hostId)

    return True, hostIdList, cliRet
    
def checkIbInitiators(cli, hostList):
    '''
    @summary: check which IB_initiator needs to check for lun feature
    @return: needRecheckIbInitiator, failedIbInitiator, cliEhos
    '''
    needRecheckIbInitiator = []
    failedIbInitiator = []
    cliEchosAll = ""
    #if no record exists, do not looping
    cmd = "show ib_initiator general"
    echoStatus, cliRet, errMsg = executeCommand(cli, cmd)
    if echoStatus == cliUtil.RESULT_NOSUPPORT:
        return True, [], [], cliEchosAll
    if echoStatus != True:
        LOGGER.logInfo("this device failed in processing show IB_initiator command: " + unicode(errMsg))
        return False, [] , [], cliEchosAll
    if "successfully" in   cliRet.lower() and len(cliRet.splitlines()) <= 4 :
        LOGGER.logInfo("this device found not matched IB_initiator configuration ")
        return True , [], [] , cliEchosAll
    if not isIbInitiatorHasWrongAlua(cliRet):
        LOGGER.logInfo("this device found not high risk IB_initiator configuration ")
        return True , [], [] , cliEchosAll
    for hostId in hostList:
        cmd = "show ib_initiator general host_id=" + hostId
        echoStatus, cliRet, errMsg = executeCommand(cli, cmd)
        cliEchosAll += cliRet
        if echoStatus == cliUtil.RESULT_NOSUPPORT:
            continue
        elif echoStatus != True:
            LOGGER.logInfo("this device failed in processing show IB_initiator command: " + unicode(errMsg))
            failedIbInitiator.append(hostId)
        elif "successfully" not in cliRet.lower():
            ibInitiators = cliUtil.getHorizontalCliRet(cliRet)
            if len(ibInitiators) < 1:
                LOGGER.logInfo("this device failed in parsing  IB_initiator configuration: " + str(hostId))
                failedIbInitiator.append(hostId)
                
            elif isHostInitiatorAtRisk(ibInitiators, HOST_INITIATOR_TYPE[2]):
                LOGGER.logInfo("Host %s's  Ib_initiator has un-optmized configuration" % str(hostId))
                needRecheckIbInitiator.append(hostId)
                continue
                
    # hosts list must not be duplicated
    needRecheckIbInitiator = [i for i in set(needRecheckIbInitiator)]
    failedIbInitiator = [i for i in set(failedIbInitiator)]
    
    return  True, needRecheckIbInitiator, failedIbInitiator, cliEchosAll 

def checkISCSIandFCInitiator(cli):
    '''
    @summary: checking which host's FC initiator or ISCSI initiator is required for lun feature checking
    @return echoStatus, needRecheckInitiator
    '''

    needRecheckInitiator = []
    cmd = "show initiator"
    echoStatus, cliRet, errMsg = executeCommand(cli, cmd)
    if echoStatus != True or not cliRet or "error:" in cliRet.lower() or "error :" in cliRet.lower() :
        LOGGER.logInfo("this device failed in processing show initiator command: " + unicode(errMsg))
        return False, [], cliRet
        
    elif "successfully" not in cliRet.lower():
        fcInitiatorPoi = cliRet.find("WWN")
        iscsiInitiatorPoi = cliRet.find("iSCSI IQN")
        if fcInitiatorPoi < 0 and iscsiInitiatorPoi < 0:
            LOGGER.logInfo("failed to parse current device initiator configuration")
            return False, [], cliRet
        else:    
            fcInitiatorList = []
            iscsiInitiatorList = []
            if fcInitiatorPoi >= 0: 
                cuttedCliEcho = cliRet[fcInitiatorPoi - 2:iscsiInitiatorPoi - 2] \
                if iscsiInitiatorPoi > fcInitiatorPoi else cliRet[fcInitiatorPoi - 2:]
                fcInitiatorList = cliUtil.getHorizontalCliRet(cuttedCliEcho)
            if iscsiInitiatorPoi >= 0:
                cuttedCliEcho = cliRet[iscsiInitiatorPoi - 2:fcInitiatorPoi - 2] \
                if fcInitiatorPoi > iscsiInitiatorPoi else cliRet[iscsiInitiatorPoi - 2:]
                iscsiInitiatorList = cliUtil.getHorizontalCliRet(cuttedCliEcho)
            if not fcInitiatorList and not iscsiInitiatorList:
                LOGGER.logInfo("checking initiator configuration but no matched records was found.") 
                return True, [], cliRet
            if fcInitiatorList:
                LOGGER.logInfo("checking Fiber channel initiator configuration...")
                needRecheckInitiator.extend(getInitiatorRecheckList(fcInitiatorList))
            if iscsiInitiatorList:
                LOGGER.logInfo("checking ISCSI initiator configuration...")
                needRecheckInitiator.extend(getInitiatorRecheckList(iscsiInitiatorList))  
    needRecheckInitiator = [i for i in set(needRecheckInitiator)]
    return True, needRecheckInitiator, cliRet

def getInitiatorRecheckList(initiatorList):
    '''
    @summary: parse initiator information(FC & ISCSI)
    '''
    riskInitiatorHosts = []
    for initiator in initiatorList:
        hostId = initiator.get("Host ID", "")
        multipathType = initiator.get("Multipath Type", "")
        failOverMode = initiator.get("Failover Mode", "")
        if not hostId or not hostId.isdigit() or not multipathType or not failOverMode :
            LOGGER.logInfo("current initiator configuration(%s, %s, %s) is not valid..." %\
                            (hostId, multipathType, failOverMode))
            continue
        if multipathType.lower() != "default" :
            if failOverMode.lower() not in FAILOVER_RIGHT_MODE_LOWER:
                LOGGER.logInfo("current initiator configuration(%s, %s, %s) needs to be optimized." % \
                               (hostId, multipathType, failOverMode))
                riskInitiatorHosts.append(hostId)
        
    riskInitiatorHosts = [i for i in set(riskInitiatorHosts)]
    return riskInitiatorHosts    

def isIbInitiatorHasWrongAlua(cmdEcho):
    '''
    @summary: check whether the Ib initiator has third party  multi-path and unconfigured ALUA
    @return: True if it does
    '''
    hasRiskInitiator = False
    initiatorList = cliUtil.getHorizontalCliRet(cmdEcho)
    for initiator in initiatorList:
        multipathType = initiator.get("Multi Path Type", "")
        failOverMode = initiator.get("Failover Mode", "")
        if multipathType.lower() != "default" :
            if failOverMode.lower() not in FAILOVER_RIGHT_MODE_LOWER:
                hasRiskInitiator = True
                break
    return hasRiskInitiator