# -*- coding: UTF-8 -*-

from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.checkitem.context_adapter import InspectContext
from utils import Products
import cliUtil
import common


LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
CHECK_VERSION = {
    "V300R001C00SPC100": "",
    "V300R001C01SPC100": "",
    "V300R001C21SPC100": "",
    "V300R001C30SPC100": "",
    "V300R002C00SPC100": "",
    "V300R002C10SPC100": "V300R002C10SPH119",
    "V300R002C20": "V300R002C20SPH006",
    "6.0.0": "",
    "6.0.1": "SPH15",
    "6.1.RC1": "",
    "6.1.RC2": "",
    "6.1.RC3": "",
    "6.1.0": "SPH2",
}
LIMIT_TIME = 90
INTERVAL = 2
RECHECK_TIME = 60


def execute(cli):
    try:
        # 进度条刷新
        common.threadUpProcess(py_java_env, LIMIT_TIME, INTERVAL, LOGGER)
        # 进度开始
        common.inProcess(py_java_env)
        context = InspectContext(py_java_env, cli)
        inspect_context = context.get_context()
        inspect_context["logger"] = LOGGER
        inspect_context["lang"] = LANG
        check_obj = CheckItem(inspect_context, cli)
        status, err_msg = check_obj.check()
        LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
        all_ret = check_obj.get_origin_info()
        return context.get_result(status, err_msg, all_ret)
    finally:
        common.finishProcess(py_java_env)


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.cli = cli
        self.dev_version = ""
        self.patch_version = ""
        self.all_msg = ""
        self.flag = CheckStatus.PASS
        self.local_logger = LOGGER
        self.disk_domain_list = []
        self.recheck_domain_list = []
        self.sys_version = ""
        self.patch_version = ""

    def check(self):
        try:
            # 判断是否为风险版本
            result, msg = self.is_risk_version()
            if result is cliUtil.RESULT_NOCHECK:
                return CheckStatus.NOCHECK, msg
            if not result:
                return CheckStatus.PASS, ""

            # 获取硬盘域信息
            is_suc, msg = self.get_domain_info()
            if is_suc is not CheckStatus.PASS:
                return CheckStatus.NOCHECK, msg

            # 第一次检查
            result, msg = self.check_domains()
            self.all_msg += msg
            if not self.recheck_domain_list:
                return result, self.all_msg
            else:
                # 不通过的硬盘域，等待60s再检查
                common.safeSleep(RECHECK_TIME)
                LOGGER.logInfo("recheck water level.")
                zone_flag, msg = self.check_domains_for_water_level()
                self.flag = self.get_result(self.flag, zone_flag)
                if zone_flag is not CheckStatus.PASS:
                    self.all_msg += msg

            display_patch = self.patch_version if self.patch_version else "--"
            sug_patch = CHECK_VERSION.get(self.sys_version)
            if self.flag is CheckStatus.NOTPASS and sug_patch:
                msg_key = "zone_recycle.water.level.check.fail.patch"
                self.all_msg += common.getMsg(LANG,
                                              msg_key,
                                              (self.sys_version,
                                               display_patch,
                                               sug_patch))
            else:
                msg_key = "zone_recycle.water.level.check.fail"
                self.all_msg += common.getMsg(LANG,
                                              msg_key,
                                              (self.sys_version, display_patch))

            return self.flag, self.all_msg
        except Exception as exception:
            self.logger.logException(exception)
            return CheckStatus.NOCHECK, common.getMsg(LANG,
                                                      "query.result.abnormal")

    def is_risk_version(self):
        """
        如果为风险版本，则返回True，否则False
        :return:
        """
        result, ret, msg, self.sys_version, self.patch_version = \
            common.getVersion(self.cli, self.lang)
        self.add_origin_info(ret)
        if result is not True:
            return CheckStatus.NOCHECK, msg

        # 如果不是风险版本，不检查
        if self.sys_version not in CHECK_VERSION:
            return False, ""

        # 如果不存在补丁，需要检查
        if not self.patch_version or self.patch_version == "--":
            return True, ""

        # 不存在解决问题的补丁，需要检查
        patch_info = CHECK_VERSION.get(self.sys_version)
        if not patch_info:
            return True, ""

        # 如果当前补丁小于解决问题的补丁，需要检查
        return Products.compareVersion(self.patch_version, patch_info) < 0, ""

    def get_domain_info(self):
        """
        获取硬盘域的归属控制器
        :return:
        """
        is_suc, disk_domain_list, ret = \
            cliUtil.get_disk_domain_ctrl_by_domain(self.cli, self.lang)
        if not is_suc:
            is_suc, disk_domain_list, disk_ret = \
                cliUtil.get_disk_domain_ctrl_by_disk(self.cli, self.lang)
            self.add_origin_info(disk_ret)
            if not is_suc:
                err_msg = common.getMsg(LANG, "cannot.get.domain.info")
                return CheckStatus.NOCHECK, err_msg
        else:
            self.add_origin_info(ret)
        self.disk_domain_list = disk_domain_list
        return CheckStatus.PASS, ""

    def check_zone_recycle_data(self, domain_id, ctrl_list):
        """
        检查回收水位的调节开关是否打开
        :param domain_id:
        :param ctrl_list:
        :return:
        """
        for ctrl in ctrl_list:
            cmd = "show zone_recycle perform pool_id=%s " \
                  "zone_type=data controller=%s" % (domain_id, ctrl)
            result, ret, msg = \
                cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.lang)
            self.add_origin_info(ret)
            if result is True:
                ret_list = cliUtil.getVerticalCliRet(ret)
                if not ret_list:
                    msg = common.getMsg(self.lang, "get.zone_recycle.fail",
                                        (domain_id, ctrl))
                    return CheckStatus.NOCHECK, msg
                band_width = ret_list[0].get("Band Width(MB)")
                if band_width == "0":
                    return CheckStatus.PASS, ""
                continue

            msg = common.getMsg(self.lang, "get.zone_recycle.fail",
                                (domain_id, ctrl))
            return CheckStatus.NOCHECK, msg

        return CheckStatus.NOTPASS, ""

    def check_zone_recycle_water_level(self, domain_id, ctrl_list):
        """
        检查各控制器的回收水位是否一致
        :param domain_id:
        :param ctrl_list:
        :return:
        """
        band_width_set = set()
        ctrl_zone = []
        for ctrl in ctrl_list:
            cmd = "show zone_recycle info info_type=waterlevel " \
                  "pool_id=%s controller=%s" % (domain_id, ctrl)
            result, ret, msg = \
                cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.lang)
            self.add_origin_info(ret)
            if result is True:
                ret_list = cliUtil.getVerticalCliRet(ret)
                if not ret_list:
                    msg = common.getMsg(self.lang,
                                        "get.zone_recycle.water.level.fail",
                                        (domain_id, ctrl))
                    return CheckStatus.NOCHECK, msg
                data_zone = ret_list[0].get("Data Zone(%)")
                ctrl_zone.append("%s:%s" % (ctrl, data_zone))
                band_width_set.add(data_zone)
                continue
            msg = common.getMsg(self.lang, "get.zone_recycle.water.level.fail",
                                (domain_id, ctrl))
            return CheckStatus.NOCHECK, msg

        if len(band_width_set) == 1:
            return CheckStatus.PASS, ""

        msg = common.getMsg(self.lang, "zone_recycle.water.level.different",
                            (domain_id, ", ".join(ctrl_zone)))
        return CheckStatus.NOTPASS, msg

    def check_domains(self):
        """
        检查硬盘域的水位调节开关和水位是否满足条件
        :return:
        """
        all_msg = ""
        recheck_domain_list = []
        flag = CheckStatus.PASS
        for disk_domain in self.disk_domain_list:
            disk_domain_id = disk_domain.get("ID", "")
            ctrl_list = disk_domain.get("Controller", "").split(",")

            # 检查水位调节开关
            data_result, msg = \
                self.check_zone_recycle_data(disk_domain_id, ctrl_list)
            if data_result in [CheckStatus.PASS, CheckStatus.NOCHECK]:
                flag = self.get_result(flag, data_result)
                all_msg += msg
                continue

            # 检查水位是否一致
            zone_flag, zone_err_msg = \
                self.check_zone_recycle_water_level(disk_domain_id, ctrl_list)
            flag = self.get_result(flag, zone_flag)
            if zone_flag is CheckStatus.NOTPASS:
                recheck_domain_list.append(disk_domain)

        self.recheck_domain_list = recheck_domain_list
        return flag, all_msg

    def check_domains_for_water_level(self):
        """
        检查硬盘域的回收水位是否一致
        :return:
        """
        all_msg = []
        flag = True
        for domian in self.recheck_domain_list:
            disk_domain_id = domian.get("ID", "")
            ctrl_list = domian.get("Controller", "").split(",")
            zone_flag, zone_err_msg = \
                self.check_zone_recycle_water_level(disk_domain_id,
                                                    ctrl_list)
            flag = self.get_result(flag, zone_flag)
            if zone_flag is not CheckStatus.PASS:
                all_msg.append(zone_err_msg)
        return flag, "".join(all_msg)

    def get_result(self, pre_result, cur_result):
        """
        存在多种结果时，取高优先级结果
        :param pre_result:
        :param cur_result:
        :return:
        """
        if pre_result is CheckStatus.NOTPASS \
                or cur_result is CheckStatus.NOTPASS:
            return CheckStatus.NOTPASS

        if pre_result is CheckStatus.NOCHECK \
                or cur_result is CheckStatus.NOCHECK:
            return CheckStatus.NOCHECK

        return CheckStatus.PASS
