# -*- coding: UTF-8 -*-

from cbb.frame.checkitem.context_adapter import InspectContext
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN

import common_utils
import common
from risk_version_config import SOFTWARE_IO_DELAY_RISK_DICT

PY_JAVA_ENV = py_java_env
JAVA_LOGGER = PY_LOGGER
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(JAVA_LOGGER, __file__)

RISK_INTERFACE_MODEL = [
    "4 ports 10Gb ETH I/O",
    "2 ports 100Gb/40Gb ETH I/O",
]
SPECIAL_SMART_IO_MODEL = "4 ports SmartIO I/O"


def execute(cli):
    context = InspectContext(PY_JAVA_ENV, cli)
    inspect_context = context.get_context()
    inspect_context["logger"] = JAVA_LOGGER
    inspect_context["lang"] = LANG
    check_obj = CheckItem(inspect_context, cli)
    status, err_msg = check_obj.check()
    all_ret = check_obj.get_origin_info()
    LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
    return context.get_result(status, err_msg, all_ret)


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.dev_version = ""
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = False
        self.cli = cli
        self.env = PY_JAVA_ENV
        self.local_logger = LOGGER
        self.target_patch = ''

    def execute(self):
        flag, msg = self.init_version_and_patch()
        if flag is not True:
            return CheckStatus.NOCHECK, msg
        if not self.is_risk_version():
            return CheckStatus.PASS, ""

        if not self.has_risk_interface():
            return CheckStatus.PASS, ""

        return CheckStatus.NOTPASS, common_utils.get_err_msg(
            LANG, "software.io.delay.risk.patch.not.pass",
            (self.dev_version, self.patch_version, self.target_patch))

    def init_version_and_patch(self):
        """
        初始化版本、补丁信息
        :return:
        """
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)
        self.patch_version = self.patch_version if self.patch_version else '--'
        if flag is not True:
            return False, common.getMsg(
                LANG, "cannot.get.hot.patch.version.info")
        return flag, ''

    def is_risk_version(self):
        """
        是否风险版本
        :return: True 风险版本, False 非风险。
        """
        self.target_patch = SOFTWARE_IO_DELAY_RISK_DICT.get(self.dev_version)
        if not self.target_patch:
            return False

        return common_utils.get_patch_value(
            self.patch_version) < common_utils.get_patch_value(
            self.target_patch)

    def has_risk_interface(self):
        """
        是否存在1822风险接口卡
        :return:True: 存在风险，False: 不存在风险。
        """
        interface_info_list = self.dsl(
            "exec_cli 'show interface_module' | horizontal_parser",
        )

        for interface_info in interface_info_list:
            interface_model = interface_info.get("Model")
            if [
                model for model in RISK_INTERFACE_MODEL
                if model in interface_model
            ]:
                return True

            if SPECIAL_SMART_IO_MODEL not in interface_model:
                continue

            interface_id = interface_info.get("ID")
            if self.is_eth_mode_interface(interface_id):
                return True

        return False

    def is_eth_mode_interface(self, interface_id):
        """
        是否Ethernet模式的smartio接口卡
        :return:
        """
        interface_info = self.dsl(
            "exec_cli 'show interface_module interface_module_id={}' "
            "| vertical_parser".format(interface_id),
        )
        return interface_info.get("Run Mode") == "Ethernet"
