# -*- coding: UTF-8 -*-
import ast
import traceback

import cliUtil
import common
import common_cache
import common_utils
from cliUtil import get_lun_id_list_with_cache

from cbb.frame.base.config import LUN_CACHE_MIRROR_PATCH_CHECK
from cbb.frame.cli.cli_with_cache import (
    execute_cmd_in_cli_mode_with_cache,
)
from common_utils import is_support_read_only_user_enter_debug

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env

WRITE_MIRROR_STATUS_UNKNOWN = 'unknown write mirror status'
FAILEED_TRANSFORM = 'failed transform object to volume'
OBJ_NOT_EXIT = "object not exsit"
NO_SNAPSHOT_DEV_LIST = ["2800 V3", "2200 V3", "2600 V3 for Video"]
DEDUPCOMPRESS_LUN_LIST = ["V300R003C20", "V300R003C20SPC100", "V300R003C20SPC200", "V300R006C00", "V300R006C00SPC100"]
CURRENTPROCESS = 0
CHECK_NODE_NUMBERS = 1
VOLUMETOLUNDICT = {}

def execute(cli):
    '''
    @summary: 存储设备通过OpenStack创建LUN的过程中如果将LUN的Cache镜像功能关闭，后续业务运行过程中，如果控制器复位，将概率导致数据丢失，业务中断。
    '''
    try:
        switch = None
        checkRet, switch = cliUtil.openDeveloperSwitch(cli, LANG)
        if checkRet[0] != True:  # 修改备注：openDeveloperSwitch返回没有False
            return (checkRet[0], checkRet[1], checkRet[2])
        chkResult, allCliRet, errMsg = doCheck(cli)
    except:
        LOGGER.logError(str(traceback.format_exc()))
        return (cliUtil.RESULT_NOCHECK, '', common.getMsg(LANG, "query.result.abnormal"))
    else:
        return chkResult, joinLines(checkRet[1], allCliRet), errMsg
    finally:
        if switch != None and switch == False:
            cliUtil.closeDeveloperSwitch(cli, LANG)
        #退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        #退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)
        refreshProcess(PY_JAVA_ENV, 100)
    
def doCheck(cli): 
    '''
    @summary: Check LUN cache status and work mode of every LUN and snapshot.
    @param cli: CLI connection
    @return: (chkResut, cliRet, errMsg)
    '''
    allCliRet, allErrMsg = '', ''
    flag, cliRet, errMsg = isSuperAdmin(cli)
    if flag != True:
        allCliRet = joinLines(allCliRet, cliRet) 
        allErrMsg = joinLines(allErrMsg, errMsg)
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)
    
    flag, contrIdList, errMsg, cliRet = cliUtil.getControllerIdListWithRet(cli, LANG)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        allCliRet = joinLines(allCliRet, cliRet)
        allErrMsg = joinLines(allErrMsg, errMsg)
        LOGGER.logError('Query controlled ID list failed, check terminated!')
        # 修改备注：getControllerIdListWithRet返回False是未检查
        if not flag:
            return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
        return (flag, allCliRet, allErrMsg)
    
    refreshProcess(PY_JAVA_ENV, 1)
    
    controllerNum = len(contrIdList)
    #单控不存在lun镜像问题，检查通过
    if controllerNum == 1:
        LOGGER.logWarning('Single controller running, check PASSED!')
        return (True, allCliRet, allErrMsg) 
    
    #>>获取设备版本    
    (
        flag,
        productVersion,
        p_patch,
        ret,
        msg
     ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
    allCliRet = joinLines(allCliRet, ret)
    if flag is not True:
        allErrMsg = joinLines(allErrMsg, msg)
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)

    refreshProcess(PY_JAVA_ENV, 5)
    
    global is_big_card
    is_big_card = cliUtil.is_dev_of_12GB_SAS_Share_Expansion(
        PY_JAVA_ENV, productVersion
    )
    LOGGER.logInfo('Device is with:' + ' big card.' if is_big_card else ' small card.')
    
    #>>获取LUN ID List
    flag, cliRet, errMsg, lunIdList = get_lun_id_list_with_cache(cli,
                                                                 LANG,
                                                                 PY_JAVA_ENV,
                                                                 LOGGER)
    allCliRet = joinLines(allCliRet, cliRet)
    refreshProcess(PY_JAVA_ENV, 10)
    
    if flag != True:
        LOGGER.logError('Query LUN ID list failed, check terminated!')
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)
    
    lunNum = len(lunIdList)
    if lunNum == 0:
        return (True, allCliRet, allErrMsg)
    
    #>>获取快照信息
    isLicenseMissing = False
    isQrySucc, cliRet, errMsg, snapshotIdList = getSnapshotIdList(cli)
    allCliRet = joinLines(allCliRet, cliRet)
    if isQrySucc != True:
        allErrMsg = joinLines(allErrMsg, errMsg)
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)

    #>>获取重删压缩LUN信息
    isQrySucc, cliRet, errMsg, dedupCompressLunIdList = getDedupCompressLunIdList(cli)
    allCliRet = joinLines(allCliRet, cliRet)
    if isQrySucc != True:
        allErrMsg = joinLines(allErrMsg, errMsg)
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg) 
        
    refreshProcess(PY_JAVA_ENV, 15)
    
    #>>获取文件系统数量
    flag, cliRet, errMsg, fileSystemNum = getFileSystemNum(cli)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        LOGGER.logWarning('Query file system number not success, the device may not support file system!')
    
    lunAndFileSysTotCnt = len(lunIdList) + fileSystemNum
    LOGGER.logInfo('the system product version is %s, the number of lun and filesystem is %s' % (productVersion, str(lunAndFileSysTotCnt)))

    global CHECK_NODE_NUMBERS
    CHECK_NODE_NUMBERS = getCheckNodeNumbers(productVersion, lunAndFileSysTotCnt, controllerNum)
    
    global CURRENTPROCESS
    CURRENTPROCESS = 20
    refreshProcess(PY_JAVA_ENV, CURRENTPROCESS)

    #>>product version in (.., V3R3C00) || [v3R6C00SPC100, )
    if productVersion < 'V300R003C00' or productVersion >= 'V300R006C00SPC100':
        # 数量小于400，且能执行对应命令才行。
        if lunAndFileSysTotCnt <= 400 and can_execute_switch_all_cmd(
                productVersion, p_patch
        ):
            if common.is18000(PY_JAVA_ENV, cli):
                ret = getConnectionByContrIp(PY_JAVA_ENV, cli)
                if ret[0] == False:
                    allCliRet = joinLines(allCliRet, ret[1])
                    allErrMsg = joinLines(allErrMsg, ret[2])
                    return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                contrCli = ret[3]
                isChkSucc, cliRet, errMsg, notMirrorState, _, failedTransformLunOrSnapshotList = chk2StatesOfAllObjsOnOneNode(contrCli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                PY_JAVA_ENV.get("sshManager").releaseConnection(contrCli)
            else:  
                isChkSucc, cliRet, errMsg, notMirrorState, _, failedTransformLunOrSnapshotList = chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
            allCliRet = joinLines(allCliRet, cliRet)
            allErrMsg = joinLines(allErrMsg, errMsg)
            notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
            if not isChkSucc:
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
            elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList:
                return False, allCliRet, allErrMsg
            elif isLicenseMissing:
                return cliUtil.RESULT_WARNING, allCliRet, allErrMsg
            elif failedTransformLunOrSnapshotList:
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
            else:
                return True, allCliRet, allErrMsg
                
        else:#LUN and file system number > 400,check one bye one.
            if common.is18000(PY_JAVA_ENV, cli):
                ret = getConnectionByContrIp(PY_JAVA_ENV, cli)
                if ret[0] == False:
                    allCliRet = joinLines(allCliRet, ret[1])
                    allErrMsg = joinLines(allErrMsg, ret[2])
                    return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                contrCli = ret[3]
                chkResult, cliRet, errMsg = chkCacheStatusOfAllObjsOnOneNodeOneByOne(contrCli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                PY_JAVA_ENV.get("sshManager").releaseConnection(contrCli)
            else:  
                chkResult, cliRet, errMsg = chkCacheStatusOfAllObjsOnOneNodeOneByOne(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
            
            allCliRet = joinLines(allCliRet, cliRet)
            allErrMsg = joinLines(allErrMsg, errMsg)
            
            if chkResult == True and isLicenseMissing:
                chkResult = cliUtil.RESULT_WARNING   
            return (chkResult, allCliRet, allErrMsg)
        
    elif  'V300R003C00' <= productVersion < 'V300R006C00SPC100':
        # 数量大于400，或者未安装补丁不能执行对应命令。
        if lunAndFileSysTotCnt > 400 or not can_execute_switch_all_cmd(
                productVersion, p_patch
        ):
            if common.is18000(PY_JAVA_ENV, cli):
                ret = getConnectionByContrIp(PY_JAVA_ENV, cli)
                if ret[0] == False:
                    allCliRet = joinLines(allCliRet, ret[1])
                    allErrMsg = joinLines(allErrMsg, ret[2])
                    return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                contrCli = ret[3]
                chkResult, cliRet, errMsg = chkCacheStatusOfAllObjsOnOneNodeOneByOne(contrCli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                PY_JAVA_ENV.get("sshManager").releaseConnection(contrCli)
            else:  
                chkResult, cliRet, errMsg = chkCacheStatusOfAllObjsOnOneNodeOneByOne(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
            
            allCliRet = joinLines(allCliRet, cliRet)
            allErrMsg = joinLines(allErrMsg, errMsg)
            if True == chkResult:
                #Remind user to check Event.txt file for further checking. 
                allErrMsg += common.getMsg(LANG, "all.lun.snapshot.write.mirror.plz.chk.further.ref.event.file")
                return cliUtil.RESULT_WARNING, allCliRet, allErrMsg
            elif cliUtil.RESULT_NOCHECK == chkResult:
                #Remind user that some LUNs or snapshots write mirror status not checked.
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
            else:
                #some LUNs or snapshots not mirror
                return False, allCliRet, allErrMsg
        else:#LUN and file system number <= 400,check all write mirror and work mode status.
            if common.is18000(PY_JAVA_ENV, cli):
                chkResult, cliRet, errMsg = chk2StatesOfAllObjsOnAllNodesByContrCli(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, contrIdList)
            else:
                chkResult, cliRet, errMsg = chk2StatesOfAllObjsOnAllNodes(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, contrIdList)
            
            if chkResult == True and isLicenseMissing:
                chkResult = cliUtil.RESULT_WARNING
            
            allCliRet = joinLines(allCliRet, cliRet)
            allErrMsg = joinLines(allErrMsg, errMsg)
            return chkResult, allCliRet, allErrMsg


def can_execute_switch_all_cmd(p_version, p_patch):
    """
    是否可以使用cache show obj v switch all命令的补丁版本
    :param p_version: 版本
    :param p_patch: 补丁
    :return: True: 可以执行。 False: 不能执行
    """
    return all(
        [
            LUN_CACHE_MIRROR_PATCH_CHECK.get(p_version),
            not common_utils.is_special_patch(p_patch),
            p_patch >= LUN_CACHE_MIRROR_PATCH_CHECK.get(p_version),
        ]
    )


def getEngineCtrlNumDict(contrIdList):
    """
    @summary: 获取引擎下的控制器数量
    """
    engNumDict = {}
    
    for contrId in contrIdList:
        engineId = contrId[:1]
        engNumDict[engineId] = engNumDict.get(engineId, 0) + 1
    
    return engNumDict

    
def getLunIdList(cli):
    """
    @summary: 获取设备上所有LUN的ID
    """

    lun_id_list = []
    cmd = "show lun general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        PY_JAVA_ENV, cli, cmd, LOGGER)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about LUN.")
        return flag, cli_ret, err_msg, lun_id_list
    
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, cli_ret, err_msg, lun_id_list
        
    lunInfoDictList = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    if len(lunInfoDictList) == 0:
        err_msg = common.getMsg(LANG, "query.result.abnormal")
        LOGGER.logInfo("The number of parsed LUN is 0.")
        return False, cli_ret, err_msg, lun_id_list
    
    for lunInfoDict in lunInfoDictList:
        lun_id_list.append(lunInfoDict.get("ID"))
    
    return True, cli_ret, err_msg, lun_id_list


def getFileSystemNum(cli):
    """
    @summary: 获取设备上文件系统的数量
    @param cli: CLI connection.
    @return: (QrySuccess, cliRet, errMsg, fileSystemNumber)
    """
    cmd = "show file_system general|filterColumn include columnList=ID"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag != True: 
        LOGGER.logInfo("Failed to get information about file system.")
        return (False, cliRet, errMsg, 0)
    
    fileSystemInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    fileSystemNum = len(fileSystemInfoDictList)
    if fileSystemNum == 0:
        LOGGER.logInfo("The result of parse file system is 0, may be wrong.")
        return (True, cliRet, errMsg, fileSystemNum)

    LOGGER.logInfo('File system number is:' + unicode(fileSystemNum))           
    return (True, cliRet, errMsg, fileSystemNum)

def getConnectedEngineId(managementPortDictList): 
    """
    @summary: Get the ID of engine that toolkit has connected.
    @param managementPortDictList: Management port information dictionary.
    @return: Number of engine that toolkit has current connected.
    """
    currentContrIp = PY_JAVA_ENV.get("devInfo").getIp()
    LOGGER.logInfo("The IP that ToolKit connected is %s." % currentContrIp)
    
    for managementPortDict in managementPortDictList:
        ip = managementPortDict.get("IPv4 Address")
        if ip == currentContrIp:
            managementPortId = managementPortDict.get("ID")
            engineId = managementPortId.split(".")[0][-1:]
            LOGGER.logInfo("The engine that ToolKit connected is %s." % engineId)
            return  engineId

    return ""


def getAllEngineReachableIP(mgtPortDictList):
    """
    @summary: Get reachable IP address of every engine.
    @param mgtPortDictList: Management port IP address information dictionary.
    @return: {engineId: reachable_IP_address}
    """
    engineIpDict = {}
    for managementPortInfoDict in mgtPortDictList:
        ip = managementPortInfoDict.get("IPv4 Address")
        portRunningStatus = managementPortInfoDict.get("Running Status")
        if portRunningStatus == "Link Up" and ip not in ["", "--"]:
            managementPortId = managementPortInfoDict.get("ID")
            if "DSW" not in managementPortId:
                engineId = managementPortId.split(".")[0][-1:]
                ipList = engineIpDict.get(engineId, [])
                ipList.append(ip)
                engineIpDict.update({engineId:ipList})
    
    return engineIpDict
        

def getSnapshotIdList(cli):
    """
    @summary: 获取所有的快照id
    @param cli: CLI connection.
    @return: (flag, cli_ret, err_msg, snap_id_list)
    """
    snap_id_list = []

    cmd = "show snapshot general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        py_java_env, cli, cmd, LOGGER)

    if flag is False:
        LOGGER.logInfo("Failed to get information about snapshot.")
        return False, cli_ret, err_msg, snap_id_list
    elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
        return True, cli_ret, err_msg, snap_id_list
    elif cliUtil.queryResultWithNoRecord(cli_ret):
        return True, cli_ret, err_msg, []
      
    snap_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    if len(snap_dict_list) == 0:
        LOGGER.logWarning("CLI result parsing may have failed.")
        return False, cli_ret, err_msg, snap_id_list
    
    for snap_info_dict in snap_dict_list:
        snapshot_id = snap_info_dict.get("ID")
        snap_id_list.append(snapshot_id)
    
    LOGGER.logInfo('Snapshot number is:' + unicode(len(snap_id_list)))
     
    return True, cli_ret, err_msg, snap_id_list


def getDedupCompressLunIdList(cli):
    """
    @summary: 获取所有ROW LUN的id
    @param cli: CLI connection.
    @return: (isQrySucc, cliRet, errMsg, snapshotIdList)
    """
    dedupCompressLunIdList = []
    
    cmd = "show dedup_compress_lun general|filterColumn include columnList=Relate\sLUN\sID"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, LANG)
    if flag == False: 
        LOGGER.logInfo("Failed to get information about dedup_compress_lun.")
        return (False, cliRet, errMsg, dedupCompressLunIdList)
    elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
        return (True, cliRet, errMsg, dedupCompressLunIdList)
    elif cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg, dedupCompressLunIdList)
      
    dedupCompressLunInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    if len(dedupCompressLunInfoDictList) == 0:
        LOGGER.logWarning("dedupCompressLun number is 0, CLI result parsing may have failed.")
        return (False, cliRet, errMsg, dedupCompressLunIdList)
    
    for dedupCompressLunInfoDict in dedupCompressLunInfoDictList:
        dedupCompressLunId = dedupCompressLunInfoDict.get("Relate LUN ID")
        dedupCompressLunIdList.append(dedupCompressLunId)
    
    LOGGER.logInfo('dedupCompressLun number is:' + unicode(len(dedupCompressLunIdList)))   
     
    return (True, cliRet, errMsg, dedupCompressLunIdList)


def chkSingleObjCacheStatus(cli, objId):
    '''
    @summary: 检查单个LUN或者快照的镜像写状态
    @param cli: CLI connection
    @param objId: LUN or snapshot ID
    @return: True --write mirror; False-- Not mirror; 'Unknown'
    '''
    flag, ret, volume_id = transformLunToVolume(cli, objId)
    all_ret = ret
    if flag is not True:
        return FAILEED_TRANSFORM, all_ret, ''

    cmd = "cache show obj %s state" % volume_id
    excuFlag, cliRet, errMsg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG, cliUtil.endWithSignToDiagnose)
    all_ret += "{}\n".format(cliRet)
    if excuFlag is not True:
        LOGGER.logSysAbnormal()
        LOGGER.logInfo("Failed to get information of LUN %s cache mirror." % objId)
        return WRITE_MIRROR_STATUS_UNKNOWN, all_ret, errMsg

    if ":" not in cliRet.lower() or "mirror state" not in cliRet.lower():
        return WRITE_MIRROR_STATUS_UNKNOWN, all_ret, ''

    for cliRetLine in cliRet.splitlines():
        fields = cliRetLine.split(":")
        if len(fields) < 2:
            continue
        fieldName = fields[0].strip().lower()
        fieldValue = fields[1].strip().lower()
        if fieldName == "mirror state" and fieldValue == "mirror":
            return True, all_ret, ''
        elif fieldName == "mirror state" and fieldValue != "mirror":
            return False, all_ret, ''
        
    return False, all_ret, ''

def chkCacheStatusOfAllObjsOnOneNodeOneByOne(cli, lunIdList, snapshotIdList, dedupCompressLunIdList):
    """
    @summary: check the write mirror status of some LUNs and some snapshots on any controller(not need to heart beat).
    @param cli: CLI connection
    @param lunIdList: LUN ID List
    @param snapshotIdList: Snapshot ID list
    @return: (check result, error message)
    """
    notMirrorLunList = []
    notMirrorSnapshotList = []
    notMirroeDedupCompressLunList = []
    
    unknownLunMirrorStaIdList = []
    unknownSnapshotMirrorStaIdList = []
    unknownDedupCompressLunStaIdList = []
    
    failedTransformLunList = []
    failedTransformSnapshotList = []
    failedTransformDedupCompressLunList = []

    notExsitLunList = []
    notExsitSnapshotList = []
    notExsitDedupCompressLunList = []
    
    allCliRet, allErrMsg = '', '' 
    
    objTotalNum = len(lunIdList) + len(snapshotIdList) + len(dedupCompressLunIdList)
    LOGGER.logInfo("The total number of object is %s." % str(objTotalNum))
    #检查一个对象的进度
    global CURRENTPROCESS
    oneObjProcess = (1.0 / objTotalNum) * (100 - CURRENTPROCESS)
    LOGGER.logInfo("oneObjProcess is %s." % str(oneObjProcess))
    
    for lunId in lunIdList:
        wmStatus, cliRet, errMsg = chkSingleObjCacheStatus(cli, lunId)
        allCliRet += ('\n' + cliRet)
        if wmStatus == False:
            notMirrorLunList.append(lunId)
        elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
            allErrMsg += ('\n' + errMsg)
            unknownLunMirrorStaIdList.append(lunId)
        elif wmStatus == OBJ_NOT_EXIT:
            notExsitLunList.append(lunId)
        elif wmStatus == FAILEED_TRANSFORM:
            failedTransformLunList.append(lunId)
        
        #刷新检查进度
        global CURRENTPROCESS
        if int(CURRENTPROCESS + oneObjProcess) - int(CURRENTPROCESS) >= 1:
            refreshProcess(PY_JAVA_ENV, CURRENTPROCESS + oneObjProcess)
        CURRENTPROCESS += oneObjProcess
            
    
    for snapshotId in snapshotIdList:
        wmStatus, cliRet, errMsg = chkSingleObjCacheStatus(cli, snapshotId)
        allCliRet += ('\n' + cliRet)
        if wmStatus == False:
            notMirrorSnapshotList.append(snapshotId)
        elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
            allErrMsg += ('\n' + errMsg)
            unknownSnapshotMirrorStaIdList.append(snapshotId)
        elif wmStatus == OBJ_NOT_EXIT:
            notExsitSnapshotList.append(snapshotId)
        elif wmStatus == FAILEED_TRANSFORM:
            failedTransformSnapshotList.append(snapshotId)
        
        #刷新检查进度
        global CURRENTPROCESS
        if int(CURRENTPROCESS + oneObjProcess) - int(CURRENTPROCESS) >= 1:
            refreshProcess(PY_JAVA_ENV, CURRENTPROCESS + oneObjProcess)
        CURRENTPROCESS += oneObjProcess 
           
    for dedupCompressLunId in dedupCompressLunIdList:
        wmStatus, cliRet, errMsg = chkSingleObjCacheStatus(cli, dedupCompressLunId)
        allCliRet += ('\n' + cliRet)
        if wmStatus == False:
            notMirroeDedupCompressLunList.append(dedupCompressLunId)
        elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
            allErrMsg += ('\n' + errMsg)
            unknownDedupCompressLunStaIdList.append(dedupCompressLunId)
        elif wmStatus == OBJ_NOT_EXIT:
            notExsitDedupCompressLunList.append(dedupCompressLunId)
        elif wmStatus == FAILEED_TRANSFORM:
            failedTransformDedupCompressLunList.append(dedupCompressLunId)
        
        ##刷新检查进度
        global CURRENTPROCESS
        if int(CURRENTPROCESS + oneObjProcess) - int(CURRENTPROCESS) >= 1:
            refreshProcess(PY_JAVA_ENV, CURRENTPROCESS + oneObjProcess)
        CURRENTPROCESS += oneObjProcess
            
    if notMirrorLunList:
        sortByInt(notMirrorLunList)
        allErrMsg += common.getMsg(LANG, "lun.not.mirror", ",".join(notMirrorLunList))        
    
    if notMirrorSnapshotList:
        sortByInt(notMirrorSnapshotList)
        allErrMsg += common.getMsg(LANG, "snapshot.not.mirror", ",".join(notMirrorSnapshotList))
    
    if notMirroeDedupCompressLunList:
        sortByInt(notMirroeDedupCompressLunList)
        allErrMsg += common.getMsg(LANG, "dedupCompress.lun.not.mirror", ",".join(notMirroeDedupCompressLunList)) 
        
    if unknownLunMirrorStaIdList:
        sortByInt(unknownLunMirrorStaIdList)
        allErrMsg += common.getMsg(LANG, "lun.mirror.status.unknown", ",".join(unknownLunMirrorStaIdList))        
    
    if unknownSnapshotMirrorStaIdList:
        sortByInt(unknownSnapshotMirrorStaIdList)
        allErrMsg += common.getMsg(LANG, "snapshot.mirror.status.unknown", ",".join(unknownSnapshotMirrorStaIdList))
        
    if unknownDedupCompressLunStaIdList:
        sortByInt(unknownDedupCompressLunStaIdList)
        allErrMsg += common.getMsg(LANG, "dedupCompress.lun.mirror.status.unknown", ",".join(unknownDedupCompressLunStaIdList))
        
    if failedTransformLunList:
        sortByInt(failedTransformLunList)
        allErrMsg += common.getMsg(LANG, "faied.to.transform.lun.to.volume", ",".join(failedTransformLunList))
    
    if failedTransformSnapshotList:
        sortByInt(failedTransformSnapshotList)
        allErrMsg += common.getMsg(LANG, "faied.to.transform.snapshot.to.volume", ",".join(failedTransformSnapshotList))
    
    if failedTransformDedupCompressLunList:
        sortByInt(failedTransformDedupCompressLunList)
        allErrMsg += common.getMsg(LANG, "faied.to.transform.dedupCompressLun.to.volume", ",".join(failedTransformDedupCompressLunList))
    
    if notExsitLunList:
        sortByInt(notExsitLunList)
        allErrMsg += common.getMsg(LANG, "cant.find.lun.mirrorinfo", ",".join(notExsitLunList))
    
    if notExsitSnapshotList:
        sortByInt(notExsitSnapshotList)
        allErrMsg += common.getMsg(LANG, "cant.find.snapshot.mirrorinfo", ",".join(notExsitSnapshotList))
    
    if notExsitDedupCompressLunList:
        sortByInt(notExsitDedupCompressLunList)
        allErrMsg += common.getMsg(LANG, "cant.find.dedupCompressLun.mirrorinfo", ",".join(notExsitDedupCompressLunList))    
         
    if notMirrorLunList or notMirrorSnapshotList or notMirroeDedupCompressLunList:
        return False, allCliRet, allErrMsg
    elif unknownLunMirrorStaIdList or unknownSnapshotMirrorStaIdList or unknownDedupCompressLunStaIdList or failedTransformLunList or failedTransformSnapshotList or failedTransformDedupCompressLunList or notExsitLunList or notExsitSnapshotList or notExsitDedupCompressLunList:
        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
    else:
        return True, allCliRet, ''
    
def chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, chkWorkMode=False):
    """
    @summary: 在CLI连接所在节点上检查每个lLUN和快照的Cache写镜像状态是否开启（当LUN与文件系统个数之和小于300）以及工作模式。
    @param cli: CLI connection(SSH connection)
    @param lunIdList: LUN ID List
    @param snapshotIdList: Snapshot ID List
    @return: (isQrySucc, cliRet, errMsg, 
                (notMirrorLunIdList, notMirrorSnapshotIdList),
                (singleWorkmodeLunIdList, singlgeWorkmodeSnapshotIdList))
    """
    
    cmd = "cache show obj v switch all"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag != True: 
        LOGGER.logSysAbnormal()
        return (False, cliRet, errMsg, (None, None, None), (None, None, None), None)
    
    lunWriteMirrorDictList = cliUtil.getHorizontalCliRet(cliRet)
    if len(lunWriteMirrorDictList) == 0:
        errMsg = common.getMsg(LANG, "query.result.abnormal")
        LOGGER.logError("Failed to get information of LN cache mirror status.")
        return (False, cliRet, errMsg, (None, None, None), (None, None, None), None)
    
    #未开启镜像功能的LUN
    notMirrorLunList = []
    #未开启镜像功能的快照
    notMirrorSnapshotList = []
    #未开启镜像功能的开启过重删压缩开关的LUN
    notMirrorDedupCompressLunList = []
    
    singleCtrlModeOfLunList = []
    singleCtrlModeOfSnapshotList = []
    singleCtrlModeOfDedupCompressLunList = []
    
    #转换LUN或者快照失败的卷ID
    failedTransformLunOrSnapshotList = []
    
    for lunWirteMirrorDict in lunWriteMirrorDictList:
        volumeId = lunWirteMirrorDict.get("volumeid")
        if not volumeId:
            volumeId = lunWirteMirrorDict.get("volumeId")
        try:
            #ID大于65535则其为元数据，不检查
            if volumeId is None or ast.literal_eval(volumeId) > \
                    ast.literal_eval('0xffff'):
                continue
        except:
            LOGGER.logInfo("volumeId is not digit!")
            
        lunWirteMirrorState = lunWirteMirrorDict.get("writeMirror")
        workMode = lunWirteMirrorDict.get("workmode")
        if not lunWirteMirrorState:
            lunWirteMirrorState = lunWirteMirrorDict.get("wmi")
        
        if not workMode:
            workMode = lunWirteMirrorDict.get("wmo")
            
        if lunWirteMirrorState != "1":
            #将卷ID转换为LUN ID/快照ID
            ret = transformVolumeToLunOrSnapshot(cli, volumeId)
            if ret[0] == False:
                failedTransformLunOrSnapshotList.append(volumeId)
                cliRet = joinLines(cliRet, ret[1])
                continue
            
            lunOrSnapshot = hexadecimal2Decimal(ret[2])
                
            if lunOrSnapshot in lunIdList:
                notMirrorLunList.append(lunOrSnapshot)
                
            if lunOrSnapshot in snapshotIdList:
                notMirrorSnapshotList.append(lunOrSnapshot)
            
            if lunOrSnapshot in dedupCompressLunIdList:
                notMirrorDedupCompressLunList.append(lunOrSnapshot)
        
        if workMode != "1":
            #将卷ID转换为LUN ID/快照ID
            ret = transformVolumeToLunOrSnapshot(cli, volumeId)
            if ret[0] == False:
                failedTransformLunOrSnapshotList.append(volumeId)
                cliRet = joinLines(cliRet, ret[1])
                continue
            
            lunOrSnapshot = hexadecimal2Decimal(ret[2])
            
            if lunOrSnapshot in lunIdList:
                singleCtrlModeOfLunList.append(lunOrSnapshot)
                
            if lunOrSnapshot in snapshotIdList:
                singleCtrlModeOfSnapshotList.append(lunOrSnapshot)
            
            if lunOrSnapshot in dedupCompressLunIdList:
                singleCtrlModeOfDedupCompressLunList.append(lunOrSnapshot)
                
    
    if len(notMirrorLunList) != 0:
        sortByInt(notMirrorLunList)
        errMsg += common.getMsg(LANG, "lun.not.mirror", ",".join(notMirrorLunList))

    if len(notMirrorSnapshotList) != 0:
        sortByInt(notMirrorSnapshotList)
        errMsg += common.getMsg(LANG, "snapshot.not.mirror", ",".join(notMirrorSnapshotList))
    
    if len(notMirrorDedupCompressLunList) != 0:
        sortByInt(notMirrorDedupCompressLunList)
        errMsg += common.getMsg(LANG, "dedupCompress.lun.not.mirror", ",".join(notMirrorDedupCompressLunList))
    
    if chkWorkMode and not is_big_card:  
        if singleCtrlModeOfLunList:
            sortByInt(singleCtrlModeOfLunList)
            errMsg += common.getMsg(LANG, "lun.work.in.single.ctrl.mode", ",".join(singleCtrlModeOfLunList))
    
        if len(singleCtrlModeOfSnapshotList) != 0:
            sortByInt(singleCtrlModeOfSnapshotList)
            errMsg += common.getMsg(LANG, "snapshot.work.in.single.ctrl.mode", ",".join(singleCtrlModeOfSnapshotList))
            
        if len(singleCtrlModeOfDedupCompressLunList) != 0:
            sortByInt(singleCtrlModeOfDedupCompressLunList)
            errMsg += common.getMsg(LANG, "dedupCompress.work.in.single.ctrl.mode", ",".join(singleCtrlModeOfDedupCompressLunList))
    
    if len(failedTransformLunOrSnapshotList) != 0:
        errMsg += common.getMsg(LANG, "faied.to.transform.volume.to.LunOrSnapshot", ",".join(failedTransformLunOrSnapshotList))
    
    global CURRENTPROCESS, CHECK_NODE_NUMBERS
    CURRENTPROCESS += (80 / CHECK_NODE_NUMBERS)
    refreshProcess(PY_JAVA_ENV, CURRENTPROCESS)
        
    return (True, cliRet, errMsg, (notMirrorLunList, notMirrorSnapshotList, notMirrorDedupCompressLunList),
            (singleCtrlModeOfLunList, singleCtrlModeOfSnapshotList, singleCtrlModeOfDedupCompressLunList), failedTransformLunOrSnapshotList)

def chk2StatesOfAllObjsOnAllNodes(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList):  
    '''
    @summary: check every LUN and snapshot write mirror and work mode status on every controller.
    @param cli: CLI connection
    @param lunIdList: LUN ID List
    @param snapshot: Snapshot ID list
    @return: (checkResult, cliRet, errMsg)
    '''   
    allCliRet, errMsg = '', ''
    
    engNumDict = getEngineCtrlNumDict(ctrlIdList)
    engineNum = len(engNumDict)
    
    if engineNum == 1:
        return chk2StatesOfAllObjsOnAllNodesInEngine(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, '0')
    else:
        qryRet, cliRet, errMsg, mgtPortInfo = cliUtil.getManagementPortInfo(cli, LANG)
        if True != qryRet:
            LOGGER.logError('Query management port information failed.')
            return cliUtil.RESULT_NOCHECK, cliRet, errMsg 
        else:
            LOGGER.logInfo('Query management port information success:' + unicode(mgtPortInfo))
            
        engReachableIPDict = getAllEngineReachableIP(mgtPortInfo)
        LOGGER.logInfo('All engine reachable IP:' + unicode(engReachableIPDict))

        allCliRet = ""
        
        curConnectedEngId = getConnectedEngineId(mgtPortInfo)
        curConnectedEngNodeNum = engNumDict.get(curConnectedEngId, 0)
        LOGGER.logInfo("The engine of ToolKit connecting is %(engId)s, and its controller number is %(engNodeNum)s."
                       % {'engId':unicode(curConnectedEngId), 'engNodeNum':unicode(curConnectedEngNodeNum)})
        
        #先在工具连接的当前引擎内检查
        if curConnectedEngId and curConnectedEngNodeNum:
            #控制器数量大于2时心跳到每个控制器检查
            checkResult, cliRet, errMsg = chk2StatesOfAllObjsOnAllNodesInEngine(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, curConnectedEngId)
            allCliRet = joinLines(allCliRet, cliRet)
            if checkResult != True:
                LOGGER.logError('Current connected engine check not PASS, check finished.')
                return checkResult, allCliRet, errMsg
            else:
                LOGGER.logError('Current connected engine check PASS, check other engine further.')
                
        for engId in engReachableIPDict:
            if engId == curConnectedEngId:
                continue
            
            cilConnection = None
            ipList = engReachableIPDict.get(engId)
            for ip in ipList:
                cilConnection = common.getCilConnectionByIp(ip, PY_JAVA_ENV, LOGGER)
                LOGGER.logInfo("Begin connect engine %s, and its management IP is %s." % (engId, ip))
                if cilConnection:
                    break
            if not cilConnection:
                LOGGER.logInfo("Fail to connect engine %s, and its management IP is %s." % (engId, str(ipList)))
                errMsg = common.getMsg(LANG, "dev.connEngine.failure", (engId, engId))
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
            
            checkResult, cliRet, errMsg = chk2StatesOfAllObjsOnAllNodesInEngine(cilConnection, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, engId)
            #关闭连接（不会抛异常）
            PY_JAVA_ENV.get("sshManager").releaseConnection(cilConnection)
            allCliRet = joinLines(allCliRet, cliRet)
            if checkResult != True:
                LOGGER.logError('This engine(Engine number:' + unicode(engId) + '), check not PASS, check finished.')
                return (checkResult, allCliRet, errMsg)
        
        LOGGER.logInfo('All connected engine has been checked.')
        
        #已经检查过的引擎   
        chkedEngIdList = engReachableIPDict.keys()
        #集群中的所有引擎
        allEngIdList = engNumDict.keys()
        
        notConnectedEngIdList = [engId for engId in allEngIdList if engId not in chkedEngIdList]
        if notConnectedEngIdList:
            notConnectedEngIdList.sort()
            LOGGER.logError("There has not connected engine that can not be checked: " + ','.join(notConnectedEngIdList))
            errMsg += common.getMsg(LANG, "dev.connEngine.failure", (','.join(notConnectedEngIdList), ','.join(notConnectedEngIdList)))
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
           
        return (True, allCliRet, "") 
    
    
def chk2StatesOfAllObjsOnAllNodesByContrCli(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList):  
    '''
    @summary: check every LUN and snapshot write mirror and work mode status on every controller, the ssh connection is to controller.
    @param cli: CLI connection
    @param lunIdList: LUN ID List
    @param snapshot: Snapshot ID list
    @return: (checkResult, cliRet, errMsg)
    '''
    engNumDict = getEngineCtrlNumDict(ctrlIdList)
    engineNum = len(engNumDict)
    
    if engineNum == 1:
        ret = getConnectionByContrIp(PY_JAVA_ENV, cli)
        if ret[0] == False:
            return cliUtil.RESULT_NOCHECK, ret[1], ret[2] 
        contrCli = ret[3]
        try:
            return chk2StatesOfAllObjsOnAllNodesInEngine(contrCli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, '0')
        except:
            LOGGER.logError("check except in 18000 branch.")
        finally:
            LOGGER.logInfo("release connection :%s."% str(contrCli))
            PY_JAVA_ENV.get("sshManager").releaseConnection(contrCli)

    #多引擎环境需要连接到每个引擎上的控制器进行检查
    flag, cliRet, errMsg, ipListDict = getContrIpListOfEngine(cli, LANG)
    if flag != True:
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
    
    allCliRet = ""
              
    for engId in ipListDict:
        ipList = ipListDict.get(engId)
        cilConnection = None
        for ip in ipList:
            cilConnection = common.getCilConnectionByIp(ip, PY_JAVA_ENV, LOGGER)
            if cilConnection:
                break
        if not cilConnection:
            LOGGER.logInfo("Fail to connect engine %s, and its management IP is %s." % (engId, str(ipList)))
            errMsg = common.getMsg(LANG, "dev.connEngine.failure", (engId, engId))
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
            
        checkResult, cliRet, errMsg = chk2StatesOfAllObjsOnAllNodesInEngine(cilConnection, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, engId)
        #关闭连接（不会抛异常）
        PY_JAVA_ENV.get("sshManager").releaseConnection(cilConnection)
        allCliRet = joinLines(allCliRet, cliRet)
        if checkResult != True:
            LOGGER.logError('This engine(Engine number:' + unicode(engId) + '), check not PASS, check finished.')
            return (checkResult, allCliRet, errMsg)
        
    LOGGER.logInfo('All connected engine has been checked.')
        
    #已经检查过的引擎   
    chkedEngIdList = ipListDict.keys()
    #集群中的所有引擎
    allEngIdList = engNumDict.keys()
        
    notConnectedEngIdList = [engId for engId in allEngIdList if engId not in chkedEngIdList]
    if notConnectedEngIdList:
        notConnectedEngIdList.sort()
        LOGGER.logError("There has not connected engine that can not be checked: " + ','.join(notConnectedEngIdList))
        errMsg += common.getMsg(LANG, "dev.connEngine.failure", (','.join(notConnectedEngIdList), ','.join(notConnectedEngIdList)))
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
           
    return (True, allCliRet, "")
       

def chk2StatesOfAllObjsOnCtrlPair(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=True): 
    '''
    @summary: Check write mirror status and work mode on 2-controllers pair in one engine.
    @param cli:CLI connection
    @param lunIdList: LUN ID List.
    @param snapshotIdList: Snapshot ID List.
    @return: (checkResult, cliRet, errMsg)
    '''
    #先在工具当前连接的控制器上检查
    allCliRet = ''
    isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
    notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
    singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
    allCliRet = joinLines(allCliRet, cliRet)
        
    if not isChkSucc:
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
    elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList or singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList:
        return False, allCliRet, errMsg
    elif failedTransformLunOrSnapshotList:
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
    else:
        LOGGER.logInfo('All LUNs or snapshots cache write mirror on this controller are enabled and work on double'
        ' controller mode, check other controller now...')
    
    if not isPeerPresent:
        LOGGER.logInfo('Peer controller not present, check finished.') 
        return True, allCliRet, errMsg
        
    priKey = PY_JAVA_ENV.get("devInfo").getPriKey()
    if priKey:
        #不支持PublicKey鉴权方式进行心跳控制器
        errMsg = common.getMsg(LANG, "no.support.publickey.forensics")
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
    #心跳到对端控制器检查
    cmd = "sshtoremote"
    passWord = PY_JAVA_ENV.get("devInfo").getLoginUser().getPassword()
    flag, cliRet, errMsg = cliUtil.sshToRemoteContr(cli, cmd, passWord, LANG)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        LOGGER.logError('SSH to remote controller failed.')
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
    try:
        isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
        notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
        singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
        allCliRet = joinLines(allCliRet, cliRet)
        if not isChkSucc:
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList or singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList:
            return False, allCliRet, errMsg
        elif failedTransformLunOrSnapshotList:
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        else:
            LOGGER.logInfo('All LUNs or snapshots cache write mirror on this controller are enabled and work on double'
            ' controller mode, check other controller now...')
            return True, allCliRet, errMsg
    except Exception, e:
        LOGGER.logError('Check LUNs and snapshots cache write mirror state and work mode exception:' + unicode(e))
        LOGGER.logException(e)
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
    finally:
        cliUtil.exitHeartbeatCli(cli, LANG)
        

def chk2StatesOfAllObjsOnAllNodesInEngine(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, engId):
    '''
    @summary: Check write mirror status and work mode on every nodes in one engine by sshtoremoteExt nodeId.
    @param cli: CLI connection.
    @param lunIdList:LUN ID list.
    @param snapshotIdList: Snapshot ID List.
    @param ctrlNodeList: Controller ID List.
    @param engId: Engine number to be checked such as engine 0, engine 1 and so on.
    @return: (checkResult, cliRet, errMsg)
    '''
    engNumDict = getEngineCtrlNumDict(ctrlIdList)
    engNodeNum = engNumDict.get(engId)
    LOGGER.logInfo('Engine number map is:' + unicode(engNumDict))
    LOGGER.logInfo('This engine node number is:' + unicode(engNodeNum))
    
    if engNodeNum == 1:
        return chk2StatesOfAllObjsOnCtrlPair(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=False)    
    elif engNodeNum == 2:
        return chk2StatesOfAllObjsOnCtrlPair(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=True)
    else:
        pass
    
    #Get controller IDs that are present.
    isQrySucc, cliRet, errMsg, presentNodeIdList = cliUtil.getPresentCtrlNodeIdListForSshToRemote(cli, LANG, engId)
    if not isQrySucc:
        LOGGER.logError('Query controller present node id list failed.')
        return cliUtil.RESULT_NOCHECK, cliRet, errMsg
    else:
        LOGGER.logInfo('Query controller node id list success:' + unicode(presentNodeIdList))
    
    priKey = PY_JAVA_ENV.get("devInfo").getPriKey()
    if priKey:
        #不支持PublicKey鉴权方式进行心跳控制器
        errMsg = common.getMsg(LANG, "no.support.publickey.forensics")
        return (cliUtil.RESULT_NOCHECK, "", errMsg)
         
    allCliRet, errMsg = '', ''
    passWord = PY_JAVA_ENV.get("devInfo").getLoginUser().getPassword()
    
    allSingleCtrlModeLunIdList, allSingleSnapshotIdList, allSingleDedupCompressLunIdList = [], [], []
    for nodeId in presentNodeIdList:
        cmd = "sshtoremoteExt %s" % str(nodeId)
        flag, cliRet, errMsg = cliUtil.sshToRemoteContr(cli, cmd, passWord, LANG)
        allCliRet = joinLines(allCliRet, cliRet)
        if flag != True:
            LOGGER.logError('sshtoremoteExt to remote node(' + unicode(nodeId) + ') failed.')
            errMsg = common.getMsg(LANG, "heart.beat.to.node.failed", unicode(nodeId))
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        else:
            LOGGER.logInfo('sshtoremoteExt to remote node(' + unicode(nodeId) + ') success.')
            
        isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
        notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
        singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
        allSingleCtrlModeLunIdList.extend(singleWorkMode[0])
        allSingleSnapshotIdList.extend(singleWorkMode[1])
        allSingleDedupCompressLunIdList.extend(singleWorkMode[2])
        
        cliUtil.exitHeartbeatCli(cli, LANG)
        
        allCliRet = joinLines(allCliRet, cliRet)
        if not isChkSucc:
            LOGGER.logError('All LUNs or snapshots cache write mirror and work mode are not checked successfully on'
            ' node(node ID:' + unicode(nodeId) + '), return NO_CHECK now!')
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList:
            LOGGER.logError('Some LUNs or snapshots cache write mirror are not enabled on'
            ' node(node ID:' + unicode(nodeId) + '), return False now!')
            return False, allCliRet, errMsg
        elif (singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList) and (not is_big_card):
            LOGGER.logError('Some LUNs or snapshots work in single controller mode on'
            ' node(node ID:' + unicode(nodeId) + '), and this is small card device, return False now!')
            return False, allCliRet, errMsg
        elif failedTransformLunOrSnapshotList:
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        else:
            LOGGER.logInfo('All LUNs or snapshots cache write mirror are enabled'
            ' on current node(node ID:' + unicode(nodeId) + ').')
    
    #End loop of node ID list (sshtoremoteExt nodeID).
    notPasswdLunIdList, notPasswdSnapshotIdList, notPasswdDedupCompressLunIdList = [], [], []
    if is_big_card:
        notPasswdLunIdList = filter(lambda lunId: allSingleCtrlModeLunIdList.count(lunId) > 2, allSingleCtrlModeLunIdList)
        notPasswdSnapshotIdList = filter(lambda snapshotId: allSingleSnapshotIdList.count(snapshotId) > 2, allSingleSnapshotIdList)
        notPasswdDedupCompressLunIdList = filter(lambda dedupCompressLunId: allSingleDedupCompressLunIdList.count(dedupCompressLunId) > 2, allSingleDedupCompressLunIdList)
        
        notPasswdLunIdList = list(set(notPasswdLunIdList))
        notPasswdSnapshotIdList = list(set(notPasswdSnapshotIdList))
        notPasswdDedupCompressLunIdList = list(set(notPasswdDedupCompressLunIdList))

        if notPasswdLunIdList:
            sortByInt(notPasswdLunIdList)
            errMsg += common.getMsg(LANG, "lun.work.in.single.ctrl.mode", ",".join(notPasswdLunIdList))
    
        if notPasswdSnapshotIdList:
            sortByInt(notPasswdSnapshotIdList)
            errMsg += common.getMsg(LANG, "snapshot.work.in.single.ctrl.mode", ",".join(notPasswdSnapshotIdList))
        
        if notPasswdDedupCompressLunIdList:
            sortByInt(notPasswdDedupCompressLunIdList)
            errMsg += common.getMsg(LANG, "dedupCompress.work.in.single.ctrl.mode", ",".join(notPasswdDedupCompressLunIdList))
   
    if notPasswdLunIdList or notPasswdSnapshotIdList or notPasswdDedupCompressLunIdList:
        return False, allCliRet, errMsg
    else:    
        return True, allCliRet, errMsg

def joinLines(oldLines, newLines):
    if oldLines:
        return '\n'.join([oldLines, newLines]) if newLines else oldLines
    else:
        return newLines


def getConnectionByContrIp(PY_JAVA_ENV, cli):
    """
    @summary: 建立到阵列控制器的SSH连接，只有登录控制器的连接才能进入debug，minisystem模式
    """
    flag, cliRet, errMsg, ipListDict = getContrIpListOfEngine(cli, LANG)
    LOGGER.logInfo('controller management ip list in the engine is %s' % str(ipListDict))
    if flag != True:
        return (False, cliRet, errMsg, cli)
    
    currentContrIp = PY_JAVA_ENV.get("devInfo").getIp()
    ipListList = ipListDict.values()
    for ipList in ipListList:
        if currentContrIp in ipList:
            #当前连接就是与控制器的连接（18000设备跳过SVP，PC直接连接控制器）
            LOGGER.logInfo('the current ip(%s) is controller management ip.' % str(currentContrIp))
            return (True, cliRet, "", cli)
    
    for engine in ipListDict:
        ipList = ipListDict.get(engine)
        for ip in ipList:
            cilConnection = common.getCilConnectionByIp(ip, PY_JAVA_ENV, LOGGER)
            if cilConnection:
                LOGGER.logInfo('Success to connect to controller(IP:%s)' % ip)
                return (True, cliRet, "", cilConnection)
            else:
                LOGGER.logInfo('Failed to connect to controller(IP:%s)' % ip)
    #连接控制器失败。请参考预警公告手动检查。
    errMsg = common.getMsg(LANG, "connect.to.controller.failed")
    return (False, cliRet, errMsg, cli)
                
        
def getContrIpListOfEngine(cli, lang):
    """
    @summary: 获取引擎对应下的控制器IP
    """
    ipListDict = {}
    
    cmd = "show upgrade package"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (flag, cliRet, errMsg, ipListDict)
    
    beginIndex = cliRet.find("Software Version")
    endIndex = cliRet.find("HotPatch Version")
    infoDictList = cliUtil.getHorizontalCliRet(cliRet[beginIndex : endIndex])
    
    for infoDict in infoDictList:
        ip = infoDict.get("IP", "")
        contrName = infoDict.get("Name", "")
        if ip != "" and contrName != "":
            engine = contrName[:1]
            ipList = ipListDict.get(engine, [])
            ipList.append(ip)
            ipListDict.update({engine:ipList})
    
    #控制器未配置管理IP，无法连接控制器。请参考预警公告手动检查。
    if len(ipListDict) == 0:
        errMsg = common.getMsg(LANG, "contr.not.config.ip")
        return (False, cliRet, errMsg, ipListDict)
    
    return (True, cliRet, errMsg, ipListDict)


def transformVolumeToLunOrSnapshot(cli, volumeId):
    """
    @summary: 将卷ID转换为LUN ID/快照ID。当需要使用debug下查询到的卷ID查找cli下命令使用的LUN ID对象时，
                                     需要使用命令 volume showctrl [volumeID]通过卷ID查找LUN ID，再进行后续操作。
    """
    global VOLUMETOLUNDICT
    lunId = VOLUMETOLUNDICT.get(volumeId, None)
    if lunId != None:
        LOGGER.logInfo('The lunId is %s of volumeId(%s).' % (lunId, volumeId))
        return (True, "", lunId)
    
    cmd = "volume showctrl %s" % volumeId
    flag, cliRet, _ = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag != True: 
        LOGGER.logSysAbnormal()
        return (False, cliRet, "")

    for line in cliRet.splitlines():
        if ":" not in line:
            continue
        
        fileds = line.split(":")
        if len(fileds) <= 1:
            continue
        
        key = fileds[0]
        if key.strip().lower() == "lun id":
            lunId = fileds[1].strip()
            global VOLUMETOLUNDICT
            VOLUMETOLUNDICT[volumeId] = lunId
            return (True, cliRet, lunId)
        
    return (False, cliRet, "")
        

def transformLunToVolume(cli, lunId):
    """
    @summary: 将LUN ID转换为卷ID。当需要使用cli下命令查询到的LUN ID或快照ID查找debug下的卷对象时，
                                    需要使用命令devlun show [LUN ID]通过LUN ID查找卷ID，再进行后续操作。
    """
    cmd = "devlun showctrl %s" % lunId
    flag, cliRet, _ = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag != True: 
        LOGGER.logSysAbnormal()
        return (False, cliRet, "")

    for line in cliRet.splitlines():
        if ":" not in line:
            continue
        
        fileds = line.split(":")
        if len(fileds) <= 1:
            continue
        
        key = fileds[0]
        if key.strip().lower() == "data volume id":
            volumeId = fileds[1].strip()
            return (True, cliRet, volumeId)
        
    return (False, cliRet, "")


def transformSnapshotToVolume(cli, snapshotId):
    """
    @summary: 将快照ID转换为卷ID。转换方式同LUN
    """
    return transformLunToVolume(cli, snapshotId)


def transformDedupCompressLunToVolume(cli, dedupCompressLun):
    """
    @summary: 将重删压缩LUN ID转换为卷ID。转换方式同LUN
    """
    return transformLunToVolume(cli, dedupCompressLun)


def refreshProcess(PY_JAVA_ENV, percentNumber):
    """
    @summary: 设置巡检当前进度
    """
    observer = PY_JAVA_ENV.get("progressObserver")
    try:
        if observer != None:
            observer.updateProgress(int(percentNumber))
    except:
        LOGGER.logInfo(str(traceback.format_exc()))
        
        
def getCheckNodeNumbers(productVersion, checkObjNum, contrNum):
    """
    @summary: 获取需要检查的节点数量
    """
    if productVersion < 'V300R003C00' or productVersion >= 'V300R006C00SPC100':
        return 1
    if ('V300R003C00' <= productVersion < 'V300R006C00SPC100') and (checkObjNum > 400):
        return 1
    if ('V300R003C00' <= productVersion < 'V300R006C00SPC100') and (checkObjNum <= 400):
        return contrNum
    
    return 1


def isSuperAdmin(cli):
    """
    @summary: 判断用户是否为超级管理员
    """
    flag, cliRet, userLevel, errMsg = cliUtil.getUserPrivilegeWithCliRet(cli, LANG)
    if flag != True:
        LOGGER.logError('Query user privilege failed, check terminated!')
        return (False, cliRet, errMsg)
    product_version = str(
        PY_JAVA_ENV.get("devInfo").getProductVersion())
    hot_patch = str(PY_JAVA_ENV.get("devInfo").getHotPatchVersion())
    str(PY_JAVA_ENV.get("devInfo").getHotPatchVersion())
    if userLevel.lower() != "super_admin" and not \
            is_support_read_only_user_enter_debug(product_version, hot_patch):
        LOGGER.logError('Current user level is %s.' % userLevel)
        errMsg = common.getMsg(LANG, "user.level.not.super_admin")
        return (False, cliRet, errMsg)
    
    return (True, cliRet, errMsg)
        

def hexadecimal2Decimal(hexadecimalStr):
    """
    @summary: 十六进制字符串转十进制字符串
    """
    decimalStr = ""
    try:
        #diagnose下获取到的LUN ID是16进制，需要转换成用户更易理解的10进制
        decimalStr = str(int(ast.literal_eval(hexadecimalStr)))
    except Exception, exception:
        LOGGER.logInfo("Transform hexadecimalStr[%s] to decimalStr exception." % str(hexadecimalStr))
        LOGGER.logException(exception)
    return decimalStr


def sortByInt(sortList):
    """
    @summary: 对元素类型为int的列表排序
    """
    try:
        sortList.sort(cmp=None, key=int, reverse=False)
    except:
        LOGGER.logInfo("list sort by int failed, maybe the list item is not int type.")
        LOGGER.logInfo(str(traceback.format_exc()))