# -*- coding: UTF-8 -*-
import traceback
import re

import cliUtil
import common
from common import UnCheckException
from com.huawei.ism.tool.obase.exception import ToolException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
START_CHECK_VERSION_V3 = "V300R003C00"
START_CHECK_VERSION_V5 = "V500R007C00"
START_CHECK_VERSION_DORADO = "V300R001C00"


def execute(cli):
    remote_lun_single_link = RemoteLunSingleLink(
        cli, LANG, PY_JAVA_ENV, LOGGER
    )
    flag, msg = remote_lun_single_link.execute_check()
    return flag, "\n".join(remote_lun_single_link.all_ret_list), msg


class RemoteLunSingleLink:
    """
    远端LUN冗余链路检查
    """

    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.product_version = ""
        self.software_version = ""
        self.no_redundant_heterogeneous_list = []

    def execute_check(self):
        try:
            if not self.is_risk_version():
                return True, ""
            device_lun_map, sum_lun_wwn = self.get_remote_lun_status()
            if not device_lun_map:
                return True, ""
            self.find_no_redundant_heterogeneous_list(
                device_lun_map, sum_lun_wwn
            )
            if not self.no_redundant_heterogeneous_list:
                return True, ""
            error_message = []
            for lun in self.no_redundant_heterogeneous_list:
                error_message.append(
                    common.getMsg(
                        self.lang, "remote_lun_single_link.no.other.links", lun
                    )
                )
            return False, "".join(error_message)
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except ToolException:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, LANG)
            LOGGER.logInfo(
                "enter cli mode from some model ret is %s" % str(ret)
            )
            if not ret[0]:
                common.reConnectionCli(self.cli, LOGGER)

    def find_no_redundant_heterogeneous_list(
        self, device_lun_map, sum_lun_wwn
    ):
        """
        轮训device id 每次取出一个不一样的lun_wwn直到遍历完所有lun_wwn或者总计超过
        400条数据
        :param device_lun_map:
        :param sum_lun_wwn
        :return:
        """
        sum_count = 400
        index = 0
        while sum_count > 0 and sum_lun_wwn > 0:
            sum_count, sum_lun_wwn = self.judge_lun_wwn(
                device_lun_map, index, sum_count, sum_lun_wwn
            )
            index += 1

    def judge_lun_wwn(self, device_lun_map, index, sum_count, sum_lun_wwn):
        """
        从每一个device id对应的列表取出一个lun wwn（通过index），判断其是否为冗余
        :param device_lun_map:
        :param index:
        :param sum_count:
        :param sum_lun_wwn:
        :return:
        """
        for device_id, lun_list in device_lun_map.items():
            if index < len(lun_list):
                lun = lun_list[index]
                path_info = self.get_remote_lun_path_status(lun)
                lun_attribute_dict = self.make_controller_info_dict(path_info)
                self.judge_no_redundant_heterogeneous_list(
                    lun_attribute_dict, lun
                )
                sum_count -= 1
                sum_lun_wwn -= 1
        return sum_count, sum_lun_wwn

    def judge_no_redundant_heterogeneous_list(
        self, lun_attribute_dict, lun_wwn
    ):
        """
        将非冗余的LUN WNN添加到列表
        :param lun_attribute_dict:
        :param lun_wwn
        """
        is_pass_way = False
        temp_local_controller = lun_attribute_dict.get("Local Controller")
        if temp_local_controller:
            is_pass_way = self.judge_local_controller(temp_local_controller)
        temp_local_port_id = lun_attribute_dict.get("Local Port ID")
        if temp_local_port_id and not is_pass_way:
            is_pass_way = self.judge_local_port_id(temp_local_port_id)
        if not is_pass_way:
            self.no_redundant_heterogeneous_list.append(lun_wwn)

    @staticmethod
    def judge_local_controller(local_controller_info):
        for (eg_id, controller_info) in local_controller_info.items():
            if ("A" in controller_info or "C" in controller_info) and (
                "B" in controller_info or "D" in controller_info
            ):
                return True

    @staticmethod
    def judge_local_port_id(local_port_id_info):
        for eg_id, controller_info in local_port_id_info.items():
            if len(controller_info) == 2:
                return True

    @staticmethod
    def make_controller_info_dict(path_info):
        """
        创建{"Local Controller": {"引擎ID":{控制器名,...},....},
        "Local Port ID": {"引擎ID"：{控制器名，....},...}}
        :param lun_wwn:
        :param path_info:
        :return: dict
        """
        local_controller_pattern = re.compile("^(\d)(\w)")
        local_port_id_pattern = re.compile("^CTE(\d).\w+.(\w)\w+")
        controller_dict = {"Local Controller": {}, "Local Port ID": {}}
        for i in path_info:
            temp_controller_info = i.get("Local Controller")
            local_controller_object = local_controller_pattern.search(
                temp_controller_info
            )
            if local_controller_object and not controller_dict[
                "Local Controller"
            ].get(local_controller_object.group(1)):
                controller_dict["Local Controller"][
                    local_controller_object.group(1)
                ] = {local_controller_object.group(2)}
                continue
            if local_controller_object and controller_dict[
                "Local Controller"
            ].get(local_controller_object.group(1)):
                controller_dict["Local Controller"][
                    local_controller_object.group(1)
                ].update({local_controller_object.group(2)})
                continue
            temp_port_id_info = i.get("Local Port ID")
            local_port_id_object = local_port_id_pattern.search(
                temp_port_id_info
            )
            if local_port_id_object and not controller_dict[
                "Local Port ID"
            ].get(local_port_id_object.group(1)):
                controller_dict["Local Port ID"][
                    local_port_id_object.group(1)
                ] = {local_port_id_object.group(2)}
                continue
            if local_port_id_object and controller_dict["Local Port ID"].get(
                local_port_id_object.group(1)
            ):
                controller_dict["Local Port ID"][
                    local_port_id_object.group(1)
                ].update({local_port_id_object.group(2)})
        return controller_dict

    def get_remote_lun_path_status(self, lun_wwn):
        """
        执行show remote_lun path remote_lun_wwn={}获取
        Path Priority为Active的LUN WWN对象
        :return:
        """
        cmd = "show remote_lun path remote_lun_wwn={}".format(lun_wwn)
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        temp_info = cliUtil.getHorizontalNostandardCliRet(ret)
        path_info = list(
            filter(lambda x: x.get("Path Priority") == "Active", temp_info)
        )
        return path_info

    def get_remote_lun_status(self):
        """
        执行show remote_lun status 获取Health Status为Normal的LUN WWN和Device ID
        处理为{"Device ID":[LUN WWN,....],.......}
        :return: map,sum_lun_wwn
        """
        device_lun_map = {}
        cmd = "show remote_lun status"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        temp_info = cliUtil.getHorizontalNostandardCliRet(ret)
        target_info = list(
            filter(lambda x: x.get("Health Status") == "Normal", temp_info)
        )
        if not target_info:
            return device_lun_map, len(target_info)
        for i in target_info:
            if i.get("Device ID") and i.get("Device ID") not in device_lun_map:
                device_lun_map[i.get("Device ID")] = [i.get("LUN WWN")]
                continue
            if i.get("Device ID") and i.get("Device ID") in device_lun_map:
                device_lun_map[i.get("Device ID")].append(i.get("LUN WWN"))
                continue
        return device_lun_map, len(target_info)

    def is_risk_version(self):
        (
            flag,
            self.product_model,
            ret,
            msg,
        ) = cliUtil.getProductModelWithCliRet(self.cli, self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)
        (
            flag,
            cli_ret,
            err_msg,
            self.software_version,
            _,
        ) = common.getVersion(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if "Dorado" not in self.product_model:
            if any(
                [
                    self.software_version.startswith("V3")
                    and self.software_version[0:len(START_CHECK_VERSION_V3)]
                    >= START_CHECK_VERSION_V3,
                    self.software_version.startswith("V5")
                    and self.software_version[0:len(START_CHECK_VERSION_V5)]
                    >= START_CHECK_VERSION_V5,
                ]
            ):
                return True
            return False
        else:
            return True
