# -*- coding: UTF-8 -*-
import common
import common_cache
import cliUtil
import re
from common_utils import get_err_msg
from common_utils import get_patch_value
from common import UnCheckException
from cbb.frame.cli import cli_with_cache
from risk_version_config import RISK_SHARD_PORT_SWITCH_VERSION
from risk_version_config import RISK_SHARD_PORT_SWITCH_MODEL

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    共享卡双活ALUA特殊模式0风险检查
    步骤1 以admin用户登录设备；
    步骤2 执行命令：show system general，查询设备型号信息；
    步骤3 执行命令：show upgrade package，查询系统软件版本信息；
    步骤4 执行命令：show hyper_metro_pair general |filterRow column=Type predict
    =equal_to value=LUN |filterColumn include columnList=ID,Local\sID，查询系统
    中双活LUN的ID信息（Local ID）；
    步骤5 执行命令：show initiator initiator_type=FC isfree=no，查询并记录Special
    Mode Type为mode0的启动器对应的主机Host ID；
    步骤6 执行命令：show host lun host_id=[Host ID]，Host ID为步骤5中记录的主机
    Host ID，查询并记录主机映射的LUN ID；

    :param cli:
    :return:shard_port_switch
    """
    shard_port_switch_check = ShardPortSwitchCheck(
        cli, LANG, PY_JAVA_ENV, LOGGER
    )
    flag, msg = shard_port_switch_check.execute_check()
    return flag, "\n".join(shard_port_switch_check.all_cli_ret), msg


class ShardPortSwitchCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.dev_version = ""
        self.dev_model = ""
        self.logger = logger
        self.all_cli_ret = []

    def execute_check(self):

        try:

            # 检查版本
            if self.check_version():
                return True, ""

            # 检查type类型为LUN的双活pair记录
            flag, hyper_metro_lun_id_list = self.check_hyper_metro_pair()
            if flag:
                return True, ""

            # 检查"Special Mode Type"字段是否为为mode0
            flag, host_id_wwn_list = self.check_special_mode_type()
            if flag:
                return True, ""

            # 检查主机映射的任意LUN属于双活LUN
            return self.check_host_id(
                hyper_metro_lun_id_list,
                host_id_wwn_list,
            )

        except UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, err_msg)

    def check_version(self):
        """
        检查软件版本
        :return:True: 检查通过
        """

        (
            flag,
            cli_ret,
            err_msg,
            self.dev_model,
        ) = common_cache.get_product_model_with_cliret_cache(
            self.env, self.cli, self.logger
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)

        (
            flag,
            self.dev_version,
            p_patch,
            cli_ret,
            err_msg,
        ) = common_cache.get_version_and_patch_cache(
            self.env, self.cli, self.logger
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)

        self.logger.logInfo(
            "dev_version is: {}; dev_model is: {}".format(
                self.dev_version, self.dev_model
            )
        )
        if self.dev_model not in RISK_SHARD_PORT_SWITCH_MODEL:
            return True
        if self.dev_version not in RISK_SHARD_PORT_SWITCH_VERSION or\
                get_patch_value(p_patch) >= \
                RISK_SHARD_PORT_SWITCH_VERSION.get(self.dev_version):
            return True
        return False

    def check_hyper_metro_pair(self):
        """
        查询系统中双活LUN的ID信息
        :return:True: 检查通过
        """
        cmd = (
            r"show hyper_metro_pair general |filterRow column=Type "
            r"predict=equal_to value=LUN |filterColumn include columnList="
            r"ID,Local\sID"
        )

        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True, []
        hyper_metro_lun_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        hyper_metro_lun_id_list = list(
            map(lambda x: x.get("Local ID"), hyper_metro_lun_info_list)
        )

        return False, hyper_metro_lun_id_list

    def check_special_mode_type(self):
        """
        查询"Special Mode Type"字段为mode0的host id 和wwn
        :return:True: 检查通过
        """
        cmd = "show initiator initiator_type=FC isfree=no"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True, ""
        ini_list = cliUtil.getHorizontalCliRet(cli_ret)
        mode0_flag = True
        host_id_wwn_list = []
        for every_ini in ini_list:
            mode_type = every_ini.get("Special Mode Type", "")
            result = re.search("mode0", mode_type, re.IGNORECASE)
            if result:
                mode0_flag = False
                host_id_wwn = (every_ini.get("Host ID"), every_ini.get("WWN"))
                host_id_wwn_list.append(host_id_wwn)

        return mode0_flag, host_id_wwn_list

    def check_host_id(self, hyper_metro_lun_id_list, host_id_wwn_list):
        """
        查询主机映射的任意LUN是否属于查询系统中双活LUN
        :param hyper_metro_lun_id_list 双活lun列表
        :param host_id_wwn_list 主机ID和WWN列表
        :return:True: 检查通过
        """
        host_id_wwn_list = list(set(host_id_wwn_list))
        host_id_wwn_list.sort()
        has_lun_id = True
        failed_host_id_wwn_list = []

        for host_id_wwn in host_id_wwn_list:
            cmd = "show host lun host_id=%s" % host_id_wwn[0]
            (
                flag,
                cli_ret,
                err_msg,
            ) = cli_with_cache.execute_cmd_in_cli_mode_with_cache(
                self.env, self.cli, cmd, self.logger
            )
            self.all_cli_ret.append(cli_ret)
            if flag is not True:
                raise UnCheckException(err_msg, cli_ret)
            if cliUtil.queryResultWithNoRecord(cli_ret):
                continue

            lun_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
            for every_lun in lun_dict_list:
                lun_id = every_lun.get("LUN ID", "")
                if lun_id in hyper_metro_lun_id_list:
                    has_lun_id = False
                    err_msg = "software.tgt.check.initiator.wwn"
                    failed_msg = get_err_msg(
                        self.lang, err_msg, (host_id_wwn[0], host_id_wwn[1])
                    )
                    failed_host_id_wwn_list.append(failed_msg)
                    break
        if not has_lun_id:
            err_msg = "software.tgt.check.shard.port.switch.check.not.pass"
            failed_msg = "\n".join(failed_host_id_wwn_list)
            return False, get_err_msg(
                self.lang,
                err_msg,
                (self.dev_version, self.dev_model, failed_msg),
            )
        return True, ""
