# coding:utf-8
"""
@version: Toolkit V200R006C00
@time: 2019/11/21
@file: cli_util_cache.py
@function: 新增模块
@modify: 公共API新增
"""


import com.huawei.ism.tool.obase.db.SqliteManager as SqliteManager

from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache
from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode
from cbb.frame.cli import running_data_analyze_util as ana_util
from cbb.frame.context import sqlite_context
from cbb.frame.base import product

import cliUtil
import common
from common_cache import is_use_running_data
from common_cache import ana_running_data
from common import UnCheckException
from common import getMsg


QUERY_SQL = "select * from '{}'"
TABLE_NAME_KEY_FILE_SYSTEM = "Number Of Fs"
TABLE_NAME_KEY_FILE_SYSTEM_V6 = "Number of fs"
TABLE_NAME_KEY_STORAGE_POOL = "Number Of Storage Pool"
TABLE_NAME_KEY_STORAGE_POOL_V6 = "Number of Storagepool"
COMMON_DISK_FILTER_CMD = "show disk general |filterColumn include " \
                         "columnList=ID,Manufacturer,Model," \
                         "Firmware\\sVersion,Serial\\sNumber,Bar\\sCode"

COMMON_DISK_FILTER_CMD_ERR = "show disk general |filterColumn include " \
                             "colunmList=ID,Manufacturer,Model," \
                             "Firmware\\sVersion,Serial\\sNumber," \
                             "Bar\\sCode"


RUNNING_DATA = "running data {0}"


def get_lun_general_with_filter(cli, py_java_env, logger, filters,
                                includes):
    """
    执行show lun general | filterRow column=ID predict=match
    value=xxx |filterColumn include columnList=xx,xxx
    类型的命令，要求filterRow条件必须为匹配某几个值
    :param cli:cli
    :param lang:lang
    :param py_java_env:env
    :param logger:logger
    :param filters:filterRowxxxx,格式为{ID:[value1,value2]}
    :param includes:filterColumn include xxx,格式为[ID1,ID2]
    :return:flag, cli_ret(show lun general), error_msg,
            lun_info_list:show lun general中所有lun转换后数据，若filter
            和include的列不再此命令中，将执行show lun general lun_id=?，补充
            查询详情，并将需要过滤的列补足在对应元素字典中，若filter不为空，
            将按filter中每一个value集合进行过滤
    """
    lun_info_list = []
    cli_ret_list = []
    cmd = "show lun general"
    flag, cli_ret, err_msg = \
        execute_cmd_in_cli_mode_with_cache(py_java_env, cli, cmd, logger)
    cli_ret_list.append(cli_ret)
    if flag is not True:
        return flag, cli_ret, err_msg, lun_info_list
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, cli_ret, '', lun_info_list
    lun_info_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    for lun_info in lun_info_dict_list:
        added_keys = []
        not_in_filters = is_all_filter_in_lun_info(lun_info, filters)
        not_in_includes = is_all_include_in_lun_info(lun_info, includes)
        if bool(not_in_filters) or bool(not_in_includes):
            added_keys.extend(not_in_filters)
            added_keys.extend(not_in_includes)
            added_keys = list(set(added_keys))
            lun_info, lun_info_cli_ret = \
                build_lun_detailed_info(lun_info, added_keys,
                                        py_java_env, cli, logger)
            cli_ret_list.append(lun_info_cli_ret)
        if filter_value_exits(lun_info, filters):
            lun_info_list.append(lun_info)
    return True, "\n".join(cli_ret_list), '', lun_info_list


def is_all_filter_in_lun_info(lun_info, filters):
    """
    是否每一个filter的key都存在于show lun general直接查询表头中
    :param lun_info:show lun general 转换的每一个lun的字典
    :param filters:filterRows
    :return:
    """
    not_in_filters = []
    if not bool(lun_info):
        return not_in_filters
    if not bool(filters):
        return not_in_filters
    filter_keys = filters.keys()
    for lun_filter in filter_keys:
        if lun_filter not in lun_info.keys():
            not_in_filters.append(lun_filter)
    return not_in_filters


def is_all_include_in_lun_info(lun_info, includes):
    """
    是否每一个include的key都存在于show lun general直接查询表头中
    :param lun_info: show lun general 转换的每一个lun的字典
    :param includes: filterColumn include
    :return:
    """
    not_in_includes = []
    if not bool(lun_info):
        return not_in_includes
    if not bool(includes):
        return not_in_includes
    for lun_include in includes:
        if lun_include not in lun_info.keys():
            not_in_includes.append(lun_include)
    return not_in_includes


def build_lun_detailed_info(lun_info, added_keys, py_java_env,
                            cli, logger):
    """
    通过show lun general lun_id=?查询lun具体信息，找到上一步查出的不在show
    lun general表头中的filters和includes，将key和值放入lun_info中
    :param lun_info:每一个lun的字典
    :param added_keys:不在show lun general中的标题
    :param py_java_env:env
    :param cli:cli
    :param logger:logger
    :param lang:lang
    :return:补充后的lun_info
    """
    lun_id = lun_info.get('ID')
    if not bool(lun_id):
        return {}
    cmd = "show lun general lun_id=%s" % lun_id
    flag, cli_ret, err_msg = \
        execute_cmd_in_cli_mode_with_cache(py_java_env, cli, cmd,
                                           logger)
    if flag is not True:
        return {}
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return {}
    cli_ret_list = cliUtil.getVerticalCliRet(cli_ret)
    for cli_ret_dict in cli_ret_list:
        for lun_filter in added_keys:
            if lun_filter in cli_ret_dict.keys():
                value = cli_ret_dict.get(lun_filter, '')
                if bool(value):
                    lun_info[lun_filter] = value
    return lun_info, cli_ret


def filter_value_exits(lun_info, filters):
    """
    lun是否满足filter的要求
    :param lun_info:lun的字典
    :param filters:过滤条件
    :return:T,F
    """
    if not bool(filters):
        return True
    if not bool(lun_info):
        return False
    for key in filters.keys():
        if not bool(filters[key]):
            return True
        if lun_info[key] not in filters[key]:
            return False
    return True


def get_disk_general_cache(env, cli, logger):
    """
    获取硬盘信息，缓存show disk general命令
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :return:硬盘信息,如果无盘，则返回空列表，回文
    """
    lang = common.getLang(env)
    cmd = "show disk general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    if not ret_dict_list:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(getMsg(lang, "cannot.get.disk.info"),
                               cli_ret)
    return ret_dict_list, cli_ret


def get_remote_device_link_cache(cli, env, logger):
    cmd = 'show remote_device link'
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)
    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    if not ret_dict_list:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret)
    return ret_dict_list, cli_ret


def get_lun_general_cache(env, cli, logger):
    """
    获取硬盘信息，缓存show lun general命令
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :return:硬盘信息,如果无盘，则返回空列表，回文
    """
    lang = common.getLang(env)
    cmd = "show lun general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    if not ret_dict_list:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(getMsg(lang, "cannot.get.lun.info"),
                               cli_ret)
    return ret_dict_list, cli_ret


def get_lun_capacity_info(env, cli, logger):
    """
    获取LUN容量信息
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :return:硬盘信息,如果无盘，则返回空列表，回文
    """
    cmd = "show lun general |filterColumn include columnList=ID,Capacity"
    lang = common.getLang(env)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    if not ret_dict_list:
        logger.logError("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(getMsg(lang, "cannot.get.lun.info"),
                               cli_ret)
    return ret_dict_list, cli_ret


def get_lun_detail_cache(env, cli, logger, lun_id):
    """
    获取lun信息，缓存show lun general lun_id=?命令
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :param lun_id: lun id
    :return:LUN信息
    """
    lang = common.getLang(env)
    cmd = "show lun general lun_id=%s" % lun_id
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getVerticalCliRet(cli_ret)
    if not ret_dict_list:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(getMsg(lang, "cannot.get.lun.info"),
                               cli_ret)
    return ret_dict_list, cli_ret


def execute_lun_detail_cache(env, cli, logger, lun_id_list):
    """
    执行lun 命令，缓存show lun general lun_id=?命令
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :param lun_id_list:lun id列表
    :return:返回执行命令结果
    """
    for lun_id in lun_id_list:
        cmd = "show lun general lun_id=%s" % lun_id
        execute_cmd_in_cli_mode_with_cache(env, cli, cmd, logger)


def get_disk_with_filter_cache(env, cli, logger, cmd=COMMON_DISK_FILTER_CMD):
    """
    获取硬盘信息，缓存"show disk general |filterColumn include " \
                         "columnList=ID,Manufacturer,Model," \
                         "Firmware\\sVersion,Serial\\sNumber,Bar\\sCode"命令
    维护版本注意：columnList已做版本兼容处理，单词拼写错误的处理。
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :param cmd:命令
    :return:硬盘信息,如果无盘，则返回空列表，回文
    """
    lang = common.getLang(env)
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    # 兼容产品命令单词拼写错误的场景
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        cmd = COMMON_DISK_FILTER_CMD_ERR
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
            env, cli, cmd, logger)

    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    ret_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    if not ret_dict_list:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(getMsg(lang, "cannot.get.disk.info"),
                               cli_ret)
    return ret_dict_list, cli_ret


def get_remote_rep_cache(env, cli, logger):
    """
    获取远程复制信息，缓存show remote_replication general命令
    :param env:上下文
    :param cli:连接
    :param logger:日志
    :return:远程复制信息,如果无远程复制信息，则返回空列表和回显
    """
    cmd = "show remote_replication general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        logger.logException("analysis cli ret error! %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return [], cli_ret

    return cliUtil.getHorizontalCliRet(cli_ret), cli_ret


def get_take_over_lun_cache(context, cli, lang, sn, logger):
    """
    获取伪装LUN信息
    :param context: 上下文
    :param cli: 连接
    :param lang: 上下文
    :param sn: 序列号
    :param logger: 日志
    :return: 获取异构LUN信息
    """
    info_dict_list = []
    cmd = "show lun_takeover general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode(
        context, cli, cmd, lang, sn, logger
    )
    if flag is not True:
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, info_dict_list

    info_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    return cli_ret, info_dict_list


def get_host_cache(env, cli, logger):
    """
    获取所有主机信息
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :return: 回文、主机列表
    """
    info_dict_list = []
    cmd = "show host general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        logger.logException("execute cmd error: %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, info_dict_list

    info_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    return cli_ret, info_dict_list


def get_fc_iscsi_initiator_cache(env, cli, logger):
    """
    获取FC和ISCSI启动器信息
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :return: cli_ret, fc_dict_list, iscsi_dict_list
    """
    fc_dict_list = []
    iscsi_dict_list = []
    cmd = "show initiator"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if flag is not True:
        logger.logException("execute cmd error: %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, fc_dict_list, iscsi_dict_list

    fc_cli_ret = cli_ret[:cli_ret.find("iSCSI IQN")]
    iscsi_cli_ret = cli_ret[cli_ret.find("iSCSI IQN") - 2:]

    fc_dict_list = cliUtil.getHorizontalCliRet(fc_cli_ret)
    iscsi_dict_list = cliUtil.getHorizontalCliRet(iscsi_cli_ret)

    return cli_ret, fc_dict_list, iscsi_dict_list


def get_ib_initiator_cache(env, cli, logger, host_id):
    """
    获取IB启动器
    当不存在或不支持IB时，返回空列表。
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :param logger: host_id
    :return: 回文、启动器信息
    """
    info_dict_list = []
    cmd = "show ib_initiator general host_id=%s" % host_id
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return cli_ret, info_dict_list

    if flag is not True:
        logger.logException("execute cmd error: %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, info_dict_list

    info_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    for info_dict in info_dict_list:
        info_dict["Host ID"] = host_id

    return cli_ret, info_dict_list


def get_ib_initiator_for_all_host_cache(env, cli, logger, host_id_list):
    """
    以主机为单位获取全部IB启动器信息
    :param env:
    :param cli:
    :param logger:
    :param host_id_list:
    :return:回文，所有IB信息列表
    """
    cli_ret_list = []
    all_ib_list = []
    for host_id in host_id_list:
        cli_ret, ib_list = get_ib_initiator_cache(env, cli, logger, host_id)
        cli_ret_list.append(cli_ret)
        if not ib_list:
            continue

    return cli_ret_list, all_ib_list


def get_host_lun_cache(env, cli, logger, host_id):
    """
    获取主机下的lun，带数据库缓存
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :param host_id:
    :return:
    """
    info_dict_list = []
    cmd = "show host lun host_id=%s" % host_id
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger)

    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return cli_ret, info_dict_list

    if flag is not True:
        logger.logError("execute cmd error: %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, info_dict_list

    info_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    return cli_ret, info_dict_list


def get_host_info_dorado_v6(context, cli, lang, sn, logger):
    info_dict_list = []
    cmd = "show host general |filterColumn include" \
          " columnList=ID,Access\\sMode,HyperMetro\\sPath\\sOptimized"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode(context,
                                                     cli, cmd,
                                                     lang, sn,
                                                     logger)
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return cli_ret, info_dict_list

    if flag is not True:
        logger.logException("execute cmd error: %s" % cli_ret)
        raise UnCheckException(err_msg, cli_ret, flag)

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return cli_ret, info_dict_list

    info_dict_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    return cli_ret, info_dict_list


def is_upgrading(cli, lang):
    """
    判断后台是否是升级状态
    :param cli:
    :param lang
    :return:
    """
    cli_ret = ''
    # 进入developer
    flag, ret, err_msg = cliUtil.enterDeveloperMode(cli, lang)
    cli_ret = common.joinLines(cli_ret, ret)
    if flag is not True:
        return False, cli_ret

    cmd = "show upgrade backgroundtask"
    flag, ret, err_msg = cliUtil.excuteCmdInDeveloper(cli, cmd, True, lang)
    cli_ret = common.joinLines(cli_ret, ret)
    # 不处理命令失败的场景
    ret_line_list = ret.splitlines()

    # 只有明确判断是升级状态时才报错。
    for line in ret_line_list:
        if "Upgrading" in line:
            return True, cli_ret

    return False, cli_ret


def get_lun_info_from_config(mem_obj, logger, sn):
    """
    获取LUN信息
    :param mem_obj: 内存对象
    :param logger: 日志
    :param sn: 阵列SN
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    # 融合存储和dorado v6和CLI字段不存在差异。如果存在差异需要处理后返回
    return ana_util.get_lun_info_from_running_data(sqlite_conn)


def get_kubernetes_data_info_from_config(mem_obj, logger, sn):
    """
    获取Kubernetes Running Data对象信息
    :param mem_obj: 内存对象
    :param logger: 日志
    :param sn: 阵列SN
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    # 融合存储和dorado v6和CLI字段不存在差异。如果存在差异需要处理后返回
    return ana_util.get_kubernetes_data_info_from_running_data(sqlite_conn)


def get_object_info_from_config(mem_obj, logger, sn, table_key_list):
    """
    从config中提取数据公共方法
    :param mem_obj: java map
    :param logger: 日志
    :param sn: 阵列SN
    :param table_key_list: 对象名称列表，可以放多个表名用于区分版本差异
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    res_list = []
    for table_key in table_key_list:
        res = ana_util.deal_dumps_value(
            SqliteManager().query(
                sqlite_conn, QUERY_SQL.format(table_key)
            )
        )
        if not res:
            continue

        res_list = res

    return res_list


def get_file_system_info_from_config(mem_obj, logger, sn):
    """
    封装的从config中获取文件系统
    :param mem_obj:
    :param logger:
    :param sn:
    :return:
    """
    table_name_list = (
        TABLE_NAME_KEY_FILE_SYSTEM,
        TABLE_NAME_KEY_FILE_SYSTEM_V6
    )
    return get_object_info_from_config(mem_obj, logger, sn, table_name_list)


def get_storage_pool_info_from_config(mem_obj, logger, sn):
    """
    封装的从config中获取存储池信息
    :param mem_obj:
    :param logger:
    :param sn:
    :return:
    """
    table_name_list = (
        TABLE_NAME_KEY_STORAGE_POOL,
        TABLE_NAME_KEY_STORAGE_POOL_V6
    )
    return get_object_info_from_config(mem_obj, logger, sn, table_name_list)


def get_host_info_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    # 融合存储和dorado v6和CLI字段不存在差异。如果存在差异需要处理后返回
    result_dict = {}
    # 由于Dorado v6和融合存储和对应CLI回文存在字段差异，需要统一处理后返回
    host_info_dict = ana_util.get_host_info_from_running_data(sqlite_conn)
    if product.is_dorado_series_version(p_version):
        for host_info in host_info_dict:
            host_info["IP Address"] = host_info.get("IP", '')
            host_info["ID"] = host_info.get("Id", '')
            host_info["Access Mode"] = host_info.get("ALUA MODE", '')
            host_info["name"] = host_info.get("Name", '')
            host_info["operSys"] = host_info.get("Operating System", '')
            host_info["ipAddr"] = host_info.get("IP", '')
            result_dict[host_info.get("Id", '')] = host_info
    else:
        for host_info in host_info_dict:
            host_info["ID"] = host_info.get("Id", '')
            host_info["IP Address"] = host_info.get("Ip", '')
            host_info["Operating System"] = host_info.get("Os Type", '')
            host_info["name"] = host_info.get("Name", '')
            host_info["operSys"] = host_info.get("Os Type", '')
            host_info["ipAddr"] = host_info.get("Ip", '')
            result_dict[host_info.get("Id", '')] = host_info

    return result_dict


def get_host_lun_info_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机LUN信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    host_lun_dict = {}
    # 由于Dorado v6和融合存储和对应CLI回文存在字段差异，需要统一处理后返回
    if product.is_dorado_series_version(p_version):
        host_lun_info_list = \
            ana_util.get_host_lun_info_from_running_data(
                sqlite_conn)
        for host_lun_info in host_lun_info_list:
            lun_id_list = host_lun_info.get("LUN ID List", '').split(
                ",")
            host_id = host_lun_info.get("Host ID", '')
            host_lun_dict[host_id] = lun_id_list
    else:
        # 融合存储
        mapping_view_info_dict = \
            ana_util.get_mapping_view_info_from_running_data(
                sqlite_conn)
        lun_group_info_dict = \
            ana_util.get_lun_group_info_from_running_data(
                sqlite_conn)
        host_group_host_info_dict = \
            ana_util.get_host_group_info_from_running_data(
                sqlite_conn)
        logger.logInfo("mapping_view_info_dict:%s\n\n, lun_group_info_dict:%s\n\n, host_group_host_info_dict:%s," % (mapping_view_info_dict, lun_group_info_dict, host_group_host_info_dict))
        for mapping_view in mapping_view_info_dict:
            host_group_id = mapping_view.get("Host Group Id")
            lun_group_id = mapping_view.get("Lun Group Id")
            # 如果任一个group为--，表示此view要么没有映射LUN，要么没有主机
            if host_group_id == "--" or lun_group_id == "--":
                continue

            lun_id_list = []
            for lun_group in lun_group_info_dict:
                if lun_group_id != lun_group.get("Id"):
                    continue

                lun_info_list = lun_group.get("LUN INFO", [])
                for lun_info in lun_info_list:
                    lun_id = lun_info.get("LUN ID")
                    if not lun_id or lun_id == "--":
                        continue
                    lun_id_list.append(lun_id)
            if not lun_id_list:
                continue

            for host_group_info in host_group_host_info_dict:
                if host_group_id != host_group_info.get("Id"):
                    continue
                host_info_list = host_group_info.get("Host List", [])
                for host_info in host_info_list:
                    host_id = host_info.get("ID")
                    if not host_id or host_id == "--":
                        continue

                    tmp_lun_list = host_lun_dict.get(host_id, [])
                    tmp_lun_list.extend(lun_id_list)
                    host_lun_dict[host_id] = tmp_lun_list
    return host_lun_dict


def get_fc_ini_from_config(mem_obj, p_version, sn, logger):
    """
    获取FC启动器信息。
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    host_fc_ini_dict = {}
    # 由于Dorado v6才能查询到FC启动器信息，融合存储config中没有
    if product.is_dorado_series_version(p_version):
        fc_info_dict = \
            ana_util.get_fc_initiator_info_from_running_data(
                sqlite_conn)
        for fc_info in fc_info_dict:
            host_id = fc_info.get("Host Id")
            wwn = fc_info.get("WWN")

            if host_id == "--":
                continue

            tmp_ini_list = host_fc_ini_dict.get(host_id, [])
            tmp_ini_list.append(wwn)
            host_fc_ini_dict[host_id] = tmp_ini_list
    return host_fc_ini_dict


def get_mapping_from_config_v6(mem_obj, p_version, sn, logger):
    """
    v6 获取映射信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return:
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        mem_obj, sn, logger
    )
    mapping = {}
    # 由于Dorado v6才有mapping信息
    if product.is_dorado_series_version(p_version):
        mapping_dict_list = \
            ana_util.get_mapping_info_from_running_data(
                sqlite_conn)
        for mapping_dict in mapping_dict_list:
            mapping_id = mapping_dict.get("Id", "")
            host_group_id = mapping_dict.get("Host GroupId", "")
            lun_group_id = mapping_dict.get("Lun GroupId", "")
            host_id = mapping_dict.get("Host Id", "")
            lun_id_list = mapping_dict.get("Lun Id List", "").replace(
                "{", "").replace("}", "").split(",")
            port_group_id = mapping_dict.get("Port GroupId", "")
            mapping[mapping_id] = {"Host Group ID": host_group_id,
                                   "Port Group ID": port_group_id,
                                   "Host ID": host_id,
                                   "Lun Group ID": lun_group_id,
                                   "Lun ID List": lun_id_list
                                   }

    return mapping


def get_mapping_view_from_config_ocean_stor(mem_obj, p_version, sn, logger):
    """
    融合存储获取映射信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :param logger: 日志
    :return: 字典列表
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    # 融合存储才有mapping信息
    if not product.is_dorado_series_version(p_version):
        return ana_util.get_mapping_view_info_from_running_data(
            sqlite_conn)
    return {}


def get_all_view_info_from_config_ocean_stor(hyper_lun_id, mem_obj,
                                             p_version, sn, logger):
    # need_xx需要通过命令重新查询的对象
    need_host_group_list = []
    need_port_group_list = []
    need_lun_group_list = []
    # view信息时候解析成功
    is_view_exist_flag = True
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    mapping_view_dict = get_mapping_view_from_config_ocean_stor(mem_obj,
                                                                p_version, sn, logger)
    if not mapping_view_dict:
        is_view_exist_flag = False
        return (is_view_exist_flag, need_host_group_list,
                need_port_group_list, need_lun_group_list)

    # 从config中获取主机组、端口组、LUN组信息
    host_group_dict = get_host_group_host_from_config(mem_obj, p_version, sn, logger)
    port_group_dict = get_port_group_port_from_config(mem_obj, p_version, sn, logger)
    lun_group_info_dict = ana_util.get_lun_group_info_from_running_data(
        sqlite_conn)

    for record in mapping_view_dict:
        lun_group_id = record.get("Lun Group Id")
        host_group_id = record.get("Host Group Id")
        port_group_id = record.get("Port Group Id")
        # 如果任一个group为--，表示此view要么没有映射LUN，要么没有主机
        if host_group_id == "--" or lun_group_id == "--":
            continue

        lun_id_list = []
        for lun_group in lun_group_info_dict:
            if lun_group_id != lun_group.get("Id"):
                continue
            lun_info_list = lun_group.get("LUN INFO", [])
            for lun_info in lun_info_list:
                lun_id = lun_info.get("LUN ID")
                if not lun_id or lun_id == "--":
                    continue

                # 如果不是双活LUN则不处理
                if lun_id not in hyper_lun_id:
                    continue

                lun_id_list.append(lun_id)

        if not lun_id_list:
            continue

        # mapping view中有，但config中未查到则使用命令查询
        if port_group_id and port_group_id != "--":
            if not port_group_dict.get(port_group_id, []):
                need_port_group_list.append(port_group_id)

        # mapping view中有，但config中未查到则使用命令查询
        if host_group_id and host_group_id != "--":
            if not host_group_dict.get(host_group_id, []):
                need_host_group_list.append(host_group_id)

    return (is_view_exist_flag, need_host_group_list,
            need_port_group_list, need_lun_group_list)


def get_host_group_host_from_config(mem_obj, p_version, sn, logger):
    """
    获取主机主下主机信息
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :return: 字典
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    host_group_host_dict = {}
    # 融合存储和dorado v6获取信息不一样
    if product.is_dorado_series_version(p_version):
        host_group_host_list = \
            ana_util.get_host_group_host_info_from_running_data(
                sqlite_conn)
        for host_group_host in host_group_host_list:
            host_group_id = host_group_host.get("Host Group Id", "")
            # 防止1对多时，跟多个host id。
            host_id_list = host_group_host.get("Host Id", "").split(",")
            tmp_host_list = host_group_host_dict.get(host_group_id, [])
            tmp_host_list.extend(host_id_list)
            host_group_host_dict[host_group_id] = tmp_host_list
    else:
        host_group_list = \
            ana_util.get_host_group_info_from_running_data(
                sqlite_conn)
        for host_group in host_group_list:
            host_group_id = host_group.get("Id", "")
            host_info_dict_list = host_group.get("Host List", [])
            tmp_host_list = host_group_host_dict.get(host_group_id, [])
            for host_info in host_info_dict_list:
                host_id = host_info.get("ID")
                tmp_host_list.append(host_id)
            host_group_host_dict[host_group_id] = tmp_host_list

    return host_group_host_dict


def get_port_group_port_from_config(mem_obj, p_version, sn, logger):
    """
    获取端口组下的端口信息。
    :param mem_obj: 内存对象
    :param p_version: 版本信息
    :param sn: 阵列SN
    :return: 字典
    """
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(mem_obj, sn,
                                                              logger)
    port_group_port_dict = {}
    # 融合存储和dorado v6获取信息不一样
    if product.is_dorado_series_version(p_version):
        port_group_port_list = \
            ana_util.get_port_group_port_info_from_running_data(
                sqlite_conn)
        for port_group_host in port_group_port_list:
            port_group_id = port_group_host.get("Port Group Id", "")
            # 防止1对个port
            port_id_list = port_group_host.get("Port Id", "").split(",")
            tmp_port_list = port_group_port_dict.get(port_group_id, [])
            tmp_port_list.extend(port_id_list)
            port_group_port_dict[port_group_id] = tmp_port_list

    # 融合存储查询出的port id 是：hysical Port Id:0x1110600无法使用
    return port_group_port_dict


def get_all_mapping_info_from_config_v6(hyper_lun_id_list, mem_obj,
                                        p_version, sn, logger):
    """
    从config中获取mapping的全部对象（包含主机组、端口组、LUN组等对象信息）。
    哪个对象不存在就使用命令查哪个对象。
    mapping又分两种场景
    1. LUN 直接映射给主机
    2. LUN 通过LUN group 映射给host group.
    :param hyper_lun_id_list:
    :param mem_obj:
    :param p_version:
    :param sn:
    :return:
    """
    mapping_flag = True
    # 需要使用命令查询的对象
    port_group_list = []
    host_group_list = []
    mapping_info_dict = get_mapping_from_config_v6(mem_obj, p_version, sn, logger)
    if not mapping_info_dict:
        mapping_flag = False

    lun_host_dict = {}
    for mapping_id in mapping_info_dict:
        tmp_host_list = []
        tmp_dict = mapping_info_dict.get(mapping_id, {})
        host_group_id = tmp_dict.get("Host Group ID")
        port_group_id = tmp_dict.get("Port Group ID")
        lun_id_list = tmp_dict.get("Lun ID List")
        host_id = tmp_dict.get("Host ID")

        cur_hyper_list = []
        for lun_id in lun_id_list:
            if lun_id not in hyper_lun_id_list:
                continue
            cur_hyper_list.append(lun_id)

        # 当前mapping没有双活LUN
        if not cur_hyper_list:
            continue

        # 如果主机不为--，
        if host_id and host_id != "--":
            tmp_host_list.append(host_id)

        if host_group_id and host_group_id != "--":
            host_group_dict = get_host_group_host_from_config(
                mem_obj, p_version, sn, logger)
            if not host_group_dict:
                host_group_list.append(host_group_id)
            host_list = host_group_dict.get(host_group_id, [])
            tmp_host_list.extend(host_list)

        for lun_id in cur_hyper_list:
            # 更新LUN映射的主机，通过主机找启动器
            tmp_lun_host_list = lun_host_dict.get(lun_id, [])
            tmp_lun_host_list.extend(tmp_host_list)
            lun_host_dict[lun_id] = list(set(tmp_lun_host_list))

        if port_group_id and port_group_id != "--":
            port_group_info_dict = get_port_group_port_from_config(mem_obj,
                                                                   p_version,
                                                                   sn, logger)
            if not port_group_info_dict:
                port_group_list.append(port_group_id)

    return mapping_flag, lun_host_dict, port_group_list, host_group_list


def get_cur_cli_configuration(env, cli, logger):
    """
    获取当前cli配置信息
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :return:
    """
    cmd = "show cli configuration"
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        raise common.UnCheckException(msg, cli_ret)

    dict_list = cliUtil.getVerticalCliRet(cli_ret)
    return dict_list, cli_ret


def enter_cli_special_mode(env, cli, logger, mode):
    """
    进入cli特殊模式
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :param mode: 指定模式
    :return:
    """
    cmd = "change cli capacity_mode={0}".format(mode)
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli,
                                                            cmd, logger)
    logger.logInfo("cmd flag is:{0}, msg:{1}".format(flag, msg))
    return cli_ret


def get_snapshot_info(env, cli, logger):
    """
    获取快照ID,源LUN ID，容量信息
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :return:
    """
    cmd = "show snapshot general |filterColumn include" \
          " columnList=ID,Source\\sLUN\\sID,Capacity"
    lang = common.getLang(env)
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        logger.logError("execute cmd error! %s" % ret)
        raise common.UnCheckException(msg, ret)

    info_dict_list = cliUtil.getHorizontalNostandardCliRet(ret)
    return info_dict_list, ret


def get_rdma_port_info_cache(env, cli, logger):
    """
    获取RDMA端口信息
    :param env: 上下文
    :param cli: 连接
    :param logger: 日志
    :return:
    """
    cmd = "show port general physical_type=RDMA"
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        raise common.UnCheckException(msg, ret)

    exp_port_index = ret.find("Expansion Port:")
    clu_port_index = ret.find("Cluster Port:")
    exp_port_dict_list = cliUtil.getHorizontalCliRet(
        ret[exp_port_index: clu_port_index]
    )
    clu_port_dict_list = cliUtil.getHorizontalCliRet(
        ret[clu_port_index:]
    )

    return exp_port_dict_list, clu_port_dict_list, ret


def get_smart_cache_partition_cache(env, cli, logger):
    """
    获取全部分区
    :param env: 上下文
    :param cli: ssh连接
    :param logger: 日志
    :return: 回文，分区列表
    """
    cmd = "show smart_cache_partition general"
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    # 某些型号可能没有命令
    if not cliUtil.hasCliExecPrivilege(ret):
        return ret, dict()
    # 可能不存在license
    if flag == cliUtil.RESULT_NOSUPPORT:
        return ret, dict()

    if flag is not True:
        raise common.UnCheckException(msg, ret)

    return ret, cliUtil.getHorizontalCliRet(ret)


def get_smart_cp_detail_cache(env, cli, logger, scp_id):
    """
    获取分区中的LUN
    :param env: 上下文
    :param cli: ssh连接
    :param logger: 日志
    :param scp_id: 分区ID
    :return: 回文，分区详情
    """
    cmd = "show smart_cache_partition lun " \
          "smart_cache_partition_id={}".format(scp_id)
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        raise common.UnCheckException(msg, ret)

    return ret, cliUtil.getHorizontalCliRet(ret)


def download_ana_config(py_java_env, logger, cli):
    lang = common.getLang(py_java_env)
    devInfo = py_java_env.get("devInfo")
    context = py_java_env.get("objectForPy")
    return download_ana_config_back_end(context, logger, cli, devInfo, lang)


def download_ana_config_back_end(py_obj, logger, cli, devInfo, lang):
    p_version = str(devInfo.getProductVersion())
    sn = devInfo.getDeviceSerialNumber()
    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(py_obj, sn,
                                                              logger)
    # 如果是使用config则不提前查询
    if is_use_running_data(p_version):
        return download_and_update_cache(
            sn, py_obj, logger, cli, lang, sqlite_conn, p_version
        )

    return False


def download_and_update_cache(sn, py_obj, logger, cli, lang, conn, p_version):
    """
    下载running data
    :param sn:
    :param py_obj:
    :param logger:
    :param cli:
    :param lang:
    :param conn:
    :param p_version:
    :return:
    """
    running_key = RUNNING_DATA.format(str(sn))
    obj_py = py_obj.get(running_key)
    if is_downloaded_config_back_end(py_obj, sn):
        logger.logInfo("Has downloaded. no need do it again.")
        return True
    else:
        logger.logInfo("not downloaded config. need to do.")
        if not ana_running_data(cli, sn, p_version, lang, conn, logger, py_obj):
            return False
        if not obj_py:
            obj_py = {}
        obj_py[sn] = {"running_data_dict": True,
                      "p_version": p_version}
        py_obj.put(running_key, obj_py)
        return True


def is_downloaded_config(env, sn):
    py_obj = env.get("objectForPy")
    return is_downloaded_config_back_end(py_obj, sn)


def is_downloaded_config_back_end(py_obj, sn):
    running_key = RUNNING_DATA.format(str(sn))
    obj_py = py_obj.get(running_key)
    return obj_py and obj_py.get(sn, {})


def get_product_version_from_mem(mem_obj, sn):
    """
    获取running data抽取的公共方法
    :param mem_obj: 内存对象
    :param sn: 阵列SN
    :return:
    """
    running_data_dicts = mem_obj.get(RUNNING_DATA.format(str(sn)))
    if not running_data_dicts:
        return ""

    tmp_dict = running_data_dicts.get(sn, {})
    p_version = tmp_dict.get("p_version", "")
    return p_version


def get_hyper_host_from_config(py_java_env, sn, logger, hyper_lun_list):
    host_list = []
    mem_obj = py_java_env.get("objectForPy")
    p_version = get_product_version_from_mem(mem_obj, sn)
    host_lun_info_list = get_host_lun_info_from_config(mem_obj,
                                                       p_version,
                                                       sn, logger)

    logger.logInfo("host and lun {0}:{1}".format(sn, host_lun_info_list))
    for host_id, lun_list in host_lun_info_list.iteritems():
        for lun_id in lun_list:
            if lun_id in hyper_lun_list:
                host_list.append(host_id)
                break

    return host_list


def get_smartcooling_mode(cli, lang):
    """
    获取控制框风扇智能调速模式
    :param cli:
    :param lang:
    :return:
    """
    cmd = "show smartcooling_mode"
    flag, cli_ret, e_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        raise UnCheckException(e_msg, cli_ret)

    ret_list = cliUtil.getVerticalCliRet(cli_ret)
    for res_dict in ret_list:
        return cli_ret, res_dict.get("Temperature"), res_dict.get("Mode")


def get_all_enclosure_list_cache(cli, env, logger):
    """
    获取所有框信息
    :param cli: cli对象
    :param env: 上下文
    :param logger: 日志
    :return:
    """
    cmd = "show enclosure"
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        return flag, ret, list(), msg

    return flag, ret, cliUtil.getHorizontalCliRet(ret), msg


def get_license_cache(cli, env, logger):
    """
    获取license信息
    :param cli: cli对象
    :param env: 上下文
    :param logger: 日志
    :return:
    """
    cmd = "show license"
    return execute_cmd_in_cli_mode_with_cache(env, cli, cmd, logger)


def get_horizontal_cache(cli, env, logger, cmd):
    """
    标准横向表格-缓存命令回文
    :param cli: cli对象
    :param env: 上下文
    :param logger: 日志
    :return:
    """
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        return flag, ret, list(), msg

    return flag, ret, cliUtil.getHorizontalCliRet(ret), msg


def get_horizontal_no_standard_cache(cli, env, logger, cmd):
    """
    非标准横向表格-缓存命令回文
    :param cli: cli对象
    :param env: 上下文
    :param logger: 日志
    :return:
    """
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        return flag, ret, list(), msg

    return flag, ret, cliUtil.getHorizontalNostandardCliRet(ret), msg


def get_vertical_cache(cli, env, logger, cmd):
    """
    标准纵向表格-缓存命令回文
    :param cli: cli对象
    :param env: 上下文
    :param logger: 日志
    :return:
    """
    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                        logger)
    if flag is not True:
        return flag, ret, list(), msg

    return flag, ret, cliUtil.getVerticalCliRet(ret), msg
