# -*- coding: UTF-8 -*-
import traceback
import datetime
import collections

from cbb.frame.dsl.adapter import get_sn
from java.lang import Exception as JException
from memory_inspect.adapter.java_adapter import get_common_adapter
from memory_inspect.db.task_db_service import DbService
from memory_inspect.rule.memory_rule import RuleContext
from memory_inspect.utils.constants import TIME_SEC_FORMATTER
from memory_inspect.adapter.java_adapter import get_msg, UnCheckException, get_lang


ITEM_SHARE_DATA_KEY_PREFIX = "hardware_timed_inspect_data_"
STATUS_TO_CHECK_RESULT = collections.OrderedDict(
    {"not_pass": False, "error": False, "warning": "WARNING", "pass": True})


class TimedLogBaseCheckItem(object):
    def __init__(self, py_java_env, logger, cli):
        self.logger = logger
        self.py_java_env = py_java_env
        self.lang = get_lang(py_java_env)
        self.context = self.get_context(self.py_java_env, cli)
        self.sn = self.context.get("dev").get("sn")
        self.dev_type = self.context.get("dev").get("type")
        self.dev_ver = self.context.get("dev").get("version")
        self.add_extra_context()
        self.db_service = DbService(self.context)
        self.start_time = self.format_start_time(self.context.get("start_time"))
        self.last_err_time = self.start_time

    def execute(self):
        status = "error"
        detail = get_msg(self.lang, "query.result.abnormal")
        try:
            self.logger.info("begin to run check item")
            status, cli_ret, detail = self.do_check()
            return STATUS_TO_CHECK_RESULT.get(status), cli_ret, detail
        except UnCheckException as ex:
            detail = ex.error_msg
            self.logger.error("execute memory inspect UnCheckException. {} {}".format(status, detail))
            return False, ex.cli_ret, detail
        except (Exception, JException):
            self.logger.error("execute memory inspect error. {}".format(traceback.format_exc()))
            return False, str(traceback.format_exc()), detail
        finally:
            self._update_check_result(status, detail)

    def do_check(self):
        raise NotImplementedError

    def get_context(self, py_java_env, cli):
        # 测试时方便mock，在方法内import
        from cbb.frame.checkitem.context_adapter import InspectContext
        py_java_env["lang"] = self.lang
        py_java_env["logger"] = self.logger
        inspect_context = InspectContext(py_java_env, cli)
        context = inspect_context.get_context()
        context["start_time"] = get_common_adapter().get_check_period_start_time()
        context["db_full_name"] = get_common_adapter().get_memory_task_db_path()
        context["work_path"] = get_common_adapter().get_work_path()
        context["inspect_work_path"] = get_common_adapter().get_inspect_work_path()
        context["log_path"] = get_common_adapter().get_work_path() + "\\log_collect\\" + get_sn(context)
        self.logger.info("log path={}, start_time={}, db full name={}".format(
            context.get("log_path"), context.get("start_time"), context.get("db_full_name")))
        return context

    def add_extra_context(self):
        self.context["collected_files"] = self.get_collected_files()

    def get_collected_files(self):
        object_py = self.py_java_env.get("objectForPy")
        share_data = object_py.get(ITEM_SHARE_DATA_KEY_PREFIX + self.sn) or {}
        return share_data.get("collected_logs", {})

    def get_sys_time(self):
        object_py = self.py_java_env.get("objectForPy")
        share_data = object_py.get(ITEM_SHARE_DATA_KEY_PREFIX + self.sn) or {}
        return share_data.get("sys_time", "")

    def save_share_data_for_all_item(self, data):
        object_py = self.py_java_env.get("objectForPy")
        object_py.put(ITEM_SHARE_DATA_KEY_PREFIX + self.sn, data)

    def _get_rule_context(self):
        rule_context = RuleContext()
        rule_context.db_service = self.db_service
        rule_context.current_time = self.format_system_time(self.get_sys_time())
        rule_context.sn = self.sn
        rule_context.logger = self.logger
        rule_context.lang = self.lang
        rule_context.last_err_time = self.last_err_time
        return rule_context

    @staticmethod
    def format_start_time(timestamp):
        start_time_obj = datetime.datetime.fromtimestamp(int(timestamp) / 1000)
        start_time_str = start_time_obj.strftime(TIME_SEC_FORMATTER)
        return start_time_str

    def format_system_time(self, time_str):
        if not time_str:
            self.logger.info("time_str is none")
            return ""
        time_str = time_str.split()[0]
        time_obj = datetime.datetime.strptime(time_str, "%Y-%m-%d/%H:%M:%S")
        return time_obj.strftime("%Y-%m-%d %H:%M:%S")

    def _update_check_result(self, status, detail):
        self.logger.info("sn={},status={},detail={}".format(self.sn, status, detail))
        check_results = self.db_service.task_check_result.get_check_result(self.sn)
        if check_results:
            old_status = check_results[0].get("status")
            sorted_status = STATUS_TO_CHECK_RESULT.keys()
            status = status if sorted_status.index(status) < sorted_status.index(old_status) else old_status
            detail = self._get_detail([check_results[0].get("detail"), detail])
        self.db_service.task_check_result.save_check_result(self.sn, status, detail)

    @staticmethod
    def _get_detail(details):
        valid_details = [detail for detail in details if detail]
        return " | ".join(valid_details)
