# coding: utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re
from datetime import datetime


class RasdaemonParser(object):
    """
    功能描述：用于解析rasdaemon.log
    接口：
    """
    _time_regex = re.compile(
        r"(\d{4}-[0-2]\d-[0-3]\d\s[0-2]\d:[0-5]\d:[0-5]\d)")
    _is_scrub_error_regex = re.compile("(Scrub)")
    _mc_regex = re.compile(r"mc: (\w+)")
    _addr_regex = re.compile(r"address: (\w+)")
    _grain_regex = re.compile(r"grain: (\w+)[\s][A-Z]{0,8}")
    _node_regex = re.compile(r"node:(\w+)")
    _card_regex = re.compile(r"card:(\d+)")
    _module_regex = re.compile(r"module:(\d+)")
    _rank_regex = re.compile(r"rank:(\d+)")
    _bank_regex = re.compile(r"bank:(\d+)")
    _device_regex = re.compile(r"device:(\d+)")
    _row_regex = re.compile(r"row:(\d+)")
    _col_regex = re.compile(r"col:(\d+)")
    _bit_pos_regex = re.compile(r"bit_pos:(\d+)")
    _key_regex_map = {
        "time": _time_regex,
        "is_scrub_error": _is_scrub_error_regex,
        "mc": _mc_regex,
        "sys_addr": _addr_regex,
        "grain": _grain_regex,
        "cpu": _node_regex,
        "card": _card_regex,
        "slot": _module_regex,
        "rank": _rank_regex,
        "bank": _bank_regex,
        "device": _device_regex,
        "row": _row_regex,
        "col": _col_regex,
        "bit_pos": _bit_pos_regex
    }

    def __init__(self, logger):
        self.logger = logger

    def parse(self, file_path):
        """
        功能描述：解析rasdaemon.log
        参数：
            file_path: rasdaemon.log所在路径
        返回值：
            params 解析rasdaemon.log结果
        异常描述：
        """
        self.logger.info("begin to parse file={}".format(file_path))
        all_result = list()
        with open(file_path, 'r') as log_file:
            for number, line in enumerate(log_file):
                err_info = self._parse_err_info(line, number)
                if err_info:
                    all_result.append(err_info)

        self.logger.info("success parse: {}".format(file_path))
        return all_result

    def _parse_err_info(self, line, line_index):
        err_info = dict()
        err_type = self._get_err_type(line)
        if err_type:
            err_info.update({"line_index": line_index, "err_type": err_type})
            err_info = self.__get_errors(line, err_info)
            err_info = self._add_err_info(err_info=err_info)
            return err_info

    def __get_errors(self, line, err_info=None):
        """
        匹配line中的错误信息。[rasdaemon中一条错误信息在一行]
        :param line: 一行错误信息
        :param err_info: 收集的错误信息
        :return: 匹配line后的错误信息
        """
        if err_info is None:
            err_info = dict()

        for err_key in self._key_regex_map:
            try:
                search_obj = self._key_regex_map[err_key].search(line)
                err_info[err_key] = search_obj.group(1) if search_obj else None
            except Exception as error:
                self.logger.error("regex={}, line={} error={}".format(
                    self._key_regex_map[err_key], line, error))
                err_info[err_key] = None
        return err_info

    def _add_err_info(self, err_info=None):
        """
        更改desc中字段信息为数据库要求格式
        :param err_info: 错误信息
        :return:
        """
        if err_info is None:
            err_info = dict()

        err_info["module"] = "DDRC"
        err_info["record_date"] = datetime.strptime(err_info["time"], "%Y-%m-%d %H:%M:%S")
        err_info["ordinal"] = "%s[%s]" % (err_info["time"],
                                          str(err_info["line_index"]).zfill(6))
        self._update_phy_addr(err_info)
        self._update_err_type(err_info)

        if len(err_info["phy_addr"].values()) > 0:
            err_info["with_phy_addr"] = True
        else:
            err_info["with_phy_addr"] = False
        return err_info

    @staticmethod
    def _format_int(value):
        if value and value.isdigit():
            return int(value)
        return None

    @staticmethod
    def _get_err_type(line):
        if "address" not in line:
            return None
        if "uncorrected error on" in line:
            return "UCE"
        if "corrected error on" in line:
            return 'CE'
        return None

    @staticmethod
    def _update_err_type(err_info):
        if err_info.get("is_scrub_error"):
            err_info["is_scrub_error"] = True
            if err_info["err_type"] == "UCE":
                err_info["err_type"] = "Scrub UCE"
            else:
                err_info["err_type"] = "Scrub CE"
        else:
            err_info["is_scrub_error"] = False
            if err_info["err_type"] == "UCE":
                err_info["err_type"] = "HA UCE"
            else:
                err_info["err_type"] = "HA CE"

    def _update_phy_addr(self, err_info):
        err_info["phy_addr"] = {
            "bank": self._format_int(err_info.get("bank")),
            "bankgroup": 0,
            "column": self._format_int(err_info.get("col")),
            "rank": self._format_int(err_info.get("rank")),
            "row": self._format_int(err_info.get("row")),
            "dev": self._format_int(err_info.get("device"))
        }
