# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
"""
内存故障检查规则实现模块
包含多地址硬失效、多Bank、多Device、多槽位、长时间报错、ce风暴规则等
每个规则对应一个类，并继承至RuleBase
"""
import re
import os
import datetime
from memory_inspect.adapter.java_adapter import get_msg
from collections import defaultdict, OrderedDict

HA_CE_ERR = "HA CE"
HA_UCE_ERR = "HA UCE"
MESSAGE_MEMORY_ERROR = "Message Memory Error"


class RuleContext(object):
    def __init__(self):
        self.db_service = None
        self.sn = ""
        self.logger = None
        self.lang = ""
        self.current_time = ""
        self.analyse_ctrls = []
        self.last_err_time = None
        self.vendor_data = {}


class RuleBase(object):

    def __init__(self, rule_context):
        self.rule_context = rule_context
        self.db_service = rule_context.db_service
        self.logger = rule_context.logger
        self.lang = rule_context.lang
        self.origin_info = ""
        self.err_msg = []

    def execute(self, **kwargs):
        """
        检查规则是否命中
        :return: True-规则命中，False-规则不命中
        """
        raise NotImplementedError()

    def filter_need_analyzed_records(self, all_ctrl_grouped_records):
        for ctrl, one_ctrl_records in all_ctrl_grouped_records.items():
            if ctrl not in self.rule_context.analyse_ctrls:
                all_ctrl_grouped_records.pop(ctrl)

    def add_db_record_to_origin_info(self, records, desc):
        if not records:
            self.origin_info += os.linesep + desc + os.linesep + str(records)
            return
        self.origin_info += os.linesep + desc + os.linesep
        self.origin_info += os.linesep.join([str(record) for record in records])

    @staticmethod
    def group_by(records, key):
        ret = defaultdict(list)
        for item in records:
            ret[item[key]].append(item)

        # 对字典进行排序
        sorted_ret = OrderedDict()
        for key in sorted(ret.keys()):
            sorted_ret[key] = ret.get(key)
        return sorted_ret

    @staticmethod
    def get_time_by_delta_process(time, **delta):
        """
        对时间进行增量处理
        :param time: 初始时间
        :param delta: 增量
        :return: 新的时间
        """
        old_time = datetime.datetime.strptime(time, "%Y-%m-%d %H:%M:%S")
        new_time = (old_time + datetime.timedelta(**delta)).strftime("%Y-%m-%d %H:%M:%S")
        return new_time


class RuleMultiAddrErr(RuleBase):
    """
    多地址硬失效：最近30天内，去重后，总错误地址数量>=20个，至少有1个地址报错>=2次以上
    报错详细信息：控制器（0A）槽位（040）存在多地址硬失效
    """

    def __init__(self, rule_context):
        super(RuleMultiAddrErr, self).__init__(rule_context)

    def execute(self, time_period_day=30, total_err_addr_count=20, same_addr_err_count=2, min_multi_addr_count=1):
        time_period_day = int(time_period_day)
        total_err_addr_count = int(total_err_addr_count)
        min_multi_addr_count = int(min_multi_addr_count)
        same_addr_err_count = int(same_addr_err_count)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot, addr, count(addr) as err_count, sum(dq) as multi_dq_num FROM err_addr_tbl " \
                  "WHERE log_time >= '{}' and sn = '{}' and err_type='{}'" \
                  "GROUP BY ctrl, slot, addr".format(start_time, self.rule_context.sn, HA_CE_ERR)

        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi addr err records:")
        return self._check_rule_by_record(records, same_addr_err_count, total_err_addr_count, min_multi_addr_count)

    def _check_rule_by_record(self, records, same_addr_err_count, total_err_addr_count, min_multi_addr_count):
        """
        records示例：
        # [{ctrl:"0A", slot: "052", addr: "0x800", err_count: 3, multi_dq_num: 2}]
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                multi_err_addr_records = [record for record in one_slot_records
                                          if record.get("err_count") >= same_addr_err_count]
                has_multi_dq_records = [record for record in one_slot_records if record.get("multi_dq_num") >= 1]
                if len(one_slot_records) >= total_err_addr_count \
                        and len(multi_err_addr_records) >= min_multi_addr_count \
                        and has_multi_dq_records:
                    self.err_msg.append(get_msg(self.lang, "memory.check.multi.addr.err",
                                                ctrl, slot))
        return bool(self.err_msg)


class RuleMultiDeviceErr(RuleBase):
    """
    ARM 设备
    多颗粒报错：最近30天内，有多个颗粒(>=2)报错
    报错详细信息：控制器（0A）槽位（040）存在多颗粒报错
    """

    def __init__(self, db_service):
        super(RuleMultiDeviceErr, self).__init__(db_service)

    def execute(self, time_period_day=30, err_device_count=2):
        time_period_day = int(time_period_day)
        err_device_count = int(err_device_count)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot, rank, bank, bgroup as device, row FROM err_addr_tbl " \
                  "where log_time >= '{}' and sn = '{}' and err_type='{}'" \
                  "GROUP BY ctrl, slot, rank, bank, device, row".format(start_time, self.rule_context.sn, HA_CE_ERR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi device check records:")
        return self._check_rule_by_record(records, err_device_count)

    def _check_rule_by_record(self, records, err_device_count):
        """
        records示例：
        # [{ctrl: "", slot: "", device: "", device_num: ""}]
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                if self.is_multi_device(one_slot_records, err_device_count):
                    self.err_msg.append(
                        get_msg(self.lang, "memory.check.multi.device.err", ctrl, slot))
        return bool(self.err_msg)

    def is_multi_device(self, one_slot_records, err_device_count):
        """
        至少存在2个不同device的相同bank下row的范围存在交集则检查不通过。。
        【3个不同device，以前的规则不通过，现在的规则规则也是不通过， 存在两个device 的 row 的范围有交集。】
        ctrl	slot	rank	bank device	 row
        0A	    300		0		11		8	45510
        0A	    300		0		11		8	45512
        0A	    300		0		11		9	45511
        0A	    300		0		11		9	45519
        0A	    300		0		11		10	45514
        0A	    300		0		11		10	45519
        不同device的 row 范围：
        device rank  bank   row
        8        0    11    45510 ~ 45512
        9        0    11    45511 ~ 45519
        10       0    11    45514 ~ 45519

        :param one_slot_records: 同槽位的记录
        :param err_device_count: 不通过的数量标准
        :return: True 是多颗粒， False 不是多颗粒
        """
        device_row_dict = {}
        # dev_rank 的数据集合 {"dev_rank": {"bank1": [min row, max row], "bank2": [min row, max row]}, ..}
        diff_dev_str = "{}_{}"

        self.get_dev_row_dict(device_row_dict, diff_dev_str, one_slot_records)

        self.logger.info("device_row_dict is: {}".format(device_row_dict))
        if len(device_row_dict) < err_device_count:
            return False

        err_dev = self._check_diff_dev(device_row_dict)
        self.logger.info("is_multi_device: {}".format(err_dev))
        return bool(err_dev)

    def get_dev_row_dict(self, device_row_dict, diff_dev_str, one_slot_records):
        """
        过滤组装数据为 dev 和 row 的字典格式如：
        dev_rank: bank: [row1 row 2]

        device_row_dict:{
            '8_1': {22: [45510, 45511, 45512], 23: [45511, 45514]},
            '8_0': {11: [45510, 45512], 22: [45512, 45516]}
        }
        :param device_row_dict: 需要的不同device的bank 下的 row 范围
        :param diff_dev_str: dev和 rank 组成的唯一device信息
        :param one_slot_records: 数据库查出来的按slot和控制器分组的初始数据。
        :return:
        """
        for record in one_slot_records:
            key_str = diff_dev_str.format(record.get("device"), record.get("rank"))
            row_value = record.get("row")
            temp_row_dict = device_row_dict.get(key_str, {})
            row_list = temp_row_dict.get(record.get("bank"), [])
            row_list.append(row_value)
            temp_row_dict[record.get("bank")] = row_list
            device_row_dict[key_str] = temp_row_dict

    def _check_diff_dev(self, dev_records):
        """
        检查不同device下是否存在相同bank数据，且两个device间的row的最小值和最大值范围内是否存在交集
        :param dev_records: 处理过的 device和bank和row 数据
        :return: 命中的多颗粒报错
        """
        all_diff_dev = []
        for dev_first, bank_row_dict_first in dev_records.items():
            for dev_sec, bank_row_dict_sec in dev_records.items():
                if dev_first == dev_sec:
                    continue
                self._check_dev_row(all_diff_dev, dev_first, dev_sec, bank_row_dict_first, bank_row_dict_sec)
        return all_diff_dev

    def _check_dev_row(self, all_diff_dev, dev_first, dev_sec, bank_row_dict_first, bank_row_dict_sec):
        """
        核心规则判断方法
        1. 不同device是否存在相同bank
        2. 判断两个row最大值和最小值范围是否存在交集
        :param all_diff_dev: 命中规则的结果
        :param dev_first: 比较dev对象1
        :param dev_sec: 比较dev对象2
        :param bank_row_dict_first: device的所有的bank下的row范围
        :param bank_row_dict_sec: device的所有的bank下的row范围
        :return:
        """
        all_bank2 = set(bank_row_dict_sec.keys())
        all_bank1 = set(bank_row_dict_first.keys())
        # 不同device是否存在相同bank，即两个集合是否存在交集
        same_bank_list = all_bank1.intersection(all_bank2)
        if not same_bank_list:
            return

        for same_bank in same_bank_list:
            row_list1 = bank_row_dict_first.get(same_bank)
            row_list2 = bank_row_dict_sec.get(same_bank)
            row_list1.sort()
            row_list2.sort()
            max_row1, min_row1 = row_list1[-1], row_list1[0]
            max_row2, min_row2 = row_list2[-1], row_list2[0]
            # 判断两个row最大值和最小值范围是否存在交集（不是判断两个row集合是否存在相同）
            # a 的最小值大于 b的最大值，a 的最大值 小于 b 的最小值
            if min_row1 > max_row2 or max_row1 < min_row2:
                continue
            # 因是双重循环所以去重判断处理。A,B 和 B,A 是同一组不同device的数据
            if (dev_first, dev_sec) in all_diff_dev or (dev_sec, dev_first) in all_diff_dev:
                continue
            all_diff_dev.append((dev_first, dev_sec))


class RuleMultiDeviceErrX86(RuleMultiDeviceErr):
    """
    多颗粒报错：最近30天内，有多个颗粒(>=2)报错
    报错详细信息：控制器（0A）槽位（040）存在多颗粒报错
    """

    def __init__(self, db_service):
        super(RuleMultiDeviceErrX86, self).__init__(db_service)

    def execute(self, time_period_day=30, err_device_count=2):
        time_period_day = int(time_period_day)
        err_device_count = int(err_device_count)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot, bgroup as device, bank, rank, row FROM err_addr_tbl " \
                  "where log_time >= '{}' and sn = '{}' and err_type='{}'" \
                  "GROUP BY ctrl, slot, device, bank, rank, row".format(start_time, self.rule_context.sn, MESSAGE_MEMORY_ERROR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi device check records:")
        return self._check_rule_by_record(records, err_device_count)


class RuleMultiBankErr(RuleBase):
    """
    ARM 设备
    多bank报错：最近30天内，有多个bank(>=2)报错
    报错详细信息：控制器（0A）槽位（040）存在多bank报错
    """

    def __init__(self, db_service):
        super(RuleMultiBankErr, self).__init__(db_service)

    def execute(self, time_period_day=30, err_bank_count=3):
        # 同rank 不同bank
        time_period_day = int(time_period_day)
        err_bank_count = int(err_bank_count)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot, bank, rank, bgroup as device FROM err_addr_tbl " \
                  "WHERE log_time >= '{}' AND sn = '{}' and err_type = '{}' " \
                  "GROUP BY ctrl, slot, bank, rank, device".format(start_time, self.rule_context.sn, HA_CE_ERR)

        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi bank check records:")
        return self._check_rule_by_record(records, err_bank_count)

    def is_multi_bank(self, one_slot_records, err_bank_count):
        device_grouped_records = self.group_by(one_slot_records, "device")
        for device, device_grouped in device_grouped_records.items():
            rank_group_records = self.group_by(device_grouped, "rank")
            for rank, bank_group in rank_group_records.items():
                if len(bank_group) >= err_bank_count:
                    return True
        return False

    def _check_rule_by_record(self, records, err_bank_count):
        """
        records示例：
        # [{ctrl: "0A", slot: "030", bank: "2", bank_num: 2"},
         {ctrl: "0A", slot: "030", bank: "3", bank_num: 2"}]
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                if self.is_multi_bank(one_slot_records, err_bank_count):
                    self.err_msg.append(
                        get_msg(self.lang, "memory.check.multi.bank.err", ctrl, slot))
        return bool(self.err_msg)


class RuleMultiBankErrX86(RuleMultiBankErr):
    """
    多bank报错：最近30天内，有多个bank(>=2)报错
    报错详细信息：控制器（0A）槽位（040）存在多bank报错
    """

    def __init__(self, db_service):
        super(RuleMultiBankErrX86, self).__init__(db_service)

    def execute(self, time_period_day=30, err_bank_count=3):
        # 同rank 不同bank
        time_period_day = int(time_period_day)
        err_bank_count = int(err_bank_count)
        has_bst_in_msg, result = self.is_multi_bank_in_message(time_period_day, err_bank_count)
        if has_bst_in_msg:
            return result

        start_time = self.get_time_by_delta_process(
            self.rule_context.current_time, days=-1 * time_period_day
        )
        sql_str = (
            "SELECT ctrl, slot, bank, rank, bgroup FROM err_addr_tbl "
            "WHERE log_time >= '{}' AND sn = '{}' and err_type = '{}' "
            "GROUP BY ctrl, slot, rank, bank, bgroup".format(
                start_time, self.rule_context.sn, HA_CE_ERR
            )
        )

        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi bank check records:")
        return self._check_rule_by_record(records, err_bank_count)

    def is_multi_bank(self, one_slot_records, err_bank_count):
        rank_group_records = self.group_by(one_slot_records, "rank")
        for rank, bank_group in rank_group_records.items():
            if len(bank_group) >= err_bank_count:
                return True
        return False

    def is_multi_bank_in_message(self, time_period_day, err_bank_count):
        time_period_day = int(time_period_day)
        err_bank_count = int(err_bank_count)
        start_time = self.get_time_by_delta_process(
            self.rule_context.current_time, days=-1 * time_period_day
        )
        sql_str = (
            "SELECT ctrl, slot, bank, rank, bgroup as device FROM err_addr_tbl "
            "where log_time >= '{}' and sn = '{}' and err_type='{}'"
            "GROUP BY ctrl, slot, bank, rank, device".format(
                start_time, self.rule_context.sn, MESSAGE_MEMORY_ERROR
            )
        )
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi bank check records:")
        return bool(records), self._check_rule_by_record_by_bst(records, err_bank_count)

    def _check_rule_by_record_by_bst(self, records, err_bank_count):
        """
        message 中的BST判断
        """
        ctrl_grouped_records = self.group_by(records, "ctrl")
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                if self.is_multi_bank_device(one_slot_records, err_bank_count):
                    self.err_msg.append(
                        get_msg(
                            self.lang,
                            "memory.check.multi.bank.err",
                            ctrl,
                            slot,
                        )
                    )
        return bool(self.err_msg)

    def is_multi_bank_device(self, one_slot_records, err_bank_count):
        device_grouped_records = self.group_by(one_slot_records, "device")
        for device, device_grouped in device_grouped_records.items():
            rank_group_records = self.group_by(device_grouped, "rank")
            for rank, bank_group in rank_group_records.items():
                if len(bank_group) >= err_bank_count:
                    return True
        return False


class RuleMultiSlotErr(RuleBase):
    """
    多内存条报错：最近30天内，有多根内存条(>=2)报错
    报错详细信息：控制器（0A）存在多槽位（040）报错
    """

    def __init__(self, db_service):
        super(RuleMultiSlotErr, self).__init__(db_service)

    def execute(self, time_period_day=30, err_slot_count=2):
        time_period_day = int(time_period_day)
        err_slot_count = int(err_slot_count)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot FROM err_addr_tbl " \
                  "WHERE log_time>='{}' AND sn='{}' AND err_type='{}'" \
                  "GROUP BY ctrl, slot".format(start_time, self.rule_context.sn, HA_CE_ERR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "multi slot check records:")
        return self._check_rule_by_record(records, err_slot_count)

    def _check_rule_by_record(self, records, err_slot_count):
        """
        record示例
        # [{ctrl: "0A", slot: "030"}, {ctrl: "0A", slot: "031"}, {ctrl: "0B", slot: "023"}]
        """
        grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(grouped_records)
        for ctrl, one_ctrl_records in grouped_records.items():
            if len(one_ctrl_records) >= err_slot_count:
                err_slots = [record.get("slot") for record in one_ctrl_records]
                self.err_msg.append(get_msg(self.lang, "memory.check.multi.slot.err", ctrl, ",".join(err_slots)))
        return bool(self.err_msg)


class RuleLongTimeErr(RuleBase):
    """
    长时间报错：最近30天，报错天数>=15天，每天报错>=20次
    报错详细信息：控制器（0A）槽位（040）存在长时间报错
    """

    def __init__(self, db_service):
        super(RuleLongTimeErr, self).__init__(db_service)

    def execute(self, time_period_day=30, err_day=15, err_count_per_day=20):
        time_period_day = int(time_period_day)
        err_day = int(err_day)
        err_count_per_day = int(err_count_per_day)
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        sql_str = "SELECT ctrl, slot, time_day, count(time_day) as time_count FROM (" \
                  "SELECT ctrl, slot, strftime('%Y-%m-%d', log_time)  as time_day FROM err_addr_tbl " \
                  "WHERE log_time >= '{}' AND sn='{}' AND err_type='{}' ) " \
                  "GROUP BY ctrl, slot,time_day".format(start_time, self.rule_context.sn, HA_CE_ERR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "long time err check records")
        return self._check_rule_by_record(records, err_count_per_day, err_day)

    def _check_rule_by_record(self, records, err_count_per_day, err_day):
        """
        records示例：
        # [{ctrl: "0A", slot: "030", time_day: "2021-03-15", time_count=10},
        #  {ctrl: "0A", slot: "030", time_day: "2021-03-16", time_count=12}]
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                matched_records = [record for record in one_slot_records if
                                   record.get("time_count", 0) >= err_count_per_day]
                if len(matched_records) >= err_day:
                    self.err_msg.append(get_msg(self.lang, "memory.check.long.time.err", ctrl, slot))
        return bool(self.err_msg)


class RuleCeStorm(RuleBase):
    """
    CE风暴：2023.10.12 更新为 最近8小时报错，每小时>=50个CE, 存在多DQ错误
    报错详细信息：控制器（0A）槽位（040）存在CE风暴
    """

    def __init__(self, rule_context):
        super(RuleCeStorm, self).__init__(rule_context)

    def execute(self, time_period_hour=8, err_count_per_hour=50):
        time_period_hour = int(time_period_hour)
        err_count_per_hour = int(err_count_per_hour)
        self.logger.info("time_period_hour:{}, err_count_per_hour:{}".format(time_period_hour, err_count_per_hour))
        start_time = self.get_time_by_delta_process(self.rule_context.current_time, hours=-1 * time_period_hour)
        sql_str = "SELECT ctrl,slot,time_hour, count(time_hour) as time_count, sum(dq) as multi_dq_num FROM (" \
                  "SELECT ctrl,slot,strftime('%Y-%m-%d %H', log_time) as time_hour," \
                  "dq FROM err_addr_tbl " \
                  "WHERE log_time >= '{}'AND sn='{}' AND err_type='{}')" \
                  "GROUP BY ctrl, slot,time_hour".format(start_time, self.rule_context.sn, HA_CE_ERR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.logger.info(
            "_check_rule_by_record:sql_str: {}, start_time:{}, \nrecords:{}".format(sql_str, start_time, records))
        self.add_db_record_to_origin_info_ce("ce storm check records:")
        return self._check_rule_by_record(records, err_count_per_hour, time_period_hour)

    def add_db_record_to_origin_info_ce(self, desc):
        sql_str = "SELECT ctrl, slot, time_hour, count(time_hour) as time_count, sum(dq) as multi_dq_num FROM (" \
                  "SELECT ctrl, slot, strftime('%Y-%m-%d %H', log_time) as time_hour," \
                  "dq FROM err_addr_tbl " \
                  "WHERE log_time >= '{}' AND sn='{}' AND err_type='{}')" \
                  "GROUP BY ctrl, slot,time_hour".format(
            self.rule_context.last_err_time, self.rule_context.sn, HA_CE_ERR)
        self.logger.info("add_db_record_to_origin_info_ce:sql_str: {}, self.rule_context.last_err_time:{}".format(
            sql_str, self.rule_context.last_err_time))

        records = self.db_service.err_addr_tbl.query(sql_str)
        self.origin_info += os.linesep + desc + os.linesep + os.linesep.join([str(record) for record in records])

    def _check_rule_by_record(self, records, err_count_per_hour, time_period_hour):
        """
        records示例：
        # [{ctrl: "0A", slot: "030", time: "2021-03-15 08", time_count=30},
        #  {ctrl: "0A", slot: "030", time: "2021-03-15 09", time_count=35}]
        最近的48（容错1小时未打满的场景计算用的 47）小时时间的每个小时都需要有30条记录才满足不通过的条件，否则任意一个小时不满足都会检查通过。
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        self.logger.info("after filter ctrl_grouped_records:{}".format(ctrl_grouped_records))
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                has_multi_dq_records = [record for record in one_slot_records if record.get("multi_dq_num") >= 1]
                matched_records = [record for record in one_slot_records if
                                   record.get("time_count", 0) >= err_count_per_hour]
                # 考虑最近1小时数据未打满，少一条也算满足
                if len(matched_records) >= time_period_hour - 1 and has_multi_dq_records:
                    self.logger.info("RuleCeStorm has muti dq and match not pass rule:%s" % matched_records)
                    self.err_msg.append(
                        get_msg(self.lang, "memory.check.ce.storm.err", ctrl, slot))
        return bool(self.err_msg)


class RuleUceErr(RuleBase):
    """
    UCE：最近30天内，出现过UCE（>=1）
    报错详细信息：控制器（0A）槽位（040）存在UCE错误
    """

    def __init__(self, rule_context):
        super(RuleUceErr, self).__init__(rule_context)

    def execute(self, time_period_day=30, uce_err_count=1):
        time_period_day = int(time_period_day)
        uce_err_count = int(uce_err_count)

        # 首次从30天前分析，后续从上次分析的时间开始分析
        first_start_time = self.get_time_by_delta_process(self.rule_context.current_time, days=-1 * time_period_day)
        start_time = max(first_start_time, self.rule_context.last_err_time)
        sql_str = "SELECT ctrl, slot, count(DISTINCT addr) as addr_num FROM err_addr_tbl " \
                  "WHERE log_time>'{}' AND sn = '{}' AND err_type='{}' " \
                  "GROUP BY ctrl, slot".format(start_time, self.rule_context.sn, HA_UCE_ERR)
        records = self.db_service.err_addr_tbl.query(sql_str)
        self.add_db_record_to_origin_info(records, "uce error check records:")
        return self._check_rule_by_record(records, uce_err_count)

    def _check_rule_by_record(self, records, uce_err_count):
        """
        records示例
        # [{ctrl: "0A", slot: "030", time_count=2},{ctrl: "0A", slot: "031", time_count=1}]
        """
        ctrl_grouped_records = self.group_by(records, 'ctrl')
        self.filter_need_analyzed_records(ctrl_grouped_records)
        for ctrl, one_ctrl_records in ctrl_grouped_records.items():
            slot_grouped_records = self.group_by(one_ctrl_records, "slot")
            for slot, one_slot_records in slot_grouped_records.items():
                if len(one_slot_records) >= uce_err_count:
                    self.err_msg.append(get_msg(self.lang, "memory.check.uce.err", ctrl, slot))
        return bool(self.err_msg)


class VendorRule(RuleBase):
    def execute(self, name="", batch_no="", model=""):
        for item in self.rule_context.vendor_data:
            if all([not name or item.get("vendor", "").lower() == name.lower(),
                    not model or item.get("model", "").lower() in model.lower(),
                    not batch_no or self._is_meet_batch_no(batch_no, item.get("year"), item.get("week"))]):
                return True
        return False

    @staticmethod
    def _is_meet_batch_no(batch_no_config, year, week):
        batch_no_configs = batch_no_config.split(",")
        for batch_config in batch_no_configs:
            match_obj = re.match("(\d+)\((\d+)-(\d+)\)", batch_config)
            if not match_obj:
                continue
            config_year = match_obj.group(1)
            start_week = match_obj.group(2)
            end_week = match_obj.group(3)
            if config_year == year and int(start_week) <= int(week) <= int(end_week):
                return True
        return False
