# -*- coding: UTF-8 -*-
import time
import cliUtil
import common

NEED_RETRY = False
RETRIED = False
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def checkFcPorts(cli):
    '''
    FC端口检查：
        1 如果FC端口不存在表示正常；
        2 Health Status必须为Normal。端口Running Status为Link Down时表示正常，
                    当为Link Up时，工作速率为2000Mbps，4000Mbps，8000Mbps或16000Mbps时表示正常；
        3 其它情况均属于不正常状态。
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    portName = "FC"
    
    try:
        global NEED_RETRY
        global RETRIED
        cmd = "show port general physical_type=FC"
        LOGGER.logExecCmd(cmd)
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        if checkRet[0] != True: 
            LOGGER.logSysAbnormal()
            return checkRet
        
        cliRet = checkRet[1]
        if cliUtil.queryResultWithNoRecord(cliRet):
            return (True, cliRet, errMsg)
        
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = common.getMsg(LANG, "cannot.get.port.info", portName)
            LOGGER.logNoPass("Cannot get information about FC port")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            portId = line.get("ID")
            healthStatus = line.get("Health Status")
            runningStatus = line.get("Running Status")
            workingRate = line.get("Working Rate(Mbps)")

            if healthStatus != common.STATUS_NORMAL:
                flag = False
                errMsg += common.getMsg(LANG, "port.health.status.abnormal", (portName, portId, healthStatus))
                LOGGER.logNoPass("The Health Status of FC port [%s] is abnormal (Health Status:%s)" % 
                    (portId, healthStatus))
            
            if runningStatus == common.STATUS_LINK_UP:
                if workingRate == "--":
                    if RETRIED:
                        flag = False
                        errMsg += common.getMsg(LANG, "cannot.get.port.working.rate", (portName, portId))
                        LOGGER.logNoPass("Cannot get working rate of %s port [%s]" % (portName, portId))
                    else:
                        NEED_RETRY = True
                        break
                elif workingRate not in ["2000", "4000", "8000", "16000"]:
                    flag = False
                    errMsg += common.getMsg(LANG, "port.working.rate.abnormal", (portName, portId, workingRate))
                    LOGGER.logNoPass("The working rate of FC port [%s] is abnormal (Working Rate:%sMbps)" % 
                        (portId, workingRate))
        
        if flag:
            LOGGER.logPass()
            
        return (flag, cliRet, errMsg)

    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cliRet, common.getMsg(LANG, "query.result.abnormal"))

def execute(cli):
    global RETRIED
    checkRet = checkFcPorts(cli)
    if NEED_RETRY:
        time.sleep(common.SLEEP_TIMES)
        RETRIED = True
        checkRet = checkFcPorts(cli)
    return checkRet
