# -*- coding: UTF-8 -*-
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
    SAS端口检查：
        1 Health Status必须为Normal。端口Running Status为Link Down时表示正常，
                   当为Link Up时，端口（不包括下行口EXP）连接必须对称且工作速率为6000Mbps或12000Mbps时表示正常；
        2 其它情况均属于不正常状态。
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    portName = "SAS"
    
    try:
        cmd = "show port general physical_type=SAS"
        LOGGER.logExecCmd(cmd)
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        if checkRet[0] != True: 
            LOGGER.logSysAbnormal()
            return checkRet
        
        cliRet = checkRet[1]
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = common.getMsg(LANG, "cannot.get.port.info", portName)
            LOGGER.logNoPass("Cannot get information about SAS port")
            return (False, cliRet, errMsg) 

        engineNameSet = set()
        normalEngineList = []
        normalPortsList = []
        
        for line in cliRetLinesList: 
            portId = line.get("ID")
            healthStatus = line.get("Health Status") 
            runningStatus = line.get("Running Status")
            workingRate = line.get("Working Rate(Mbps)")
            engName = portId.split(".")[0]
            
            if cliUtil.isEngineId(portId):
                engineNameSet.add(engName)
                
            if healthStatus != common.STATUS_NORMAL:
                flag = False
                errMsg += common.getMsg(LANG, "port.health.status.abnormal", (portName, portId, healthStatus))
                LOGGER.logNoPass("The Health Status of SAS port [%s] is abnormal (Health Status:%s)" % 
                 (portId, healthStatus))
            
            if runningStatus != common.STATUS_LINK_UP:
                if portId in ["ENG0.A0.P0", "ENG0.B0.P0", "CTE0.A0.P0", "CTE0.B0.P0"]:
                    flag = False
                    errMsg += common.getMsg(LANG, "sas.port.running.status.abnormal", (portId, runningStatus))
                    LOGGER.logNoPass("The Running Status of SAS port [%s] is abnormal (Running Status:%s)" % 
                     (portId, runningStatus))
            
            if  healthStatus == common.STATUS_NORMAL and runningStatus == common.STATUS_LINK_UP:
                if cliUtil.isEngineId(portId):
                    normalEngineList.append(portId)
                    
                if (".A" in portId or ".B" in portId) and (".EXP" not in portId):
                    normalPortsList.append(portId)
                    
                if workingRate not in ["6000", "12000"]:
                    flag = False
                    errMsg += common.getMsg(LANG, "port.working.rate.abnormal", (portName, portId, workingRate))
                    LOGGER.logNoPass("The working rate of %s port [%s] is abnormal (Working Rate:%sMbps)" % 
                     (portName, portId, workingRate))
        
        for normalPort in normalPortsList:
            for item in [(".A", ".B"), (".B", ".A")]:
                item0 = item[0]
                item1 = item[1]
                symmetricalPort = normalPort.replace(item0, item1)
                if symmetricalPort not in normalPortsList:
                    flag = False
                    errMsg += common.getMsg(LANG, "sas.port.link.none.symmetrical", (symmetricalPort, normalPort))
                    LOGGER.logNoPass("The SAS port [%s] is improperly connected and is not symmetrical with [%s]" % 
                     (symmetricalPort, normalPort))
        
        storageTypeRet = cliUtil.getStorageType(cli, LANG)
        if storageTypeRet[0] != True:
            errMsg += storageTypeRet[2]
            return (storageTypeRet[0], cliRet, errMsg)
        product_series = storageTypeRet[1]
        
        if product_series == cliUtil.SERIES_18000:
            if len(engineNameSet) == 0:
                flag = False
                errMsg += common.getMsg(LANG, "engine.zero.circle.not.connected")
                LOGGER.logNoPass("Loop 0 of the engine is not connected")
            for engineName in engineNameSet:
                for item in [".A0.P0", ".B0.P0"]:
                    engineId = engineName + item
                    if engineId not in normalEngineList:
                        flag = False
                        errMsg += common.getMsg(LANG, "engine.zero.circle.connect.abnormal", engineId)
                        LOGGER.logNoPass("Loop 0 of the engine must be connected. The SAS port [%s] is disconnected" % (engineId))
        
        if flag:
            LOGGER.logPass()
        
        return (flag, cliRet, errMsg)
                                   
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cliRet, common.getMsg(LANG, "query.result.abnormal"))
