# -*- coding: UTF-8 -*-
import re
import time
import resource
import config
import cliUtil
import os
import logger
import traceback
import threading

STATUS_NORMAL = "Normal"
STATUS_FAULT = "Fault"
STATUS_INTERRUPT = "Interrupt"
STATUS_ONLINE = "Online"
STATUS_RUNNING = "Running"
STATUS_BALANCING = "Balancing"
STATUS_CHARGING = "Charging"
STATUS_DISCHARGING = "Discharging"
STATUS_ENABLED = "Enabled"
STATUS_YES = "Yes"
STATUS_NO = "No"
STATUS_LINK_UP = "Link Up"
STATUS_LINK_DOWN = "Link Down"
STATUS_ON = "On"
STATUS_OFF = "Off"
STATUS_START = "Start"
STATUS_LOW_BATTERY = "Low Battery"

SLEEP_TIMES = 30
PORTS_BIT_ERROR_SPEC = 60
PORTS_BIT_ERROR_INTERVAL = 10
SERIAL_NUMBER_LENGTH = 20
THRESHOLD_BBU_REMAINING_LIFETIME = 90
THRESHOLD_CONTRL_CPU_USAGE = 80
LEVEL2_THRESHOLD_CONTRL_CPU_USAGE = 60

FILE_SUFFIX = "."

def getLang(py_java_env):
    '''
    @summary: 从上下文中获取lang
    @param py_java_env: 上下文对象
    @return: lang
    '''
    return py_java_env.get("lang")
    
def getMsg(lang, msg, args=""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    '''
    return cliUtil.getMsg(lang, msg, args, resource.MESSAGES_DICT)

def getDiskGeneralList(cli, lang):
    '''
    @summary: 获取所有硬盘信息集合
    @param cli: cli对象
    @param lang: 语言lang
    '''
    cmd = "show disk general"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    cliRet = checkRet[1]
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, [], "")
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.disk.info")
        return (False, cliRet, errMsg)
    
    return (True, cliRetLinesList, "")

def getFaultDiskInfo(cli, lang):
    '''
    @summary: 判断是否存在故障硬盘
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg，list)
        flag:
            True： 判断时正常
            False： 判断时异常
        ret: 命令回显
        errMsg: 错误时的消息
        list: 存放故障硬盘的ID号的列表
    '''

    faultDiskCmd = "show disk general |filterRow " \
                    "column=Health\sStatus predict=not predict2=equal_to value=Normal " \
                    "logicOp=or column=Running\sStatus predict=not predict2=equal_to value=Online "

    cliRet = ""
    errMsg = ""
    faultDiskList = []
    diskStatusInfoFlag, diskStatusInfoCliRet, diskStatusInfoErrMsg = cliUtil.excuteCmdInCliMode(cli, faultDiskCmd, True, lang)
    cliRet = diskStatusInfoCliRet

    if diskStatusInfoFlag != True:
        errMsg = diskStatusInfoErrMsg
        return (diskStatusInfoFlag, cliRet, errMsg, [])

    #判断命令回显是否为Command executed successfully
    isCmdExeSucc = cliUtil.queryResultWithNoRecord(diskStatusInfoCliRet)
    if isCmdExeSucc:
        return (True, cliRet, "", [])

    #按逐行字典的方式获取水平表格形式的cli回显集合
    diskInfoDictList = cliUtil.getHorizontalCliRet(diskStatusInfoCliRet)

    for diskInfoDict in diskInfoDictList:
        diskId = diskInfoDict.get('ID')
        healthStatus = diskInfoDict.get('Health Status')
        runningStatus = diskInfoDict.get('Running Status')
        #存在信息就表示有故障盘
        errMsg += getMsg(lang, "disk.status.abnormal.with.status.info", (diskId, healthStatus, runningStatus))
        faultDiskList.append(diskId)

    #存在信息就表示有故障盘
    if faultDiskList:
        return (True, cliRet, errMsg, faultDiskList)

    return (True, cliRet, "", [])

def getDiskSectorSize(cli, lang):
    '''
    @summary: 执行show disk general |filterColumn include columnList=ID,Sector\sSize命令，获取扇区大小
    '''
    sectorSizeDictList = []
    cmd = "show disk general |filterColumn include columnList=ID,Sector\sSize"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    
    #兼容产品命令Bug
    if not cliUtil.hasCliExecPrivilege(checkRet[1]):
        cmd = "show disk general |filterColumn include colunmList=ID,Sector\sSize"
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    
    cliRet = checkRet[1]   
    if checkRet[0] != True: 
        return (checkRet[0], cliRet, checkRet[2], sectorSizeDictList)
    
    sectorSizeDictList = cliUtil.getHorizontalCliRet(cliRet)
    return (True, cliRet, "", sectorSizeDictList)

def is4KDiskWithSectorSize(diskSectorSizeDictList, lang):
    '''
    @summary: 根据扇区大小判断是否为4K盘
    '''
    is4KDiskList = []
    for diskSectorSizeDict in diskSectorSizeDictList:
        diskId = diskSectorSizeDict.get("ID")
        sectorSizeStr = diskSectorSizeDict.get("Sector Size")
        
        #4196B=4.0625KB，显示为4.062KB，需要处理
        if sectorSizeStr == "4.062KB":
            sectorSizeStr = "4.0625KB"
        result, sectorSizeNum = changUnit2B(sectorSizeStr)
        if not result:
            errMsg = getMsg(lang, "change.unit.failed")
            return (False, [], errMsg)
        if sectorSizeNum == 4096.0 or sectorSizeNum == 4160.0:
            is4KDiskList.append(diskId)
        
    if is4KDiskList:
        errMsg = getMsg(lang, "exist.4K.disk", ','.join(is4KDiskList))
        return (False, is4KDiskList, errMsg)
    
    return(True, [], "")

def changUnit2B(strValue):
    '''
    @summary: 根据传入的值转换单位为B
    @return: True/False, sectorSizeNum
    '''
    sectorSizeNum = 0.0
    floatValue = 0.0
    try:
        if not strValue:
            return(False, floatValue)
        
        if re.search("TB" , strValue):
            floatValue = float(strValue.split('T')[0].strip()) * 1024 * 1024 * 1024 * 1024
        elif re.search("GB" , strValue):
            floatValue = float(strValue.split('G')[0].strip()) * 1024 * 1024 * 1024
        elif re.search("MB" , strValue):
            floatValue = float(strValue.split('M')[0].strip()) * 1024 * 1024
        elif re.search("KB" , strValue):
            floatValue = float(strValue.split('K')[0].strip()) * 1024
        elif re.search("B" , strValue):
            floatValue = float(strValue.split('B')[0].strip())
        else:
            return(False, floatValue)

        return(True, floatValue)
    
    except Exception, exception :
        return (False, floatValue)

def hasSSDDisks(enclosureId, diskGeneralList):
    '''
    @summary: 判断框中是否下挂SSD盘
    @param enclosureId: 框id
    @param diskGeneralList: 所有硬盘信息集合
    @return: 
        True: 框中下挂SSD盘
        False: 框中没有下挂SSD盘
    '''
    for line in diskGeneralList:
        diskID = line.get("ID")
        diskType = line.get("Type")
        if enclosureId in diskID and diskType == "SSD":
            return True
    return False

def getEngineSet(cli, lang):
    '''
    @summary: 获取引擎集合（不包含引擎下的所有控制器健康状态和运行状态都不正常）
    @param cli: cli对象
    @param lang: 语言lang
    @return:
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，引擎集合
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show controller general|filterColumn include columnList=Controller,Health\sStatus,Running\sStatus"
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = cliUtil.getVerticalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return (False, cliRet, errMsg)
    
    engineSet = set()
    for retDict in cliRetLinesList:
        ctrlId = retDict.get("Controller")
        healthStatus = retDict.get("Health Status")
        runningStatus = retDict.get("Running Status")
        
        if healthStatus == STATUS_NORMAL and runningStatus == STATUS_ONLINE:
            ctrlId = ctrlId.replace("B", "A")
            ctrlId = ctrlId.replace("D", "C") 
            engineSet.add(ctrlId)
    
    return (True, engineSet, cliRet)

def getPCIeSwitchNum(cli, lang):
    '''
    @summary: 获取PCIe交换机个数
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，PCIe交换机个数
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        cmd = "show enclosure"
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        dataSwitchList = []
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.enclosure.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            logicType = line.get("Logic Type")
            frameId = line.get("ID")
            if not "Data Switch" in logicType:
                continue
            else:
                dataSwitchList.append(frameId)
                
        return (True, len(dataSwitchList), "")
    except:
        errMsg = getMsg(lang, "cannot.get.pcie.switch.info")
        return (False, cliRet, errMsg)
            
def checkPCIePortExists(cli, lang):
    '''
    @summary: 判断是否需要检查PCIe端口
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，需要检查PCIe端口
            flag为False时，不需要检查PCIe端口
        errMsg: 错误消息
    '''
    storageTypeRet = cliUtil.getStorageType(cli, lang)
    if storageTypeRet[0] != True:
        return storageTypeRet
    
    product_series = storageTypeRet[1]
    
    getControllerIdListRet = cliUtil.getControllerIdList(cli, lang)
    if getControllerIdListRet[0] != True:
        return getControllerIdListRet
    
    ctrlNum = len(getControllerIdListRet[1])
    
    if ctrlNum <= 2 or product_series == cliUtil.SERIES_V3:
        return (True, False, "")
    
    return (True, True, "")
        
def getBBUConfig(cli, lang):
    '''
    @summary: 获取单个引擎的BBU规格配置
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，BBU规格
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = cliUtil.getProductModel(cli, lang)
    if getProductModelRet[0] != True:
        return getProductModelRet
    
    pdtModel = getProductModelRet[1]
    
    if config.BBU_OF_ENGINE_CONFIG_DICT.has_key(pdtModel):
        return (True, config.BBU_OF_ENGINE_CONFIG_DICT.get(pdtModel), "")
    else:
        errMsg = getMsg(lang, "cannot.get.bbu.config")
        return (False, "", errMsg)

def getEnclosureFanConfig(cli, enclosureInfo, controllerNum, highDensityDiskEnclosureIdList, lang):
    '''
    @summary: 获取风扇的规格配置
    @param cli: cli对象
    @param enclosureInfo: 机框信息
    @param highDensityDiskEnclosureIdList: 高密框集合
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，风扇的规格
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    enclosureID = enclosureInfo.get("ID")
    enclosureHeight = enclosureInfo.get("Height(U)")
    
    controllers = "--"
    if enclosureHeight == "6":
        if controllerNum <= 2:
            controllers = "2"
        elif controllerNum <= 4:
            controllers = "4"
    
    daeType = "--"
    if enclosureID in highDensityDiskEnclosureIdList:
        daeType = "high-density"
            
    key = ("", "", "", "")
    for enclosureType in ["DSW", "DAE", "ENG", "CTE"]:
        if enclosureType in enclosureID:
            key = (enclosureType, enclosureHeight, controllers, daeType)
    
    if config.FAN_CONFIG_DICT.has_key(key):
        return (True, config.FAN_CONFIG_DICT.get(key), "")

    errMsg = getMsg(lang, "cannot.get.fan.config")
    return (False, "", errMsg)

def checkSystemStatus(cliRet, lang):
    '''
    @summary: 根据回显判断设备是否开工正常，Health Status及Running Status都要为Normal，版本号要正常
    @param cliRet: cli回显
    @param lang: 语言lang
    @return: (flag, errMsg)
        flag:
            True: 系统状态正常
            False： 系统状态异常
        errMsg: 错误消息
    '''
    healthStatus = ""
    runningStatus = ""
    productModel = ""
    productVersion = ""
    
    lineList = cliRet.splitlines()
    for line in lineList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Health Status":
            healthStatus = fieldValue
        elif fieldName == "Running Status":
            runningStatus = fieldValue
        elif fieldName == "Product Model":
            productModel = fieldValue
        elif fieldName == "Product Version":
            productVersion = fieldValue
    
    if len(healthStatus) == 0 or len(runningStatus) == 0:
        return (False, getMsg(lang, "cannot.get.system.info"))
    
    if len(productModel) == 0 or len(productVersion) == 0 or productModel == "--" or productVersion == "--":
        return (False, getMsg(lang, "cannot.get.product.version.info"))
    
    if healthStatus != STATUS_NORMAL:
        return (False, getMsg(lang, "system.health.status.abnormal", healthStatus))
    
    if runningStatus != STATUS_NORMAL:
        return (False, getMsg(lang, "system.running.status.abnormal", runningStatus))
    
    return (True, "")

def getRealCapacity(capacity):
    '''
    @summary: 获取数值
    '''
    matchCapacity = re.search("\d+\.?\d*", capacity)
    if matchCapacity:
        return matchCapacity.group()
    else:
        return ""

def getBaseName(file_path):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param file_path:文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name
    
def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)

def checkPortsBitError(cli, cmd, ports, lang):
    '''
    @summary: 检查所有端口在特定时间段内是否有持续增加的误码
    @param cli: cli对象
    @param cmd: cli命令
    @param ports: 端口列表
    @param lang: 语言对象
    @return:
        flag:
            True: 检查通过
            False: 检查不通过
        cliRetAll: cli回显
        errMsg: 错误时的消息
    '''
    flag = True
    errMsg = ""
    cliRetAll = ""
        
    firstResultDictRet = getPortsBitErrorResultDict(cli, cmd, ports, lang)
    firstCliRet = firstResultDictRet[1]
    if firstResultDictRet[0] != True:
        return (firstResultDictRet[0], firstCliRet, firstResultDictRet[2])
    firstResultDict = firstResultDictRet[3]
    
    retriedTimes = int(PORTS_BIT_ERROR_SPEC / PORTS_BIT_ERROR_INTERVAL)
    for i in range(0, retriedTimes):
        time.sleep(PORTS_BIT_ERROR_INTERVAL)
        nextResultDictRet = getPortsBitErrorResultDict(cli, cmd, ports, lang)
        nextCliRet = nextResultDictRet[1]
        cliRetAll = firstCliRet + nextCliRet
        if nextResultDictRet[0] != True:
            return (nextResultDictRet[0], cliRetAll, nextResultDictRet[2])
        nextResultDict = nextResultDictRet[3]
        
        for portName in ports:
            firstBitErrorList = firstResultDict.get(portName)
            nextbitErrorList = nextResultDict.get(portName)
            
            if len(firstBitErrorList) != len(nextbitErrorList):
                continue
            
            cmpResult = cmpDictList(firstBitErrorList, nextbitErrorList)
            if cmpResult == -1:
                flag = False
                errMsg += getMsg(lang, "port.exists.error.bits", portName)
        
        if not flag:
            return (False, cliRetAll, errMsg)
            
    return (True, cliRetAll, "")
        
def getPortsBitErrorResultDict(cli, cmd, ports, lang):
    '''
    @summary: 获取指定端口列表对应的误码列表
    @param cli: cli对象
    @param cmd: cli命令
    @param ports: 端口列表
    @param lang: 语言对象
    @return: 
        flag: 
            True: 获取成功
            False: 获取失败
        cliRet: cli回显
        errMsg: 错误时的消息
        resultDict: 以字典的形式返回对应端口的误码列表。键对应端口，值对应该端口的误码列表
    '''
    resultDict = {}
    
    checkRet = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, lang)
    cliRet = checkRet[1]
    if checkRet[0] != True:
        return (checkRet[0], cliRet, checkRet[2], None)
    
    for portName in ports:
        getBitErrorResultListRet = getPortBitErrorResultList(cliRet, portName, lang)
        if getBitErrorResultListRet[0] != True:
            return (getBitErrorResultListRet[0], cliRet, getBitErrorResultListRet[2], None)
        
        resultDict.setdefault(portName, getBitErrorResultListRet[1])
    return (True, cliRet, "", resultDict)

def getPortBitErrorResultList(cliRet, portName, lang):
    '''
    @summary: 获取每个端口的误码列表
    @param cliRet: cli回显
    @param portName: 端口名
    @param lang: 语言对象
    @return: 
        flag: 
            True: 获取成功
            False: 获取失败
        resultList: 误码列表
        errMsg: 错误时的消息
    '''
    portInfoRet = cliUtil.getSplitedCliRet(cliRet, getSplitedPortWords(portName))
    if len(portInfoRet) == 0:
        return (True, [], "")
        
    cliRetLinesList = cliUtil.getHorizontalCliRet(portInfoRet)
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.port.info", portName)
        return (False, None, errMsg)
    
    return (True, cliRetLinesList, "")
    
def cmpDictList(firstList, secondList):
    '''
    @summary: 判断误码数是否有增加
    @param firstList: 第一个列表（列表中的内容为字典）
    @param secondList: 第二个列表（列表中的内容为字典，且字典的键与第一个列表中字典的键一致，同时第二个列表的长度与第一个列表的长度一致）
    @return: 
        -1: 第一个列表中的元素小于第二个列表中的元素
         0: 第一个列表与第二个列表相等
         1: 第一个列表中的元素大于第二个列表中的元素
    '''
    for i in range(0, len(firstList)):
        firstDict = firstList[i]
        secondDict = secondList[i]
        
        for key in firstDict.keys():
            firstValue = str(firstDict.get(key))
            secondValue = str(secondDict.get(key))
            
            if firstValue.isdigit() and secondValue.isdigit():
                ret = cmp(long(firstValue), long(secondValue))
                if ret != 0:
                    return ret
    return 0

def getSplitedPortWords(portName):
    '''
    @summary: 根据端口名获取对应的分割字
    @param portName: 端口名
    @return: 分割字
    '''
    if portName == "PCIe":
        portName = "PCIE" 
    return portName + " port:"

def parse_upgradePackage(cli, lang):
    '''
    @summary: 执行show upgrade packge命令，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型,并返回CLI回显
    @param cli: cli对象
    @param lang: 语言lang
    '''    
    cmd = "show upgrade package"
    softwareVersionList = []
    hotPatchVersionList = []

    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return ((flag, cliRet, errMsg),softwareVersionList,hotPatchVersionList)
    
    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = cliUtil.getHorizontalCliRet(cliRet[(softwareVersionIndex):hotPatchVersionIndex])
    hotPatchVersionList = cliUtil.getHorizontalCliRet(cliRet[(hotPatchVersionIndex):])
    
    if len(softwareVersionList) == 0 or len(hotPatchVersionList) == 0:
        errMsg += getMsg(lang, "cannot.get.upgrade.package.info")
        return ((False, cliRet, errMsg) , softwareVersionList , hotPatchVersionList)
        
    for softwareVersion in softwareVersionList:
        if softwareVersion.get("Current Version") == "--":   
            flag = False
            errMsg += getMsg(lang, "cannot.get.contrller.version.info", softwareVersion.get("Name"))
    
    if not isControllerVersionSame(softwareVersionList):
        errMsg += getMsg(lang, "controller.version.not.consistence")
        return ((False, cliRet, errMsg),softwareVersionList,hotPatchVersionList)
    
    return ((flag, cliRet, errMsg),softwareVersionList,hotPatchVersionList)

def isControllerVersionSame(softwareVersionList): 
    '''
    @summary:校验控制器版本是否一致
    @param softwareVersionList: 控制器版本列表
    @return: 
        True:控制器版本一致
        False:控制器软件版本不一致
    '''  
    VersionSet = set(softwareVersion.get("Current Version") for softwareVersion in softwareVersionList)
    if len(VersionSet) !=1:
        return False
    return True

def getCurrentVersion(softwareVersionList, lang):
    '''
    @summary: 通过software信息字典列表获取版本信息
    '''
    currentVersion = ""
    errMsg = ""
    for controller in softwareVersionList:
        currentVersion = controller["Current Version"]
        return (True, currentVersion, "")
    
    errMsg = getMsg(lang, "cannot.get.product.version.info")
    return (False, "", errMsg)

def getCurSysDate(cli):
    '''
    @summary: 获取设备当前时间（年-月-日）
    '''
    cliRet = cli.execCmd("show system general")
    if not cliRet:
        return False, '', ''
    
    for line in cliRet.splitlines():
        #eg:"  Time                : 2016-10-12/11:48:34 +08:00 "
        if line.strip().startswith('Time'):
            sysDate = line.split(':')[1].split('/')[0].strip()
            return True, sysDate, cliRet
    
    return False, '', cliRet


def qryEventDetail(cli, almSn):
    '''
    @summary: 获取告警详情
    '''
    cliRet = cli.execCmd("show event sequence=" + almSn)
    if not cliRet:
        return False, '', ''
    
    for line in cliRet.splitlines():
        if line.split(':')[0].strip() == 'Detail':
            detail = line.split(':')[1].strip()
            return detail, cliRet
    
    return '', cliRet

def getBureauScore(py_java_env):
    '''
    @summary: 从上下文中获取bureauScore分数
    @param py_java_env: 上下文对象
    @return: bureauScore
    '''
    return py_java_env.get("bureauScore")

def refreshProcess(py_java_env, percentNumber, LOGGER):
    """
    @summary: 设置巡检当前进度
    """
    observer = py_java_env.get("progressObserver")
    try:
        if observer != None:
            observer.updateProgress(int(percentNumber))
            LOGGER.logInfo("refreshProcess percentNumber is %s" % percentNumber)
    except:
        LOGGER.logInfo(str(traceback.format_exc()))

#刷新进度公共方法
def threadUpProcess(py_java_env, totalTime, intervalTime, logger):
    try:
        inProcess(py_java_env)
        t = threading.Thread(target=updateProcess, args=(py_java_env, totalTime, intervalTime, logger))
        t.start()
        py_java_env.put("thread", t)
    except:
        return

# 扩容评估检查项8s优化提取添加的更新进度的公共方法    
def updateProcess(py_java_env, totalTime, interval, logger):
    #剩余时间总数
    totalReaminTime = totalTime
    refreshProcess(py_java_env, 1, logger)
    
    #进度平滑处理
    #0~50 interval/50~75 2*interval/75~90 4*interval
    #/90~100 8*interval /99停留直到100
    tmpInterval = interval
    
    #isSet[0] range(50,75)  isSet[1] range(75,90)  isSet[2] range(90,100)
    isSet = [False, False, False]
    
    while py_java_env.get("checkState") == config.PROCESS_STATE_CHECKING:
        #更新进度条
        totalReaminTime -= tmpInterval
        if  totalReaminTime <= 0:
            #最后的剩余时间保持为上一次显示的剩余时间
            totalReaminTime += tmpInterval
            refreshProcess(py_java_env, 99, logger)
            safeSleep(interval)
            continue
        
        currentPro = int(100*(totalTime - totalReaminTime)/totalTime)
        if tmpInterval != 1:
            if not isSet[0] and currentPro in range(50,75):
                tmpInterval = 1 if tmpInterval/2 == 0 else tmpInterval/2
                isSet[0] = True
            elif not isSet[1] and currentPro in range(75,90):
                tmpInterval = 1 if tmpInterval/2 == 0 else tmpInterval/2
                isSet[1] = True
            elif not isSet[2] and currentPro in range(90,100):
                tmpInterval = 1 if tmpInterval/2 == 0 else tmpInterval/2
                isSet[2] = True
        
        logger.logInfo("[UI_thread]totalReaminTime:%s currentPro : %s" % (str(totalReaminTime),str(currentPro)))
        refreshProcess(py_java_env, currentPro, logger)
        
        safeSleep(interval)
    
    if py_java_env.get("checkState") == config.PROCESS_UPGRADE_FINISHED:
        refreshProcess(py_java_env, 100, logger)
    return

# 扩容评估检查项8s优化提取添加的正在刷进度的公共方法    
def inProcess(py_java_env):
    py_java_env.put("checkState", config.PROCESS_STATE_CHECKING)
    return

# 扩容评估检查项8s优化提取添加的完成进度的公共方法    
def finishProcess(py_java_env):
    py_java_env.put("checkState", config.PROCESS_UPGRADE_FINISHED)
    #主线程等待刷进度的线程将剩余时间置于0
    threadJoin(py_java_env)
    return

def threadJoin(py_java_env):
    try:
        if py_java_env.get("thread") != None:
            py_java_env.get("thread").join()
    except:
        return

def safeSleep(seconds):
    '''
    @summary: 安全睡眠时间
    @param param: seconds为睡眠时间，单位：秒；数据类型：整数或小数
    '''    
    try:
        if type(seconds) not in [int,long,float] or seconds <= 0:
            return
        
        startTime = time.clock()
        except_counter = 0
        while True:
            if (time.clock() - startTime) >= seconds:
                return
            
            #睡眠一下，避免长时间占用cpu，该时间设置过长会影响睡眠时间精度
            try:
                time.sleep(0.1)
            except:
                except_counter += 1
        return
    except:
        return
