# -*- coding:utf-8 -*-
import re
import cliUtil
import common

from cbb.frame.rest import restUtil
from cbb.frame.cli import cliUtil as framecliUtil
from cbb.frame.base import baseUtil
from common_utils import get_err_msg
from frameone.util import contextUtil

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

context = contextUtil.getContext(py_java_env)


def execute(cli):
    """
    智能硬盘框级联规格检查
    :param cli:
    :return:
    """
    enclosures_check = EnclosuresCheck(cli, LANG, py_java_env, LOGGER)
    flag, err_msg = enclosures_check.execute_check()
    return flag, "\n".join(enclosures_check.cli_ret_list), err_msg


class EnclosuresCheck:
    def __init__(self, cli, LANG, py_java_env, LOGGER):
        self.cli = cli
        self.LANG = LANG
        self.py_java_env = py_java_env
        self.LOGGER = LOGGER
        self.cli_ret_list = []
        self.product_version = ""

    def execute_check(self):
        try:
            # 设备型号检查
            if self.is_risk_model():
                return True, ""

            # 检测软件版本和补丁
            if self.is_risk_version():
                return True, ""

            # 检查扩容到几控
            ctrl_number = int(self.py_java_env.get("expCtrl"))
            # 扩到4控，则通过，否则继续检查
            if ctrl_number <= 4:
                return True, ""

            # 内双活检測
            if self.check_license():
                return True, ""

            # 检测级联深度
            if self.check_deep_num():
                return False, get_err_msg(
                    self.LANG,
                    "software.enclosure.expansion.depth.not.pass",
                    self.product_version)
            return True, ""

        except common.UnCheckException as e:
            self.LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            err_msg = common.getMsg(self.LANG, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, err_msg

    def is_risk_model(self):
        """
        判断产品型号是否是高级
        :return: True: 检查通过
                 False: 检查不通过
        """
        flag, product_model, cli_ret, err_msg = \
            cliUtil.getProductModelWithCliRet(self.cli, self.LANG)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        self.cli_ret_list.append(cli_ret)

        return not baseUtil.isDoradoV6HighEnd(product_model)

    def is_risk_version(self):
        """
        判断软件版本是否为6.0.1.SPH6以上
        :return: True: 检查通过
                 False: 检查不通过
        """
        flag, software_version, hotpatch_version = \
            framecliUtil.getSystemVersion(self.cli, self.LANG)
        if flag is not True:
            err_msg = common.getMsg(self.LANG,
                                    "cannot.get.product.version.info")
            raise common.UnCheckException(err_msg, "")
        self.product_version = software_version
        if hotpatch_version:
            hotpatch = re.search(r"SPH(\d+)", hotpatch_version)
            self.product_version = software_version + "." + hotpatch_version
            if hotpatch:
                hotpatch_version_num = hotpatch.group(1)
                return all(["6.0.1" in software_version,
                            int(hotpatch_version_num) >= 6])
        return False

    def check_license(self):
        """
        判断是否为内双活，
        :return:True: 检查通过
                 False: 检查不通过
        """
        rest = contextUtil.getRest(context)
        hasLicenseFlag, licenseRec = \
            restUtil.CommonRest.hasInnerLicense(rest)
        # 从上下文中获取是否需要内双活特性的选择
        requireInnerMetro = self.py_java_env.get("requireInnerMetro")
        LOGGER.logInfo("requireInnerMetro: {}".format(requireInnerMetro))

        return all([not hasLicenseFlag, not requireInnerMetro])

    def check_deep_num(self):
        """
        检查级联深度
        :param product_model: 产品型号
        :return: True: 检查不通过
                 False: 检查通过
        """
        cmd = r"show enclosure |filterColumn include " \
              r"columnList=ID,Type,Expansion\sDepth"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.LANG)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        self.cli_ret_list.append(cli_ret)
        dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        deep_num_list = []
        for item in dict_list:
            deep_num = item.get("Expansion Depth", 0)
            deep_num_list.append(int(deep_num))
        return max(deep_num_list) > 1
