# -*- coding: UTF-8 -*-
import cliUtil
import common
import common_cache
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

ITEM = "exp_alua"


def execute(cli):
    """开启ALUA场景下的扩容检查

    :param cli:
    :return:
    """
    allCliRet = list()
    allErrHostIdList = list()

    try:
        context = contextUtil.getContext(py_java_env)
        product_model = contextUtil.getDevType(context)
        product_version = contextUtil.getCurVersion(context)

        flag, cliRet, errMsg, hostInfo = cliUtil.getHostDict(cli, LANG)
        allCliRet.append(cliRet)
        if flag == cliUtil.RESULT_NOCHECK:
            return flag, "\n".join(allCliRet), errMsg

        # 无主机直接通过
        if not hostInfo:
            return True, "\n".join(allCliRet), errMsg

        # 获取主机ID
        hostIdList = hostInfo.keys()
        hostIdList.sort(key=int)

        # 检查FC启动器
        cliRet, errAluaInitList, errHostIdList = getErrInitiatorInfo(
            cli, "FC")
        allCliRet.append(cliRet)
        allRet = common_cache.save_cli_ret_to_file(allCliRet, ITEM,
                                                   py_java_env, LOGGER)
        # 查询异常未检查
        if errAluaInitList is None:
            return (
                cliUtil.RESULT_NOCHECK,
                allRet,
                common.getMsg(LANG, "alua.host.error"),
            )

        # 存在ALUA直接不通过
        if errAluaInitList:
            errMsg = common.getMsg(LANG, "alua.exist.alua.multipathType")
            return False, allRet, errMsg
        # 保存需要确认的主机
        allErrHostIdList.extend(errHostIdList)

        # 检查ISCSI启动器
        cliRet, errAluaInitList, errHostIdList = getErrInitiatorInfo(
            cli, "iSCSI")
        allCliRet.append(cliRet)
        allRet = common_cache.save_cli_ret_to_file(allCliRet, ITEM,
                                                   py_java_env, LOGGER)
        if errAluaInitList is None:
            return (
                cliUtil.RESULT_NOCHECK,
                allRet,
                common.getMsg(LANG, "query.result.abnormal"),
            )

        if errAluaInitList:
            errMsg = common.getMsg(LANG, "alua.exist.alua.multipathType")
            return False, allRet, errMsg
        allErrHostIdList.extend(errHostIdList)

        # DoradoV6和V5 Kunpeng不支持IB
        if all([not baseUtil.isDoradoV6Dev(product_model),
                "Kunpeng" not in product_version]):
            # 检查IB启动器
            cliRet, errAluaInitList, errHostIdList = getErrInitIBInfo(
                cli, hostIdList
            )
            allCliRet.append(cliRet)
            allRet = common_cache.save_cli_ret_to_file(allCliRet, ITEM,
                                                       py_java_env, LOGGER)
            if errAluaInitList is None:
                return (
                    cliUtil.RESULT_NOCHECK,
                    allRet,
                    common.getMsg(LANG, "query.result.abnormal"),
                )
            if errAluaInitList:
                errMsg = common.getMsg(LANG, "alua.exist.alua.multipathType")
                return False, allRet, errMsg
            # 刷新待确认的主机信息
            allErrHostIdList.extend(errHostIdList)

        if allErrHostIdList:
            errMsg = getHostErrMsg(allErrHostIdList, hostInfo)
            return False, allRet, errMsg

        return True, allRet, errMsg

    except Exception as exception:
        LOGGER.logException(exception)
        allRet = common_cache.save_cli_ret_to_file(allCliRet, ITEM,
                                                   py_java_env, LOGGER)
        return (
            cliUtil.RESULT_NOCHECK,
            allRet,
            common.getMsg(LANG, "query.result.abnormal"),
        )


def getHostErrMsg(errHostIdList, hostInfo):
    """ 生成错误消息

    :param errHostIdList:
    :param hostInfo:
    :return:
    """
    errHostIdSet = set(errHostIdList)
    hostList = sorted(errHostIdSet, cmp=None, key=None, reverse=False)
    errMsg = common.getMsg(LANG, "alua.host.error.config")
    for hostId in hostList:
        host = hostInfo.get(hostId)
        if not host:
            continue
        OperatingSystem = host.get("Operating System")
        errMsg += common.getMsg(
            LANG, "alua.host.info", (hostId, OperatingSystem)
        )

    return errMsg


def getErrAluaInfo(initInfoList):
    """ 解析命令回显

    :param initInfoList:
    :return:
    """
    errThirdPartyList = list()
    errAluaInitList = list()
    errHostIdList = list()
    for info in initInfoList:
        multipathType = info.get("Multipath Type", info.get("Multi Path Type"))
        hostId = info.get("Host ID", "--")
        if not hostId or hostId == "--":
            continue

        if "Default" == multipathType:
            continue

        # 同一个版本ALUA和Third-Party只会存在一种
        if "ALUA" == multipathType:
            errAluaInitList.append(info.get("WWN", info.get("iSCSI IQN")))
            return errThirdPartyList, errAluaInitList, errHostIdList

        failoverMode = info.get("Failover Mode")
        if "Third-party" == multipathType and "No ALUA" != failoverMode:
            errThirdPartyList.append(info.get("WWN", info.get("iSCSI IQN")))
            errHostIdList.append(info.get("Host ID", "--"))
            continue

    return errThirdPartyList, errAluaInitList, errHostIdList


def getErrInitiatorInfo(cli, initType):
    """检查FC/iSCSI启动器是否满足ALUA要求

    :param cli:
    :param initType:
    :return:
    """
    cliRet = ""
    errHostIdList = list()
    errAluaInitList = list()

    cmd = "show initiator initiator_type={}".format(initType)
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cliRet += checkRet[1]
    if checkRet[0] is not True:
        LOGGER.logSysAbnormal()
        return cliRet, None, None

    if cliUtil.queryResultWithNoRecord(checkRet[1]):
        return cliRet, errAluaInitList, errHostIdList

    cliRetLinesList = cliUtil.getHorizontalCliRet(checkRet[1])
    errThirdPartyList, errAluaInitList, errHostIdList = \
        getErrAluaInfo(cliRetLinesList)

    return cliRet, errAluaInitList, errHostIdList


def getErrInitIBInfo(cli, hostIdList):
    """检查IB启动器是否满足ALUA要求

    :param cli:
    :param hostIdList:
    :return:
    """
    cliRet = ""
    errHostIdList = []
    errAluaInitList = []
    for hostId in hostIdList:
        cmd = "show ib_initiator general host_id=%s" % hostId
        LOGGER.logExecCmd(cmd)
        checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        cliRet += checkRet[1]

        if not cliUtil.hasCliExecPrivilege(checkRet[1]):
            break

        if checkRet[0] is not True:
            LOGGER.logSysAbnormal()
            return cliRet, None, None

        if cliUtil.queryResultWithNoRecord(checkRet[1]):
            continue

        cliRetLinesList = cliUtil.getHorizontalCliRet(checkRet[1])
        errThirdPartyList, errAluaInitList, _ = getErrAluaInfo(cliRetLinesList)
        # 存在ALUA直接返回不通过
        if errAluaInitList:
            return cliRet, errAluaInitList, errHostIdList

        # 获取手动检查的主机
        if errThirdPartyList:
            errHostIdList.append(hostId)
            continue

    return cliRet, errAluaInitList, errHostIdList
