# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.

import traceback

import cliUtil
import common
import common_utils
import query_hardware_bom_info

import cbb.frame.base.config as config
from cbb.frame.base import baseUtil as cbbBaseUtil
from cbb.frame.rest import restUtil

# noinspection PyUnresolvedReferences
from frameone.util import contextUtil
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException
# noinspection PyUnresolvedReferences
from utils import Products

# noinspection PyUnresolvedReferences
PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)
# 内核版本
CURRENT_KERNEL_VERSION = ""

NVME_ENC = "NVMe"
IP_SAS_ENC = "IP SAS"
SAS_ENC = "SAS"

# 微存储1600
MICRO_1600_DEV = ("OceanDisk 1600", "OceanDisk 1600T")

# QLC型号背板支持C+
QLC_DEV = ("OceanStor 5310 Capacity Flash", "OceanStor 5510 Capacity Flash")

# A800型号,背板只有C+
A800_MODEL = "OceanStor A800"


def execute(cli):
    """
    扩控场景检查控制器和控制框兼容性
    :param cli:
    :return:
    """
    check_item = CheckHardwareType(cli, PY_JAVA_ENV, LANG, LOGGER)
    check_item.start()
    return check_item.get_result()


class CheckHardwareType(object):
    """
    扩控场景检查控制器和控制框兼容性
    """
    result_ret = list()
    result_err_msg = list()
    # 不在硬盘框bom表中的，默认为备件框，提示建议优化
    disk_enc_bom_num = list()

    def __init__(self, cli, py_java_env, lang, logger):
        self.cli = cli
        self.env = py_java_env
        self.lang = lang
        self.logger = logger
        self.result_flag = True
        self.pdt_model = str(py_java_env.get("devInfo").getDeviceType())
        self.pdt_version = str(py_java_env.get("devInfo").getProductVersion())
        self.origin_enc_bom = str(common.get_sn_from_env(py_java_env)).upper()[2:10]
        self.logger.logInfo("pdt_model: {}, pdt_version: {}".format(self.pdt_model, self.pdt_version))

    def start(self):
        """
        执行检查
        :return:
        """
        try:
            self.get_system_info()
            self.result_ret.append("Origin Enclosure Bom:{}".format(self.origin_enc_bom))

            # 检查控制器BOM
            CheckCtrlBom(self.cli, self.env, self.lang, self.logger).execute_check()

            # 检查接口卡BOM
            CheckInterfaceBom(self.cli, self.env, self.lang, self.logger).execute_check()

            # 检查硬盘框BOM
            CheckDiskEnclosureBom(self.cli, self.env, self.lang, self.logger).execute_check()

            if CURRENT_KERNEL_VERSION:
                self.result_ret.append("Current Kernel Version:{}".format(CURRENT_KERNEL_VERSION))
            self.set_result_flag()

        except (Exception, JException) as ex:
            self.logger.logException(ex)
            self.logger.logError("except is: {}".format(traceback.format_exc()))
            self.result_flag, self.result_err_msg = \
                cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, "query.result.abnormal")
            return

    def set_result_flag(self):
        if len(self.result_err_msg) == len(self.disk_enc_bom_num) == 1:
            self.result_err_msg[0] = common.getMsg(self.lang, "new.bom.code.not.right.and.match",
                                                   self.disk_enc_bom_num[0])
            self.result_flag = cliUtil.RESULT_WARNING
        elif self.result_err_msg:
            self.result_flag = False

    def check_version(self, required_software_version, required_kernel_version):
        """
        检查软件版本
        @param required_software_version: 需要的软件版本
        @param required_kernel_version:需要的内核版本版本
        @return: True：通过， False：不通过
        """
        global CURRENT_KERNEL_VERSION
        software_check_flag = True
        kernel_check_flag = True
        # 检查软件版本
        if Products.compareVersion(self.pdt_version, required_software_version) < 0:
            software_check_flag = False
        # 检查内核版本
        if not required_kernel_version:
            return software_check_flag, kernel_check_flag
        if not CURRENT_KERNEL_VERSION:
            CURRENT_KERNEL_VERSION = self.query_current_kernel_version()
        if Products.compareVersion(CURRENT_KERNEL_VERSION, required_kernel_version) < 0:
            kernel_check_flag = False
        return software_check_flag, kernel_check_flag

    def query_current_kernel_version(self):
        kernel_version = "1.0.0.1"
        if any([cbbBaseUtil.is_micro_dev(self.pdt_model),
                cbbBaseUtil.is_ocean_protect(self.pdt_model),
                Products.compareVersion(self.pdt_version, "6.1.0") >= 0]):
            flag, ret, kernel_version = cliUtil.get_kernel_version(self.cli, self.lang)
            if not flag:
                raise Exception("Failed to get kernel version.")
        self.logger.logInfo("kernel_version:{}".format(kernel_version))
        return kernel_version

    def has_inner_hyper_metro(self):
        # 查询是否有内双活license
        try:
            rest = contextUtil.getRest(contextUtil.getContext(self.env))
            has_license_flag, _ = restUtil.CommonRest.hasInnerLicense(rest)
            return has_license_flag
        except (JException, Exception) as ex:
            self.logger.logError("Query license exception: {}".format(ex))
            return False

    def get_system_info(self):
        """
        获取系统信息回显，用于界面展示
        """
        cmd = "show system general"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.result_ret.append(ret)
        if flag is not True:
            raise Exception("Failed to get system info.")

    def get_result(self):
        """ 获取检查结果

        :return:
        """
        return (self.result_flag, "\n".join(self.result_ret),
                "\n".join(self.result_err_msg))


class CheckCtrlBom(CheckHardwareType):
    """
    检查控制器BOM编码
    """

    def __init__(self, cli, py_java_env, lang, logger):
        super(CheckCtrlBom, self).__init__(cli, py_java_env, lang, logger)
        self.wrong_ctrl_bom = list()
        self.ctrl_bom_info_dict = query_hardware_bom_info.QueryCtrlBomInfo(logger).get_bom_info()

    def execute_check(self):
        new_ctrl_bom_list = str(self.env.get("newCtrlBom")).upper().split(',')
        self.result_ret.append("New Control Bom List:{}".format(self.env.get("newCtrlBom").upper()))
        ctrl_bom_list = remove_error_input(new_ctrl_bom_list)
        new_enc_bom_list = str(self.env.get("newEncBom")).upper().split(',')
        self.result_ret.append("New Enclosure Bom List:{}".format(self.env.get("newEncBom").upper()))
        enc_bom_list = remove_error_input(new_enc_bom_list)

        for new_ctrl_bom in ctrl_bom_list:
            self.check_new_bom(new_ctrl_bom)
        for new_enc_bom in enc_bom_list:
            self.check_new_bom(new_enc_bom, is_ctrl_module=False)

        if self.wrong_ctrl_bom:
            self.result_err_msg.append(common.getMsg(
                self.lang, "bom.code.not.match", (",".join(self.wrong_ctrl_bom), self.origin_enc_bom)))

    def check_new_bom(self, new_bom, is_ctrl_module=True):
        """
        检查新的BOM编码
        @param new_bom: BOM编码
        @param is_ctrl_module: 是否为控制器
        """
        # 原BOM不在列表中，直接通过
        if self.origin_enc_bom not in self.ctrl_bom_info_dict:
            return
        # 新BOM不在列表中，则不配套
        if new_bom not in self.ctrl_bom_info_dict:
            self.result_err_msg.append(common.getMsg(
                self.lang, "bom.code.not.right", (new_bom, self.origin_enc_bom)))
            return
        new_bom_info = self.ctrl_bom_info_dict.get(new_bom)
        origin_bom_info = self.ctrl_bom_info_dict.get(self.origin_enc_bom)
        # 控制器BOM输成控制框BOM，或者控制框BOM输成控制器BOM，不通过
        if any([is_ctrl_module and "ctrl enc" in new_bom_info[2],
                not is_ctrl_module and "ctrl module" in new_bom_info[2]]):
            self.wrong_ctrl_bom.append(new_bom)
            return
        # BOM编码不支持该型号, 单控内存不一致
        if self.pdt_model not in new_bom_info[0].split(";") or new_bom_info[1] != origin_bom_info[1]:
            self.wrong_ctrl_bom.append(new_bom)
            return
        # 微存储和二级存储 全闪和混闪不能互扩
        if cbbBaseUtil.is_micro_dev(self.pdt_model) or cbbBaseUtil.is_ocean_protect(self.pdt_model):
            if new_bom_info[7] != origin_bom_info[7]:
                self.wrong_ctrl_bom.append(new_bom)
                return
        # 背板类型不一致
        if not self.check_back_model(new_bom_info, is_ctrl_module):
            self.wrong_ctrl_bom.append(new_bom)
            return
        # 检查软件代次和CPU
        if not self.check_software_generation_and_cpu(new_bom_info):
            self.wrong_ctrl_bom.append(new_bom)
            return

        # 高端4P扩2P不支持内双活
        if origin_bom_info[5] == "4P" and new_bom_info[5] == "2P" and self.has_inner_hyper_metro():
            self.wrong_ctrl_bom.append(new_bom)
            return

        self.check_software_kernel_version(new_bom, new_bom_info, origin_bom_info)

    def check_software_kernel_version(self, new_bom, new_bom_info, origin_bom_info):
        # 检查版本
        required_version_list = new_bom_info[6].split(";")
        required_software_version = required_version_list[0]
        required_kernel_version = "" if len(required_version_list) < 2 else required_version_list[1]
        # Dorado 5000/5500 3P扩1P和高端4P扩2P，软件版本不低于6.1.5RC1，内核版本不低于1.1.5.0
        if self.is_need_check_version_615(origin_bom_info, new_bom_info):
            required_software_version = "6.1.5RC1"
            required_kernel_version = "1.1.5.0"
        software_check_flag, kernel_check_flag = self.check_version(required_software_version, required_kernel_version)
        if not software_check_flag:
            self.result_err_msg.append(common.getMsg(
                self.lang, "soft.version.not.match", (new_bom, self.pdt_version, required_software_version)))
        if not kernel_check_flag:
            self.result_err_msg.append(common.getMsg(
                self.lang, "kernel.version.not.match", (new_bom, CURRENT_KERNEL_VERSION, required_kernel_version)))

    def is_need_check_version_615(self, origin_bom_info, new_bom_info):
        if origin_bom_info[5] == "4P" and new_bom_info[5] == "2P":
            return True
        if all([origin_bom_info[5] == "3P" and new_bom_info[5] == "1P",
                self.pdt_model in ("OceanStor Dorado 5000 V6", "OceanStor Dorado 5500 V6")]):
            return True
        return False

    def check_back_model(self, new_bom_info, is_ctrl_module):
        """
        检查背板类型
        @param new_bom_info: 新BOM编码信息
        @param is_ctrl_module: 是否为控制器编码
        @return: True：检查通过， False: 检查不通过
        """
        # 控制器BOM不检查背板类型
        if is_ctrl_module:
            return True
        origin_bom_info = self.ctrl_bom_info_dict.get(self.origin_enc_bom)
        new_back_model = new_bom_info[3].split("&")
        origin_back_model = origin_bom_info[3].split("&")
        # 针对新融合，sas只能和sas扩，nvme只能和nvme扩，sas&nvme只能和sas&nvme扩
        if cbbBaseUtil.is_new_oceanstor(self.pdt_model):
            return new_bom_info[3].strip() == origin_bom_info[3].strip()
        return any(["SAS" in new_back_model and "SAS" in origin_back_model,
                    "NVMe" in new_back_model and "NVMe" in origin_back_model,
                    "SAS RDMA" in new_back_model and "SAS RDMA" in origin_back_model])

    def check_software_generation_and_cpu(self, new_bom_info):
        """
        检查软件代次和cpu
        @param new_bom_info: 新BOM信息
        """
        # 控制器BOM
        if "ctrl module" in new_bom_info[2]:
            return self.check_new_ctrl_generation_and_cpu(new_bom_info)
        return self.check_new_enc_generation_and_cpu(new_bom_info)

    def check_new_ctrl_generation_and_cpu(self, new_bom_info):
        """
        检查控制器BOM的软件代次和CPU
        @param new_bom_info: 新控制器BOM信息
        """
        origin_bom_info = self.ctrl_bom_info_dict.get(self.origin_enc_bom)
        # 软件代次（C或C+）C+扩C不通过
        if origin_bom_info[4].upper() == "CPLUS" and new_bom_info[4].upper() == "C":
            return False
        # CPU数量（1P/2P/3P/4P）相等
        return new_bom_info[5] == origin_bom_info[5]

    def check_new_enc_generation_and_cpu(self, new_bom_info):
        """
        检查控制框BOM的软件代次和CPU
        @param new_bom_info: 新控制框BOM信息
        """
        origin_bom_info = self.ctrl_bom_info_dict.get(self.origin_enc_bom)
        # 软件代次（C或C+）C+扩C不通过
        if origin_bom_info[4].upper() == "CPLUS" and new_bom_info[4].upper() == "C":
            return False
        # CPU数量（1P/2P/3P/4P）不同类型只支持3P扩1P和4P扩2P
        if all([new_bom_info[5] != origin_bom_info[5],
                origin_bom_info[5] != "3P" and new_bom_info[5] != "1P",
                origin_bom_info[5] != "4P" and new_bom_info[5] != "2P"]):
            return False
        return True


class CheckInterfaceBom(CheckHardwareType):
    """
    检查接口卡BOM编码
    """

    def __init__(self, cli, py_java_env, lang, logger):
        super(CheckInterfaceBom, self).__init__(cli, py_java_env, lang, logger)
        # 不配套的BOM编码
        self.wrong_interface_bom = list()
        self.software_generation = ""
        self.interface_bom_info_dict = {}

    def execute_check(self):
        new_interface_bom_list = str(self.env.get("newInterfaceBom")).upper().split(',')
        interface_bom_list = remove_error_input(new_interface_bom_list)
        if not interface_bom_list:
            self.logger.logInfo("no new interface bom!")
            return
        self.result_ret.append("New Interface Bom List:{}".format(self.env.get("newInterfaceBom").upper()))
        self.interface_bom_info_dict = query_hardware_bom_info.QueryInterfaceBomInfo(self.logger).get_bom_info()
        for bom_code in interface_bom_list:
            if bom_code not in self.interface_bom_info_dict:
                self.result_err_msg.append(common.getMsg(
                    self.lang, "new.interface.bom.code.not.right", bom_code))
                continue
            self.check_one_interface_bom(bom_code)
        if self.wrong_interface_bom:
            self.result_err_msg.append(common.getMsg(
                self.lang, "new.interface.bom.code.not.match", ",".join(self.wrong_interface_bom)))

    def check_one_interface_bom(self, bom_code):
        product_info_list = self.interface_bom_info_dict.get(bom_code)
        required_version = ""
        self.software_generation = get_origin_software_generation(self.pdt_model)
        is_model_and_generation_match = False
        for product_info in product_info_list:
            # 产品型号匹配，去代次和带V6：一致；其他型号要求相等：例如OceanDisk 1500 和 OceanDisk 1500T不同
            if self.pdt_model != product_info[0] and self.pdt_model + " V6" != product_info[0]:
                continue
            # 原集群框软件代次和待扩框软件代次要相同
            if self.software_generation != product_info[3]:
                continue
            # C+框不能扩C卡
            if not (self.software_generation == "C+" and product_info[1] == "C"):
                required_version = product_info[2]
                is_model_and_generation_match = True

        if not is_model_and_generation_match:
            self.wrong_interface_bom.append(bom_code)
            return
        # 检查版本
        required_version_list = required_version.split(";")
        required_software_version = required_version_list[0]
        required_kernel_version = "" if len(required_version_list) < 2 else required_version_list[1]
        software_check_flag, kernel_check_flag = self.check_version(required_software_version, required_kernel_version)
        if not software_check_flag:
            self.result_err_msg.append(common.getMsg(self.lang, "new.interface.soft.version.not.match",
                                                     (bom_code, self.pdt_version, required_software_version)))
        if not kernel_check_flag:
            self.result_err_msg.append(common.getMsg(self.lang, "new.interface.kernel.version.not.match",
                                                     (bom_code, CURRENT_KERNEL_VERSION, required_kernel_version)))


class CheckDiskEnclosureBom(CheckHardwareType):
    """
    检查硬盘框BOM编码
    """

    def __init__(self, cli, py_java_env, lang, logger):
        super(CheckDiskEnclosureBom, self).__init__(cli, py_java_env, lang, logger)
        self.required_software_version = "6.1.0"
        self.required_kernel_version = "1.1.1.0"
        # 不配套的BOM编码
        self.wrong_bom = list()
        self.software_generation = ""
        self.bom_info_dict = {}

    def execute_check(self):
        new_bom_list = str(self.env.get("newDiskEnclosureBom")).upper().split(",")
        bom_list = remove_error_input(new_bom_list)
        if not bom_list:
            self.logger.logInfo("no new disk enclosure bom!")
            return
        self.bom_info_dict = query_hardware_bom_info.QueryDiskEnclosureBomInfo(self.logger).get_bom_info()
        self.result_ret.append("New Disk Enclosure Bom List:{}".format(self.env.get("newDiskEnclosureBom").upper()))
        self.logger.logInfo("New enclosure Bom List:{}".format(self.env.get("newDiskEnclosureBom").upper()))
        for bom_code in bom_list:
            self.logger.logInfo("start check bom_code:{}".format(bom_code))
            if bom_code not in self.bom_info_dict:
                self.disk_enc_bom_num.append(bom_code)
                self.result_err_msg.append(common.getMsg(self.lang, "new.enc.bom.code.not.right", bom_code))
                continue
            self.check_one_bom(bom_code)

    def check_one_bom(self, bom_code):
        """
        1. 常规SAS框
        2. 查询内部型号，根据BOM确定硬盘框类型是否配套。
        3. C IP硬盘框如果配套的C的控制框通过，C IP硬盘框如果配套C+高端直接通过；否则不通过。
        4. C+ IP硬盘框 和 C控制框 版本6.1.0 内核版本1.1.1.0以上。
        :param bom_code: 输入的bom code
        :return:
        """
        if not self.is_inner_type_support(bom_code):
            self.logger.logInfo("inner type is not supported")
            self.result_err_msg.append(common.getMsg(self.lang, "new.enc.bom.code.not.match", bom_code))
            return False
        # 普通SAS框检查通过，不需要检查C/C+配套
        if self._is_common_sas_enclosure(bom_code):
            return True

        # 其他框（除SAS框）类型的软件代次要么是C，要么是C+，不会有C和C+共用的情况，如果有，则软件代次的判断逻辑需要修改
        return self.is_software_generation_right(bom_code)

    def is_software_generation_right(self, bom_code):
        software_generation = self.bom_info_dict.get(bom_code)[1]
        origin_software_generation = get_origin_software_generation(self.pdt_model)
        self.logger.logInfo("software generation new:{}, origin:{}, kenner version:{}, pdt_version:{}".format(
            software_generation, origin_software_generation, CURRENT_KERNEL_VERSION, self.pdt_version))
        if "C" == software_generation and "C" == origin_software_generation:
            return True
        if "C" == software_generation and "C+" == origin_software_generation:
            if cbbBaseUtil.isDoradoV6HighEnd(self.pdt_model):
                return True
            self.result_err_msg.append(
                common.getMsg(self.lang, "mid.product.not.support.onboard.rdma.backend", bom_code)
            )
            return False
        if "C+" == software_generation and "C+" == origin_software_generation:
            return True
        # 二级存储，新融合、微存储，C扩C+框，没有版本限制
        if any([cbbBaseUtil.is_micro_dev(self.pdt_model),
                cbbBaseUtil.is_ocean_protect(self.pdt_model),
                cbbBaseUtil.is_new_oceanstor(self.pdt_model)]):
            return True
        if "C+" == software_generation and "C" == origin_software_generation:
            software_check_flag, kernel_check_flag = self.check_version(
                self.required_software_version, self.required_kernel_version
            )
            if not software_check_flag:
                self.result_err_msg.append(
                    common.getMsg(
                        self.lang,
                        "new.enc.soft.version.not.match",
                        (bom_code, self.pdt_version, self.required_software_version),
                    )
                )
            if not kernel_check_flag:
                self.result_err_msg.append(
                    common.getMsg(
                        self.lang,
                        "new.enc.kernel.version.not.match",
                        (bom_code, CURRENT_KERNEL_VERSION, self.required_kernel_version),
                    )
                )
            return software_check_flag and kernel_check_flag
        return True

    def check_inner_type(self, new_type):
        msg, ctrl_inner_model, ret = common_utils.get_internal_product_model(self.env)
        self.logger.logInfo("inner model: {}, new: {}".format(ctrl_inner_model, new_type))
        # 微存储全闪支持NVMe,混闪1500，只支持SAS，混闪1600支持SAS和NVMe
        if cbbBaseUtil.is_micro_dev(self.pdt_model):
            return self.get_micro_dev_inner_type(ctrl_inner_model, new_type)

        origin_type = SAS_ENC
        if not ctrl_inner_model:
            return SAS_ENC == new_type
        # 180500K_N支持NVMe和IP SAS
        if "18500K_N" in ctrl_inner_model:
            return new_type in [NVME_ENC, IP_SAS_ENC]
        # 新融合内部型号带_M。支持SAS和 NVMe
        if "_M" in ctrl_inner_model:
            return new_type in [SAS_ENC, NVME_ENC]
        if "_N" in ctrl_inner_model:
            origin_type = NVME_ENC
        if "_I" in ctrl_inner_model:
            origin_type = IP_SAS_ENC
        if "_Q" in ctrl_inner_model:
            origin_type = NVME_ENC
        return origin_type == new_type

    def get_micro_dev_inner_type(self, ctrl_inner_model, new_type):
        # 微存储全闪支持NVMe,混闪1500，只支持SAS，混闪1600支持SAS和NVMe
        if not cbbBaseUtil.is_micro_flashing_dev(ctrl_inner_model):
            return NVME_ENC == new_type
        return new_type in [SAS_ENC, NVME_ENC] if self.pdt_model in MICRO_1600_DEV else SAS_ENC == new_type

    def is_inner_type_support(self, enc_bom):
        """
        查询内部型号，根据BOM确定硬盘框类型是否配套当前存储。
        :param enc_bom:
        :return:
        """
        new_type = self.bom_info_dict.get(enc_bom)[0]
        return self.check_inner_type(new_type)

    def _is_common_sas_enclosure(self, enc_bom):
        return self.bom_info_dict.get(enc_bom) and self.bom_info_dict.get(enc_bom)[0] == SAS_ENC


def get_origin_software_generation(pdt_model):
    """
    获取原集群软件代次
    @param pdt_model: 设备型号
    @return: 原集群软件代次
    """
    # 微存储1600都是C，其余微存储为C+
    if cbbBaseUtil.is_micro_dev(pdt_model):
        return "C" if pdt_model in MICRO_1600_DEV else "C+"
    if pdt_model in config.HYBRID_V6_ENTRY_LEVEL_END:
        return "C+"
    # E8000为C+设备
    if any(pdt_model in model_list for model_list in [QLC_DEV, config.OCEAN_PROTECT_SUPPORT_SWITCH, (A800_MODEL,)]):
        return "C+"
    exp_info = PY_JAVA_ENV.get("expInfo")
    # 内部型号
    internal_product_model = exp_info.getInternalModel()
    LOGGER.logInfo("internal_product_model is: {}".format(internal_product_model))
    # 内部型号中存在"_C"，则为C+
    return "C+" if "_C" in internal_product_model else "C"


def remove_error_input(bom_list):
    if bom_list[-1] == "":
        bom_list.pop()
    return bom_list
