# -*- coding: UTF-8 -*-
import traceback

import cliUtil
import common
import cli_util_cache as cache_util
from common_utils import get_err_msg
import re

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    检查系统中的LUN的创建容量是否大于32TB
    步骤1 以admin用户登录设备；
    步骤2 执行命令：show upgrade package，获取系统软件版本和热补丁版本信息；
    步骤3 执行命令：show lun general，查询当前阵列上lun的创建容量（Capacity）。

    1.若步骤2中系统软件版本为6.0.1,且热补丁版本低于6.0.1.SPH5，
    则继续检查，否则检查结果为通过；
    2.若步骤3中任意一个lun的Capacity的数值大于或者等于31.875TB，
    则检查不通过，否则检查结果为通过。

    :param cli:
    :return:
    """
    smart_fc_check = LunCapacityCheck(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = smart_fc_check.execute_check()
    return flag, "\n".join(smart_fc_check.all_ret_list), msg


class LunCapacityCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []

    def execute_check(self):
        try:

            (
                flag,
                soft_version,
                patch_version,
                cli_ret,
                err_msg,
            ) = common.get_soft_and_patch_version(self.cli, LOGGER, LANG)
            self.all_ret_list.append(cli_ret)
            # 检查是否为风险版本
            if self.check_risk_version(soft_version, patch_version):
                return True, ""
            err_msg_id_list = self.get_32T_lun_id()
            if not err_msg_id_list:
                return True, ""

            return (
                False,
                get_err_msg(
                    self.lang,
                    "cli.32Tlun.check.not.pass",
                    (", ".join(err_msg_id_list)),
                ),
            )
        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            self.all_ret_list.append(e.cliRet)
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def check_risk_version(self, soft_version, patch_version):
        """
        检查是否为风险版本
        :return: 有风险的为False
        """
        if "6.0.1" not in soft_version:
            return True

        pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
        match_hot_path = pattern_hot_patch.search(patch_version)
        if match_hot_path and int(match_hot_path.group(1)) >= 5:
            return True
        return False

    def get_32T_lun_id(self):
        """获取lun容量大于或者等于31.875TB
        :return: 大于或者等于31.875TB 的lunID
        """
        not_pass_id_list = list()
        lun_dict_list, cli_ret = cache_util.get_lun_general_cache(
            self.env, self.cli, self.logger
        )

        self.all_ret_list.append(cli_ret)
        for lun_dict in lun_dict_list:
            lun_id = lun_dict.get("ID", "--")
            lun_capacity = lun_dict.get("Capacity", "")
            flag, capacity = common.changUnit2GBDecimal(lun_capacity)
            if capacity >= 32640:
                not_pass_id_list.append(lun_id)

        self.logger.logInfo(
            "The following LUN capacity is greater than 31.875TB {}".format(
                str(not_pass_id_list)
            )
        )
        return not_pass_id_list
