# -*- coding:utf-8 -*-
import traceback

import cliUtil
import common
from common_utils import get_err_msg
from common_utils import is_expansion_scene
from cli_util_cache import (
    get_vertical_cache,
    get_horizontal_no_standard_cache,
)
from common import UnCheckException
from storage_obj_constant import StoragePoolAttribute as PoolAttr

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
START_CHECK_VERSION_V3 = "V300R003C20SPC200"
END_CHECK_VERSION_V3 = "V300R006C61"
START_CHECK_VERSION_V5 = "V500R007C00"
END_CHECK_VERSION_V5 = "V500R007C70"
SAFETY_DISK_AMOUNT = 13

# pool中各层级名称对应到domain中的层级名称.
NAME_MAP_TIER_DISK = {
    PoolAttr.TIER_NAME_EXTREME_PERFORMANCE: PoolAttr.TIER_ZERO_DISK_NUMBER,
    PoolAttr.TIER_NAME_PERFORMANCE: PoolAttr.TIER_ONE_DISK_NUMBER,
    PoolAttr.TIER_NAME_CAPACITY: PoolAttr.TIER_TWO_DISK_NUMBER,
}
NAME_MAP_TIER_SPARE_STRATEGY = {
    PoolAttr.TIER_NAME_EXTREME_PERFORMANCE: PoolAttr.TIER_ZERO_HOT_SPARE_STRATEGY,
    PoolAttr.TIER_NAME_PERFORMANCE: PoolAttr.TIER_ONE_HOT_SPARE_STRATEGY,
    PoolAttr.TIER_NAME_CAPACITY: PoolAttr.TIER_TWO_HOT_SPARE_STRATEGY,
}


def execute(cli):
    check_disk_domain_raid10_pool = CheckDiskDomainRaid10Pool(
        cli, LANG, PY_JAVA_ENV, LOGGER
    )
    flag, msg = check_disk_domain_raid10_pool.execute_check()
    return flag, "\n".join(check_disk_domain_raid10_pool.all_ret_list), msg


class CheckDiskDomainRaid10Pool:
    """
    硬盘域RAID10存储池检查。
    """

    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.conn_cli = ""
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.safety_disks = []
        self.software_version = ""
        self.hot_patch_version = ""
        self.name_map_tier_spare_strategy = NAME_MAP_TIER_SPARE_STRATEGY
        self.name_map_tier_disk = NAME_MAP_TIER_DISK

    def execute_check(self):
        try:
            # 因扩容场景无法评估在哪个存储池上，所以直接通过。
            if is_expansion_scene(self.env, self.logger):
                self.all_ret_list.append("Expansion scene check pass!")
                return True, ""
            # 查询到的版本不在名单内检查项直接通过
            if not self.is_risk_version():
                return True, ""
            flag, msg = self.risk_judgment()
            return flag, "".join(msg)
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def risk_judgment(self):
        """
        2 若步骤4不存在RAID LEVEL为RAID10的存储池，则检查结果为通过，否则继续检查；
        3 若步骤4存在RAID10的存储池，在步骤5中所对应层级的热备策略不为NONE，则检查结果为通过，否则继续检查；
        4 若步骤5中RAID10的存储池所对应层级的热备策略为NONE，在步骤6中对应层级的硬盘数量大于12块盘，则检查结果为通过，否则继续检查；
        5 若步骤7记录的硬盘域成员盘包含步骤8的保险箱盘，则检查结果为不通过，否则检查结果为通过。
        :return:
        """
        pool_disk_domain_id = self.get_pool_and_disk_domain_id()
        if not pool_disk_domain_id:
            return True, ""
        disk_domain_id_map_pool = self.create_disk_domain_id_map_pool(
            pool_disk_domain_id
        )
        error_message_list = []
        for disk_domain_id, pool_id_list in disk_domain_id_map_pool.items():
            risk_pool_ids_info_list = self.get_risk_ids_info_list(pool_id_list)
            tier_status = self.get_hot_spare_strategy(disk_domain_id)
            hot_spare_strategy_is_risk = self.judge_hot_spare_strategy(
                risk_pool_ids_info_list, tier_status
            )
            if not hot_spare_strategy_is_risk:
                continue
            disk_amount_is_risk = []
            self.get_risk_disk_amount(
                disk_amount_is_risk, disk_domain_id, hot_spare_strategy_is_risk
            )
            if not disk_amount_is_risk:
                continue
            member_disk_id = self.get_member_disk_id(disk_domain_id)
            if not self.safety_disks:
                self.get_safety_disk_id()
            risk_flag = False
            for i in member_disk_id:
                if i in self.safety_disks:
                    risk_flag = True
                    break
            if risk_flag:
                self.get_error_message_list(
                    disk_amount_is_risk, disk_domain_id, error_message_list
                )
        if not error_message_list:
            return True, error_message_list
        return False, error_message_list

    def get_error_message_list(
            self, disk_amount_is_risk, disk_domain_id, error_message_list
    ):
        """
        错误信息列表
        :param disk_amount_is_risk:
        :param disk_domain_id:
        :param error_message_list:
        :return:
        """
        err_key = "disk.domain.raid10.pool.check"
        for i in disk_amount_is_risk:
            error_message = get_err_msg(
                self.lang,
                err_key,
                (
                    disk_domain_id,
                    i.get("Name"),
                    i.get("Disk Number"),
                ),
            )
            if error_message not in error_message_list:
                error_message_list.append(error_message)

    def get_risk_disk_amount(
            self, disk_amount_is_risk, disk_domain_id,
            hot_spare_strategy_is_risk
    ):
        """
        将判定有pool id对应的硬盘域的成员盘数量更新到pool id对象字典里面
        :param disk_amount_is_risk:
        :param disk_domain_id:
        :param hot_spare_strategy_is_risk:
        :return:
        """
        tiers_disk_amount = self.get_tier_disk_amount(disk_domain_id)
        for pool_info in hot_spare_strategy_is_risk:
            flag, disk_amount = self.is_risk_disk_amount(
                pool_info, tiers_disk_amount
            )
            if flag:
                pool_info["Disk Number"] = str(disk_amount)
                disk_amount_is_risk.append(pool_info)

    def judge_hot_spare_strategy(self, risk_pool_ids_info_list, tier_status):
        """
        将有热备策略有风险的添加到hot_spare_strategy_is_risk列表
        :param risk_pool_ids_info_list:
        :param tier_status:
        :return:
        """
        hot_spare_strategy_is_risk = []
        for pool_id_info in risk_pool_ids_info_list:
            flag = self.is_risk_hot_spare_strategy(pool_id_info, tier_status)
            if flag:
                hot_spare_strategy_is_risk.append(pool_id_info)
        return hot_spare_strategy_is_risk

    def get_risk_ids_info_list(self, pool_id_list):
        """
        将有风险的pool info添加到risk_pool_ids_info_list
        :param pool_id_list:
        :return:
        """
        risk_pool_ids_info_list = []
        for i in pool_id_list:
            risk_pool_info = self.get_risk_pool_info(i)
            if risk_pool_info:
                risk_pool_ids_info_list.append(risk_pool_info)
        return risk_pool_ids_info_list

    def is_risk_disk_amount(self, risk_pool_info, tiers_disk_amount):
        """
        判断成员盘数量是否小于等于13块，若小于则有风险
        :param risk_pool_info:
        :param tiers_disk_amount:
        :return:
        """
        name = risk_pool_info.get("Name")
        tier = self.name_map_tier_disk.get(name)
        tiers_disk_amount = int(tiers_disk_amount.get(tier))
        if tiers_disk_amount < SAFETY_DISK_AMOUNT:
            return True, str(tiers_disk_amount)
        else:
            return False, ""

    def is_risk_hot_spare_strategy(self, risk_pool_info, tier_statues):
        """
        判断层tier对应的热备策略是否为NONE是则有风险
        :param risk_pool_info:
        :param tier_statues:
        :return:
        """
        name = risk_pool_info.get("Name")
        tier = self.name_map_tier_spare_strategy.get(name)
        tier_statue = tier_statues.get(tier)
        if tier_statue.upper() == "NONE":
            return True
        else:
            return False

    def get_pool_and_disk_domain_id(self):
        """
        步骤3:
        执行show storage_pool general，查询出ID和Disk Domain ID
        :return:list
        """
        cmd = "show storage_pool general"
        flag, ret, pool_info_list, msg = get_horizontal_no_standard_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        return pool_info_list

    def create_disk_domain_id_map_pool(self, pool_and_disk_domain_id):
        """
        一个disk_domain_id可能对应多个pool_id,为避免重复查询便以disk_domain_id为
        key,pool_id添加到其对应的disk_domain_id的列表里面
        :param pool_and_disk_domain_id:
        :return:{disk_domain_id：[pool_id,...],...}
        """
        disk_domain_id_map_pool = dict()
        for i in pool_and_disk_domain_id:
            disk_domain_id = i.get("Disk Domain ID")
            pool_id = i.get("ID")
            if disk_domain_id not in disk_domain_id_map_pool:
                disk_domain_id_map_pool[disk_domain_id] = [pool_id]
            else:
                disk_domain_id_map_pool.get(disk_domain_id).append(pool_id)
        return disk_domain_id_map_pool

    def get_risk_pool_info(self, pool_id):
        """
        步骤4:
        执行show storage_pool tier pool_id=[ID]
        |filterColumn include columnList=RAID\sLevel,Pool\sID,Name，
        记录所有RAID LEVEL为RAID10的存储池对应的层级信息，
        Extreme Performance对应Tier0，Performance对应Tier1，Capacity对应Tier2；
        :param:pool_id
        :return:dict
        """
        cmd = (
            "show storage_pool tier pool_id={}|"
            "filterColumn include"
            " columnList=RAID\\sLevel,Pool\\sID,Name".format(pool_id)
        )
        flag, cli_ret, pool_info_list, msg = get_vertical_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(msg, cli_ret)
        if not pool_info_list:
            return dict()
        pool_info = pool_info_list[0]
        if pool_info.get("RAID Level") == "RAID10":
            return pool_info
        else:
            return dict()

    def get_hot_spare_strategy(self, disk_domain_id):
        """
        步骤5:
        执行show disk_domain general disk_domain_id=[ID]|filterColumn include
        columnList=Tier0\sHot\sSpare\sStrategy,Tier1\sHot\sSpare\sStrategy,
        Tier2\sHot\sSpare\sStrategy，记录硬盘域所有层级的热备策略；
        :param disk_domain_id:
        :return: dict
        """
        cmd = (
            "show disk_domain general disk_domain_id={}|filterColumn include"
            " columnList=Tier0\\sHot\\sSpare\\sStrategy,"
            "Tier1\\sHot\\sSpare\\sStrategy,"
            "Tier2\\sHot\\sSpare\\sStrategy".format(disk_domain_id)
        )
        flag, ret, hot_strategy_list, msg = get_vertical_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        return hot_strategy_list[0]

    def get_tier_disk_amount(self, disk_domain_id):
        """
        步骤6:
        执行show disk_domain general disk_domain_id=[ID]|filterColumn include
        columnList=Tier0\sDisk\sNumber,Tier1\sDisk\sNumber,Tier2\sDisk\sNumber，
        记录硬盘域每层盘的数量；
        :param disk_domain_id:
        :return:dict
        """
        cmd = (
            "show disk_domain general disk_domain_id={}|"
            "filterColumn include columnList=Tier0\\sDisk\\sNumber,"
            "Tier1\\sDisk\\sNumber,Tier2\\sDisk\\sNumber".format(
                disk_domain_id
            )
        )
        flag, ret, disks_info_list, msg = get_vertical_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        return disks_info_list[0]

    def get_member_disk_id(self, disk_domain_id):
        """
        步骤7:
        执行命令：show disk in_domain disk_domain_id={}|
        filterColumn include columnList=ID，记录硬盘域的成员盘ID；
        :param disk_domain_id:
        :return:list
        """
        cmd = (
            "show disk in_domain disk_domain_id={}|"
            "filterColumn include columnList=ID".format(disk_domain_id)
        )
        flag, ret, member_list, msg = get_horizontal_no_standard_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        member_disk_id = [i.get("ID") for i in member_list]
        return member_disk_id

    def get_safety_disk_id(self):
        """
        步骤8:
        执行执行命令：show disk system，记录保险箱盘的盘ID。
        :return:list
        """
        cmd = "show disk system"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        disk_info = cliUtil.getHorizontalNostandardCliRet(ret)
        self.safety_disks = [i.get("ID") for i in disk_info]

    def is_risk_version(self):
        """
        步骤2:
        执行命令：show upgrade package，查询系统软件版本信息,并做判断
        :return:bool
        """
        (
            flag,
            cli_ret,
            err_msg,
            self.software_version,
            _,
        ) = common.getVersion(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        return any(
            [
                all(
                    [
                        self.software_version.startswith("V3"),
                        self.software_version >= START_CHECK_VERSION_V3,
                        self.software_version <= END_CHECK_VERSION_V3,
                    ]
                ),
                all(
                    [
                        self.software_version.startswith("V5"),
                        self.software_version >= START_CHECK_VERSION_V5,
                        self.software_version < END_CHECK_VERSION_V5,
                    ]
                ),
            ]
        )
