# -*- coding: UTF-8 -*-
import cliUtil
import common
import traceback
import config
from common import UnCheckException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ALL_CLI_RET = ''

# 开始检查版本，主要用于截取版本信息
START_CHECK_VERSION = "V300R001C00"

def execute(cli):
    """
        步骤1 以admin用户登录设备；
        步骤2 执行show system general命令，查询设备型号；
        步骤3 执行show upgrade package命令，查询当前系统版本和补丁号；
        步骤4 执行show enclosure命令，查询是否配套V1高密框；
        步骤5 执行show enclosure enclosure_id=DAEXXX命令，查询V1高密框级联深度；
        步骤6 执行show version all命令，查询V1高密框版本号；
        备注：查询配套表版本"Version4=hds_ses.bin"字段后的版本号方法
        步骤1：进入developer模式；
        步骤2：进入minisystem；
        步骤3：cat /startup_disk/image/boot/versions.conf，搜索"Version4=hds_ses.bin"
        例如：
        Version4=hds_ses.bin,29.60T69
        Version4=hds_ses.bin,29.01.60T69
        获取到版本号信息"29.60T69"和"29.01.60T69"。
    software_enclosure_version_hds_ses
    """
    cliRet = ""
    errMsg = ""

    try:

        # 判断软件型号是否有风险
        riskFlag = isRiskProductMode(cli)
        if not riskFlag:
            return (True, ALL_CLI_RET, "")

        # 获取软件产品版本和补丁版本号
        productVersion, hostPatchVersion = getProductVersionAndHotPatchVersion(cli)
        LOGGER.logInfo("productVersion is: %s, hostPatchVersion is:%s" % (productVersion, 
                                                                          hostPatchVersion))

        # 场景一，有补丁版本的检查。
        riskFlagWithPatch = hasPatchScenceFirmwareVersionCheck(productVersion, 
                                                               hostPatchVersion)
        LOGGER.logInfo("riskFlagWithPatch is: %s" % riskFlagWithPatch)

        # 场景二，无补丁版本的检查。
        riskFlagWithoutPatch = noPatchScenceFirmwareVersionCheck(productVersion, 
                                                                 hostPatchVersion)
        LOGGER.logInfo("riskFlagWithoutPatch is: %s" % riskFlagWithoutPatch)

        # 获取高密框
        enclosureList = getHighDensityEnclosure(cli)
        LOGGER.logInfo("high density enclosureList is: %s" % enclosureList)

        if not enclosureList:
            return (True, ALL_CLI_RET, '')

        # 记录错误的硬盘框ID
        errEnclosureList = []

        if riskFlagWithPatch:
            enclosureInfoDict = {}
            # 获取高密框版本号
            getEnclosureVersion(cli, enclosureInfoDict)

            # 获取全部固件版本
            fWVersionList = getFWVersion(cli)

            LOGGER.logInfo(
                "case 1: the high density enclosureList:%s, enclosureInfoDict is: %s, fWVersionList:%s" % (
                    enclosureList, enclosureInfoDict, fWVersionList))

            for enclosureId in enclosureList:
                sesVersion = enclosureInfoDict.get(enclosureId, {}).get("sesVer")
                isConfigOk = False
                for fwVersion in fWVersionList:
                    if sesVersion and sesVersion in fwVersion:
                        isConfigOk = True
                        break

                if not isConfigOk:
                    errEnclosureList.append(enclosureId)

            # 返回CLI模式
            cliUtil.enterCliModeFromDebugModel(cli, LANG)

        elif riskFlagWithoutPatch:
            # 获取高密框级联深度
            enclosureInfoDict = getEnclosureExpansionDepth(cli, enclosureList)
            LOGGER.logInfo(
                "case 2: the high density enclosureInfoDict is: %s, enclosureList:%s" % (
                enclosureInfoDict, enclosureList))

            for enclosureId in enclosureInfoDict:
                isConfigOk = True
                depth = enclosureInfoDict[enclosureId].get("depth")
                if depth is None or depth == "":
                    continue

                if int(depth) >= 2:
                    isConfigOk = False

                if int(depth) == 0 and len(enclosureList) >= 2:
                    isConfigOk = False

                if not isConfigOk:
                    LOGGER.logInfo("case 2: the high density enclosure is lower Id is: %s" % enclosureId)
                    errEnclosureList.append(enclosureId)

        if errEnclosureList:
            errMsg = common.getMsg(LANG, "enclosure.version.hds.ses.lower", ",".join(errEnclosureList))  
            return (cliUtil.RESULT_WARNING, ALL_CLI_RET, errMsg)

        return True, ALL_CLI_RET, ""

    except UnCheckException, unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo("UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        return (cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg)

    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal"))

    finally:
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def hasPatchScenceFirmwareVersionCheck(productVersion, hostPatchVersion):
    """
    场景一检查
    :param productVersion: 产品版本
    :param hostPatchVersion: 补丁版本
    :return:Flag True:有风险，False 未打补丁，继续检查
    """
    if productVersion[0: len(
        START_CHECK_VERSION)] in config.RISK_PRODUCT_AND_HAS_PATCH and hostPatchVersion >= config.RISK_PRODUCT_AND_HAS_PATCH.get(
        productVersion[0: len(START_CHECK_VERSION)], ""):
        return True
    else:
        return False


def noPatchScenceFirmwareVersionCheck(productVersion, hostPatchVersion):
    """
    场景二检查
    :param productVersion: 产品版本
    :param hostPatchVersion: 补丁版本
    :return:Flag True:有风险，False，无风险，不检查
    """
    if productVersion in config.RISK_PRODUCT_AND_NO_PATCH and hostPatchVersion < config.RISK_PRODUCT_AND_NO_PATCH.get(
            productVersion, ""):
        return True
    else:
        return False


def isRiskProductMode(cli):
    """
    判断是否是风险型号
    :param cli:
    :return: True, 有风险
              False, 无风险
    """
    global ALL_CLI_RET
    # 获取软件型号
    flag, productModel, cliRet, errMsg = cliUtil.getProductModelWithCliRet(cli, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    LOGGER.logInfo("Product Model is :%s" % productModel)
    if flag != True:
        LOGGER.logInfo("Failed to get product model errMsg:%s" % errMsg)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", "system"),
                               ALL_CLI_RET)

    if productModel not in config.ENCLOSURE_RISK_PRODUCT_MODE:
        return False

    return True


def getHighDensityEnclosure(cli):
    """
    获取高密框信息
    :param cli:
    :return:enclosureList
    """
    global ALL_CLI_RET
    enclosureList = []
    cmd = "show enclosure"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        LOGGER.logInfo("Failed to get lun errMsg:%s" % errMsg)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", "enclosure"),
                               ALL_CLI_RET)
    enclosureInfoList = cliUtil.getHorizontalCliRet(cliRet)
    for enclosureInfo in enclosureInfoList:
        type = enclosureInfo.get("Type", "")
        id = enclosureInfo.get("ID", "")
        if type == config.HIGH_DENSITY_ENCLOSURE_TYPE:
            enclosureList.append(id)

    return enclosureList


def getEnclosureExpansionDepth(cli, enclosureList):
    """
    获取硬盘框级联深度
    :param cli:
    :return:enclosureInfoDict
    """
    global ALL_CLI_RET
    enclosureInfoDict = {}
    for enclosureId in enclosureList:
        cmd = "show enclosure enclosure_id=%s" % enclosureId
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
        if flag != True:
            LOGGER.logInfo("Failed to get lun errMsg:%s" % errMsg)
            raise UnCheckException(common.getMsg(LANG, "cannot.get.info", "enclosure[ID:%s]" % enclosureId),
                                   ALL_CLI_RET)

        enclosureInfoList = cliUtil.getVerticalCliRet(cliRet)
        for enclosureInfo in enclosureInfoList:
            depth = enclosureInfo.get("Expansion Depth", "")
            enclosureInfoDict[enclosureId] = {"depth": depth}
            break
    return enclosureInfoDict


def getEnclosureVersion(cli, enclosureInfoDict):
    """
    获取硬盘框版本号
    :param cli:
    :return:
    """
    global ALL_CLI_RET

    cmd = "show version all"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        LOGGER.logInfo("Failed to get lun errMsg:%s" % errMsg)
        msgInfo = {"zh": u"高密框固件版本",
                   "en": "the firmware version of the high-density enclosure"}.get(LANG)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", msgInfo),
                               ALL_CLI_RET)

    enclosureVersionInfoList = cliUtil.getHorizontalCliRet(
        cliRet[cliRet.find("Expansion Module:"):cliRet.find("BBU")])
    for enclosureVersionInfo in enclosureVersionInfoList:
        sesVer = enclosureVersionInfo.get("SES Version", "")
        enclosureId = enclosureVersionInfo.get("ID", "").split(".")[0]
        if not enclosureId or not sesVer:
            LOGGER.logInfo("enclosureId or sesVer is not exist:enclosureId:%s, sesVer:%s" % (enclosureId, sesVer))
            continue

        enclosureInfo = enclosureInfoDict.get(enclosureId, {})
        enclosureInfo["sesVer"] = sesVer
        enclosureInfoDict[enclosureId] = enclosureInfo


def getFWVersion(cli):
    """
     获取固件版本
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    cmd = "cat /startup_disk/image/boot/versions.conf"
    flag, cliRet, errMsg = cliUtil.excuteCmdInMinisystemModel(cli, cmd, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)

    # 回显样例"Version4=hds_ses.bin,29.01.60T64"
    # 回显样例"Version4=hds_ses.bin,29.60T64"
    if flag != True:
        LOGGER.logInfo("Failed to get lun errMsg:%s" % errMsg)
        msgInfo = {"zh": u"固件版本",
                   "en": "the firmware version"}.get(LANG)
        raise UnCheckException(common.getMsg(LANG, "cannot.get.info", msgInfo),
                               ALL_CLI_RET)

    cliRetList = cliRet.splitlines()
    versionList = []
    for line in cliRetList:
        if "=hds_ses.bin" in line:
            versionList.append(line)

    return versionList


def getProductVersionAndHotPatchVersion(cli):
    """
    获取当前SPC版本信息,补丁版本信息
    :param cli:
    :return:productVersion 产品版本信息, hotPatchVersion补丁版本信息
    """
    global ALL_CLI_RET
    (succFlag, cliRet, errMsg), softwareVersionList, hotPatchVersionList = common.parse_upgradePackage(cli, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if succFlag != True:
        LOGGER.logInfo("get package info fail! softwareVersionList:%s, hotPatchVersionList%s" %
                       (softwareVersionList, hotPatchVersionList))
        raise UnCheckException(errMsg, ALL_CLI_RET)

    flag, productVersion, errMsg = common.getCurrentVersion(softwareVersionList, LANG)
    if flag != True:
        LOGGER.logInfo("get current product version error. errMsg: %s" % errMsg)
        raise UnCheckException(errMsg, ALL_CLI_RET)

    flag, hotPatchVersion, errMsg = common.getHotPatchVersion(hotPatchVersionList, LANG)
    if flag != True:
        LOGGER.logInfo("get hotPatchVersion error. errMsg: %s" % errMsg)
        raise UnCheckException(errMsg, ALL_CLI_RET)

    return productVersion, hotPatchVersion
