# -*- coding: UTF-8 -*-
import cliUtil
import common
import decimal

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
def execute(cli):
    '''
    @summary: 在文件系统所在的存储池可用容量充足的前提下，文件系统转透写的原因为8（pool容量不足），则说明文件系统透写状态异常。
            此异常情况将导致系统性能严重下降甚至丢失数据。到2017-12-19日止，已发布V300R003C20SPH205补丁只能解决部分问题场景，其它场景需要手动恢复。
    '''
    flag = True
    allCliRet = ""
    errMsg = ""

    try:
        # 检查是否为问题版本
        flag, cliRet, errMsg = checkVersion(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != False:
            return (flag, allCliRet, errMsg)
                #刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 1, LOGGER)  
        # 获取文件系统信息
        flag, cliRet, errMsg, fileSystemDictList = getFileSystem(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not fileSystemDictList:
            return (flag, allCliRet, errMsg)
                #刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 5, LOGGER)
        # 获取转透写状态异常的文件系统
        flag, cliRet, errMsg, abnormalFileSystemDict = getAbnormalFileSystem(cli, fileSystemDictList)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not abnormalFileSystemDict:
            return (flag, allCliRet, errMsg)

        # 获取pool下类型为Thin的文件系统的数量
        flag, errMsg, thinFileSystemNumDict = getThinFileSystemNum(fileSystemDictList)
        if not thinFileSystemNumDict:
            return (flag, allCliRet, errMsg)
        
        common.refreshProcess(PY_JAVA_ENV, 90, LOGGER)
        
        # 获取pool信息 
        flag, cliRet, errMsg, poolDictList = getPool(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not poolDictList:
            return (flag, allCliRet, errMsg)
        
        common.refreshProcess(PY_JAVA_ENV, 93, LOGGER)
        
        # 获取容量充足的存储池
        flag, errMsg, sufficientCapacityPoolList = getSufficientCapacityPool(poolDictList, thinFileSystemNumDict)
        if not sufficientCapacityPoolList:
            return (flag, allCliRet, errMsg)
        
        common.refreshProcess(PY_JAVA_ENV, 96, LOGGER)
        
        # 检查文件系统转透写状态
        flag, errMsg = checkFileSystem(abnormalFileSystemDict, sufficientCapacityPoolList)
        return (flag, allCliRet, errMsg)
        
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))
    finally:
        common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)

def checkVersion(cli):
    """
    @summary: 检查系统软件版本是否为问题版本
    @return: True 非问题版本，检查通过；
             False 问题版本，继续检查；
             cliUtil.RESULT_NOCHECK 获取版本信息失败
    """
    flag, cliRet, errMsg, productVersion = common.getProductVersionByUpgradePackage(cli, LANG)
    if flag != True:
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
    
    # 问题版本
    if productVersion >= "V300R002C10" and productVersion <= "V300R003C20SPC200":
        return (False, cliRet, "")
    
    # 非问题版本
    LOGGER.logInfo("The product version is not bug version, checking result is pass.")
    return (True, cliRet, "")


def getFileSystem(cli):
    """
    @summary: 获取文件系统信息
    """
    cmd = "show file_system general|filterColumn include columnList=ID,Pool\sID,Type,Health\sStatus,Running\sStatus,Work\sController"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloper(cli, cmd, True, LANG)
    if flag != True:
        # 不存在license，检查通过
        if cliUtil.isNoneLicense(cliRet):
            LOGGER.logInfo("The system does not have file system license, checking result is pass.")
            return (True, cliRet, errMsg, [])
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    # 不存在文件系统检查通过
    if cliUtil.queryResultWithNoRecord(cliRet):
        LOGGER.logInfo("The system does not have file system, checking result is pass.")
        return (True, cliRet, "", [])
        
    fileSystemDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    if len(fileSystemDictList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.filesystem.info")
        LOGGER.logNoPass("Cannot get information about file system")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    return (True, cliRet, "", fileSystemDictList)


def getAbnormalFileSystem(cli, fileSystemDictList):
    """
    @summary: 获取状态转透写状态异常（原因为8,代表归属的pool容量不足）的文件系统及其归属的pool
    @param fileSystemDictList: 所有文件系统的信息
    @return: abnormalFileSystemDict， 所有因pool容量不足导致的转透写的文件系统及其归属的pool
    """
    allCliRet = ""
    abnormalFileSystemDict = {}
    cmd = "show space statistic_info file_system_id=%s controller=%s|filterColumn include columnList=Write\sThrough\sReason"
    
    currentProcess = 6
    singleStep = 80/len(fileSystemDictList)
    nodeProcess = singleStep
    for fileSystemDict in fileSystemDictList:         
        healthStatus = fileSystemDict.get("Health Status")
        runningStatus = fileSystemDict.get("Running Status")
        if healthStatus is None or runningStatus is None:
            errMsg = common.getMsg(LANG, "cannot.get.filesystem.info")
            LOGGER.logNoPass("Cannot get information about File System")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
         
        # 状态异常的文件系统不检查         
        if healthStatus != "Normal" and runningStatus != "Online":
            continue
        
        id = fileSystemDict.get("ID")
        workController = fileSystemDict.get("Work Controller")
        if id is None or workController is None:
            errMsg = common.getMsg(LANG, "cannot.get.filesystem.info")
            LOGGER.logNoPass("Cannot get information about File System")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
        
        flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloper(cli, cmd % (id, workController), True, LANG)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
        
        writeThroughReasonDictList = cliUtil.getVerticalCliRet(cliRet)
        if len(writeThroughReasonDictList) == 0:
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"文件系统转透写的原因", "en":"cause of the file system write through"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about cause of the file system write through")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
        
        writeThroughReason = writeThroughReasonDictList[0].get("Write Through Reason")
        if writeThroughReason is None:
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"文件系统转透写的原因", "en":"cause of the file system write through"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about cause of the file system write through")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
        
        # 文件系统转透写原因为8(代表所归属的pool容量不足导致的转透写)
        if writeThroughReason == "8":
            poolId = fileSystemDict.get("Pool ID")
            if poolId is None:
                errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"文件系统转透写的原因", "en":"cause of the file system write through"}.get(LANG))
                LOGGER.logNoPass("Cannot get information about cause of the file system write through")
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg, {})
            
            # 保存因pool容量不足导致的转透写的文件系统及其归属的pool
            abnormalFileSystemDict[id] = poolId
            
        if nodeProcess > 1:
            currentProcess += int(nodeProcess)
            nodeProcess = singleStep
            common.refreshProcess(PY_JAVA_ENV, currentProcess, LOGGER)
        else:
            nodeProcess += singleStep
    
    return (True, allCliRet, "", abnormalFileSystemDict)
    
    
def getThinFileSystemNum(fileSystemDictList):
    """
    @summary: 获取每个pool下类型为Thin的文件系统数量
    @param fileSystemDictList: 所有文件系统的信息
    @return: thinFileSystemNumDict， 所有pool下类型为Thin的文件系统数量
    """
    thinFileSystemNumDict = {}
    
    for fileSystemDict in fileSystemDictList:
        type = fileSystemDict.get("Type")
        if type is None:
            errMsg = common.getMsg(LANG, "cannot.get.filesystem.info")
            LOGGER.logNoPass("Cannot get information about File System")
            return (cliUtil.RESULT_NOCHECK, errMsg, {})
        
        if type == "Thin":
            poolId = fileSystemDict.get("Pool ID")
            if poolId is None:
                errMsg = common.getMsg(LANG, "cannot.get.filesystem.info")
                LOGGER.logNoPass("Cannot get information about File System")
                return (cliUtil.RESULT_NOCHECK, errMsg, {})
            
            thinFileSystemNum = thinFileSystemNumDict.get(poolId, 0)
            thinFileSystemNumDict[poolId] = thinFileSystemNum + 1

    return (True, "", thinFileSystemNumDict)


def getPool(cli):
    """
    @summary: 获取存储池信息
    """
    cmd = "show storage_pool general|filterColumn include columnList=ID,Total\sCapacity,Free\sCapacity"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloper(cli, cmd, True, LANG)
    if flag != True:
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    # 不存在存储池，检查通过
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, "", [])
    
    poolDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    if len(poolDictList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
        LOGGER.logNoPass("Cannot get information about storage pool")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    return (True, cliRet, "", poolDictList)


def getSufficientCapacityPool(poolDictList, thinFileSystemNumDict):
    """
    @summary: 获取所有容量充足的存储池
    @param poolDictList: pool信息字典列表
    @param thinFileSystemNumDict: pool下类型为Thin的文件系统数量
    """
    sufficientCapacityPoolList = []
    
    for poolDict in poolDictList:
        # pool下类型为Thin的文件系统数量
        id = poolDict.get("ID")
        if id is None:
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about storage pool")
            return (cliUtil.RESULT_NOCHECK, errMsg, [])
        
        thinFileSystemNum = thinFileSystemNumDict.get(id, 0)
        # 只需要检查存在类型为Thin的文件系统的pool
        if thinFileSystemNum == 0:
            LOGGER.logInfo("The storage pool(%s) does not have thin file system, it does not need check its file system." % id)
            continue
        
        # pool的总容量
        totalCapacity = poolDict.get("Total Capacity")
        if totalCapacity is None:
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about storage pool")
            return (cliUtil.RESULT_NOCHECK, errMsg, [])
        
        flag, totalCapacity = common.changUnit2GBDecimal(totalCapacity)
        if flag == False:
            errMsg = errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about storage pool")
            return (cliUtil.RESULT_NOCHECK, errMsg, [])
        
        # pool的空闲容量
        freeCapacity = poolDict.get("Free Capacity")
        if freeCapacity is None :
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about storage pool")
            return (cliUtil.RESULT_NOCHECK, errMsg, [])
        
        flag, freeCapacity = common.changUnit2GBDecimal(freeCapacity)
        if flag == False:
            errMsg = errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"存储池", "en":"storage pool"}.get(LANG))
            LOGGER.logNoPass("Cannot get information about storage pool")
            return (cliUtil.RESULT_NOCHECK, errMsg, [])
        
        # 检查pool的可用容量充足
        if freeCapacity >= min(thinFileSystemNum * 1, 20, totalCapacity * decimal.Decimal("0.005")) * 4:
            sufficientCapacityPoolList.append(id)
    
    return (True, "", sufficientCapacityPoolList)


def checkFileSystem(abnormalFileSystemDict, sufficientCapacityPoolList):
    """
    @summary: 检查转透写状态异常的文件系统是否存在问题，文件系统转透写原因异常且所归属的存储池可用容量充足，则当前文件系统的透写状态异常
    @param abnormalFileSystemDict: 转透写原因异常的文件系统及其归属的pool
    @param sufficientCapacityPoolList: 可用容量充足的pool
    """
    abnormalFileSystemList = []
    
    for abnormalFileSystem in abnormalFileSystemDict.keys(): 
        pool = abnormalFileSystemDict.get(abnormalFileSystem)
        if pool in sufficientCapacityPoolList:
            abnormalFileSystemList.append(abnormalFileSystem)
            
    if abnormalFileSystemList:
        common.sortListByInt(abnormalFileSystemList, LOGGER)
        errMsg = common.getMsg(LANG, "fileSystem.writeThrough.abnormal", ",".join(abnormalFileSystemList))
        return (False, errMsg)   
    
    return (True, "") 

