# -*- coding: UTF-8 -*-

import traceback

import common
import cliUtil
from common_utils import get_err_msg
from common import UnCheckException

from common_cache import not_support_nas_domain

from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache

PY_JAVA_ENV = py_java_env
LOGGER = common.getLogger(PY_LOGGER, __file__)
LANG = common.getLang(PY_JAVA_ENV)


def execute(cli):
    """
    Dorado V6 NAS双活检查
    """
    nfs_service_check = NasPairCheck(cli)
    flag, msg = nfs_service_check.execute_check()
    return flag, "\n".join(nfs_service_check.all_cli_ret), msg


class NasPairCheck:
    """
    NAS双活检查
    """
    def __init__(self, cli):
        self.cli = cli
        self.all_cli_ret = []
        self.err_msg_list = []

    def execute_check(self):
        try:
            self.check_quorum_serve_link()
            self.check_quorum_serve()
            self.check_domain_status()
            self.check_nas_pair_status()
            if self.err_msg_list:
                return False, "".join(self.err_msg_list)
            return True, ''

        except UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(traceback.format_exc())
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(LANG, err_msg)

    def check_quorum_serve_link(self):
        """
        1 双活仲裁服务器链路状态为Link Up；
        :return:
        """
        cmd = "show quorum_server_link general"
        flag, ret, msg = execute_cmd_in_cli_mode_with_cache(
            PY_JAVA_ENV, self.cli, cmd, LOGGER
        )
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)

        error_id_list = []
        res_list = cliUtil.getHorizontalCliRet(ret)
        for res_info in res_list:
            if res_info.get("Link Status") != "Link Up":
                error_id_list.append(res_info.get("Link ID"))

        if error_id_list:
            self.err_msg_list.append(get_err_msg(
                LANG, "nas.quorum.link.status.is.abnormal",
                ", ".join(error_id_list)
            ))

    def check_quorum_serve(self):
        """
        1 双活仲裁服务器状态为Online；
        :return:
        """
        cmd = "show quorum_server general|filterColumn include columnList=Server\sID,Running\sStatus"
        flag, ret, msg = execute_cmd_in_cli_mode_with_cache(PY_JAVA_ENV, self.cli, cmd, LOGGER)
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)

        error_id_list = []
        res_list = cliUtil.getHorizontalCliRet(ret)
        for res_info in res_list:
            if res_info.get("Running Status") != "Online":
                error_id_list.append(res_info.get("Server ID"))

        if error_id_list:
            self.err_msg_list.append(get_err_msg(
                LANG, "nas.quorum.server.status.is.abnormal", ", ".join(error_id_list)
            ))

    def check_domain_status(self):
        """
        1 双活domain为 Normal；
        :return:
        """
        cmd = "show fs_hyper_metro_domain general"
        flag, ret, msg = execute_cmd_in_cli_mode_with_cache(PY_JAVA_ENV, self.cli, cmd, LOGGER)
        self.all_cli_ret.append(ret)
        if not_support_nas_domain(ret):
            return

        if flag is not True:
            raise UnCheckException(msg, ret)

        error_id_list = []
        res_list = cliUtil.getHorizontalCliRet(ret)
        for res_info in res_list:
            if res_info.get("Running Status") != "Normal":
                error_id_list.append(res_info.get("ID"))

        if error_id_list:
            self.err_msg_list.append(get_err_msg(
                LANG, "nas.domain.status.is.abnormal", ", ".join(error_id_list)
            ))

    def check_nas_pair_status(self):
        """
        1 双活pair状态为Normal；
        :return:
        """

        normal_pair_status = (
            common.STATUS_NORMAL,
            common.STATUS_SYNCHRONIZING,
            common.STATUS_PAUSE
          )

        cmd = "show hyper_metro_pair general |filterRow column=Type " \
              "predict=equal_to value=FS |filterColumn include " \
              "columnList=ID,Health\sStatus,Link\sStatus,Running\sStatus"
        flag, ret, msg = execute_cmd_in_cli_mode_with_cache(
            PY_JAVA_ENV, self.cli, cmd, LOGGER
        )
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)

        error_id_list = []
        res_list = cliUtil.getHorizontalCliRet(ret)
        for res_info in res_list:
            domain_id = res_info.get("ID")
            link_status = res_info.get("Link Status")
            health_status = res_info.get("Health Status")
            running_status = res_info.get("Running Status")
            if any([health_status != "Normal",
                    running_status not in normal_pair_status,
                    link_status not in ('Linkup', 'Link Up')]):
                error_id_list.append(domain_id)

        if error_id_list:
            self.err_msg_list.append(get_err_msg(
                LANG, "nas.pair.status.is.abnormal", ", ".join(error_id_list)
            ))
