# -*- coding: UTF-8 -*-
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
NORMAL_DISK = 'Normal'
VERSION_NAME = "Logic"


def execute(cli):
    """
    Logic版本检查:
        步骤1 以admin用户登录设备；
        步骤2 执行命令：show enclosure，查询框类型和ID；
        步骤3 执行命令：show version all，查询版本号。
    """
    flag = True
    cli_ret = ""
    err_msg = ""
    try:
        # 获取框类型和ID
        enclosure_flag, enclosure_cli_ret, disk_type_id_dict = \
            get_disk_enclosure_info(cli)
        cli_ret = common.joinLines(cli_ret, enclosure_cli_ret)
        if enclosure_flag is not True:
            return enclosure_flag, cli_ret, \
                   common.getMsg(LANG, "cannot.get.enclosure.info")
        # 获取版本号
        cmd = "show version all"
        LOGGER.logExecCmd(cmd)
        version_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        if version_ret[0] is not True:
            LOGGER.logSysAbnormal()
            return version_ret

        cli_ret = common.joinLines(cli_ret, version_ret[1])
        controller_ret = cliUtil.getSplitedCliRet(cli_ret, "Controller:")
        controller_cli_list = cliUtil.getVerticalCliRet(controller_ret)

        if len(controller_cli_list) == 0:
            err_msg = common.getMsg(LANG,
                                    "cannot.get.version.of.controller")
            LOGGER.logNoPass(
                "Cannot get information about version of controllers")
            return False, cli_ret, err_msg

        logic_version_set = set()
        for ret_dict in controller_cli_list:
            controller_id = ret_dict.get("Controller")
            logic_version = ret_dict.get("Logic Version")
            if len(logic_version) > 0:
                logic_version_set.add(logic_version)
            else:
                flag = False
                err_msg += common.getMsg(
                    LANG, "cannot.get.logic.version.of.controller",
                    controller_id)
                LOGGER.logNoPass(
                    "Cannot get information about "
                    "Logic version of controller [%s]" % controller_id)

        if len(logic_version_set) > 1:
            flag = False
            err_msg += common.getMsg(
                LANG, "version.of.controller.inconsistent", (
                    VERSION_NAME, ", ".join(logic_version_set)))
            LOGGER.logNoPass(
                "The Logic versions of all controllers are inconsistent")
        # 若无硬盘框对应版本信息，则不继续检查
        exp_module_ret = cliUtil.getSplitedCliRet(cli_ret,
                                                  "Expansion Module:")
        if len(exp_module_ret) == 0:
            return flag, cli_ret, err_msg
        cli_ret_lines_exp_module_list = \
            cliUtil.getHorizontalCliRet(exp_module_ret)
        check_flag, check_error = check_logic_version(
            cli, cli_ret_lines_exp_module_list, disk_type_id_dict)
        flag = flag and check_flag
        err_msg = common.joinLines(err_msg, check_error)
        return flag, cli_ret, err_msg
    except Exception as exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                common.getMsg(LANG, "query.result.abnormal"))


def get_disk_enclosure_info(cli):
    """
    获取硬盘框信息
    :param cli:
    :return:
    """
    disk_type_id_dict = {}
    # 获取框类型和ID
    enclosure_flag, enclosure_info, enclosure_cli_ret = \
        common.getDiskEnclosure(cli, LANG)
    if enclosure_flag is not True:
        return enclosure_flag, enclosure_cli_ret, disk_type_id_dict
    enclosure_cli_list = cliUtil.getHorizontalCliRet(enclosure_cli_ret)
    if len(enclosure_cli_list) == 0:
        return False, enclosure_cli_ret, disk_type_id_dict
    # 获取硬盘框类型-ID字典{type:[id_list]}
    for line in enclosure_cli_list:
        id_list = []
        enclosure_type = line.get('Type')
        enclosure_id = line.get('ID')
        # 高密框
        if "4U 75" in enclosure_type:
            curr_type = enclosure_type
        else:
            # 普通硬盘框
            curr_type = NORMAL_DISK
        if bool(disk_type_id_dict.get(curr_type)):
            id_list = disk_type_id_dict.get(curr_type)
        id_list.append(enclosure_id)
        disk_type_id_dict[curr_type] = id_list
    return True, enclosure_cli_ret, disk_type_id_dict


def check_logic_version(cli, cli_ret_lines_exp_module_list,
                        disk_type_id_dict):
    """
    检查各类型硬盘框Logic版本是否一致
    :param cli:
    :param cli_ret_lines_exp_module_list:
    :param disk_type_id_dict:
    :return:
    """
    error_msg = ''
    flag = True
    # 是否为单控环境
    is_single_pdt_model = common.isSigleModel(cli, LANG)
    # 解析每个类型对应的Logic版本号集合
    type_versions_dict = {}
    for line in cli_ret_lines_exp_module_list:
        version_set = set()
        exp_module_id = line.get("ID")
        enclosure_id = exp_module_id.split(".")[0]
        logic_version_exp = line.get("Logic Version")
        # 单控环境，不检查B级联板的Logic版本号
        if is_single_pdt_model and ".B" in exp_module_id:
            continue
        if not bool(logic_version_exp) or logic_version_exp == '--':
            flag = False
            error_msg = common.joinLines(
                error_msg,
                common.getMsg(
                    LANG,
                    "cannot.get.logic.version.of.expansion.module",
                    exp_module_id))
        curr_type = get_enclosure_type_by_id(enclosure_id,
                                             disk_type_id_dict)
        if curr_type in type_versions_dict:
            version_set = type_versions_dict.get(curr_type)
        version_set.add(logic_version_exp)
        type_versions_dict[curr_type] = version_set
    # 检查各类型是否一致
    for disk_type in type_versions_dict:
        disk_versions = type_versions_dict.get(disk_type)
        if not bool(disk_versions):
            continue
        if len(disk_versions) > 1:
            if disk_type == NORMAL_DISK:
                flag = False
                error_msg = \
                    common.joinLines(
                        error_msg,
                        common.getMsg(
                            LANG,
                            'version.of.expansion.module.'
                            'on.disk.inconsistent',
                            (VERSION_NAME, ", ".join(disk_versions))))
            else:
                flag = False
                error_msg = \
                    common.joinLines(
                        error_msg,
                        common.getMsg(
                            LANG, "version.of.expansion.module."
                                  "on.high-density.inconsistent",
                            (disk_type, VERSION_NAME,
                             ", ".join(disk_versions))))
    return flag, error_msg


def get_enclosure_type_by_id(enclosure_id, disk_type_id_dict):
    """
    根据硬盘框ID，获取该硬盘框为普通硬盘框还是高密框，以及高密框的类型
    :param enclosure_id:
    :param disk_type_id_dict:
    :return:
    """
    for curr_type, id_list in disk_type_id_dict.items():
        if enclosure_id in id_list:
            return curr_type
    return ''
