# -*- coding: UTF-8 -*-

import datetime
import time
import traceback
import common
import cliUtil
from cbb.business.operate.fru.common.checkItem import compute_storage_common
from cbb.frame.checkitem.context_adapter import InspectContext

import java.lang.Exception as JException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
CRITICAL_URL = "/fce/monitor/v1/events?severity=critical&eventType=Cleared&startTime={}&endTime={}"
MAJOR_URL = "/fce/monitor/v1/events?severity=major&eventType=Cleared&startTime={}&endTime={}"


def execute(cli):
    """
    已恢复重复告警
    @param cli: cli
    """
    ret = []
    try:
        flag, sys_date, cli_ret = common.getCurSysDate(cli)
        if not flag:
            LOGGER.logNoPass("get system date abnormal.")
            return False, cli_ret, common.getMsg(LANG, "query.system.date.abnormal")
        ret.append(cli_ret)
        year, month, day = map(int, sys_date.split('-'))
        end_time = get_timestamp(str(datetime.date(year, month, day)))
        start_time = get_timestamp(str(datetime.date(year, month, day) - datetime.timedelta(30)))

        critical_alarm_records, major_alarm_records = get_critical_and_major_alarms(cli, end_time, start_time)
        ret.append("critical_alarm_records:{}".format(critical_alarm_records))
        ret.append("major_alarm_records:{}".format(major_alarm_records))

        critical_alarm_dict = get_alarm_num_dict(critical_alarm_records)
        major_alarm_dict = get_alarm_num_dict(major_alarm_records)

        err_msg = get_err_msg(critical_alarm_dict, major_alarm_dict)

        if not err_msg:
            return True, "\n".join(ret), err_msg
        return cliUtil.RESULT_WARNING, "\n".join(ret), err_msg

    except (Exception, JException) as ex:
        LOGGER.logException(ex)
        LOGGER.logError(str(traceback.format_exc()))
        return cliUtil.RESULT_NOCHECK, "\n".join(ret), common.getMsg(LANG, "query.result.abnormal")


def get_err_msg(critical_alarm_dict, major_alarm_dict):
    """
    获取错误信息
    @param critical_alarm_dict: 紧急告警字典
    @param major_alarm_dict: 重要告警字典
    @return: 错误信息
    """
    err_msg = ""
    for alarm_id, num in critical_alarm_dict.items():
        # 紧急告警出现次数大于等于2次
        if num >= 2:
            err_msg += common.getMsg(LANG, "alarm.3times.repeat.Occur", (num, alarm_id))
    for alarm_id, num in major_alarm_dict.items():
        # 严重告警出现次数大于等于3次
        if num >= 3:
            err_msg += common.getMsg(LANG, "alarm.3times.repeat.Occur", (num, alarm_id))
    return err_msg


def get_alarm_num_dict(alarm_records):
    """
    获取告警次数字典
    @param alarm_records: 告警信息
    @return: 告警次数字典
    """
    alarm_dict = dict()
    for critical_alarm in alarm_records:
        alarm_id = critical_alarm.get("eventId")
        if alarm_id in alarm_dict:
            alarm_dict[alarm_id] += 1
            continue
        alarm_dict[alarm_id] = 1
    return alarm_dict


def get_critical_and_major_alarms(cli, end_time, start_time):
    """
    获取紧急和重要告警
    @param cli: cli
    @param end_time: 结束时间
    @param start_time: 开始时间
    @return: 紧急告警和重要告警
    """
    context = InspectContext(PY_JAVA_ENV, cli)
    inspect_context = context.get_context()
    inspect_context["devNode0"] = inspect_context.get("devInfo")
    critical_alarm_records = compute_storage_common.get_fcv_rest_record(
        inspect_context, CRITICAL_URL.format(start_time, end_time), "events")
    major_alarm_records = compute_storage_common.get_fcv_rest_record(
        inspect_context, MAJOR_URL.format(start_time, end_time), "events")
    LOGGER.logInfo("critical_alarm, major_alarm={}".format(critical_alarm_records, major_alarm_records))
    return critical_alarm_records, major_alarm_records


def get_timestamp(time_str):
    """将时间字符串转换为时间戳

    :param time_str: 时间格式为 2017-7-19
    :return:
    """
    time_array = time.strptime(time_str, "%Y-%m-%d")
    timestamp = int(time.mktime(time_array)) * 1000
    return timestamp
