# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import time
import json

import common
import dpa_common
import cliUtil
from java.lang import Exception as JException
import com.huawei.ism.tool.obase.entity.User as User
import com.huawei.ism.tool.obase.entity.DevNode as DevNode
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils

PY_JAVA_ENV = py_java_env
LANG = PY_JAVA_ENV.get("lang")
LOGGER = common.getLogger(PY_LOGGER, __file__)
PASSWORD_PROTECTION = "*********"
MASTER_NODE = "MASTER"
NO_FILE_FLAG = "no such file or directory"


def execute(dpa_rest):
    check_item = CheckItem(dpa_rest, LANG, LOGGER)
    return check_item.execute()


class CheckItem:
    def __init__(self, dpa_rest, lang, logger):
        self.dpa_rest = dpa_rest
        self.lang = lang
        self.logger = logger
        self.err_msg = str()
        self.cli_ret = list()
        self.upgrade_dir_node_list = list()
        self.node_info_list = list()

    @staticmethod
    def need_check_dev():
        cur_device_ip = PY_JAVA_ENV.get("devInfo").getIp()
        return [dev_node
                for dev_node in PY_JAVA_ENV.get('selectDevs')
                if dev_node.getIp() in list(PY_JAVA_ENV.get('dpaRelationshipHost').get(cur_device_ip))]

    @staticmethod
    def get_connection_by_ip(ip, port, name, pwd):
        java_dev = CheckItem.get_java_dev(ip, port, name, pwd)
        return PY_JAVA_ENV.get("sshManager").getSshConnection(java_dev)

    @staticmethod
    def get_java_dev(ip, port, name, pwd):
        java_dev = DevNode()
        dev = PY_JAVA_ENV.get("devInfo")
        java_dev.setSshForwardList(dev.getSshForwardList())
        java_dev.setSocks5Proxy(dev.getSocks5Proxy())
        java_dev.setPort(port)
        java_dev.setLoginUser(User(name, pwd))
        device_sn = ip + str(time.time())
        java_dev.setDeviceSerialNumber(device_sn)
        java_dev.setIp(ip)
        return java_dev

    @staticmethod
    def modify_prompts(connection):
        cmd = 'PS1=\"\\u:\\h:~$\"'
        cli_ret = connection.execCmd(cmd)
        cli_ret_list = cli_ret.encode("utf8").splitlines()
        return cli_ret_list[1]

    def execute(self):
        need_check_dev_list = CheckItem.need_check_dev()
        for dev_node in need_check_dev_list:
            connection = None
            try:
                dev_node = EntityUtils.cloneDevNode(dev_node)
                device_sn = dev_node.getIp() + str(time.time())
                dev_node.setDeviceSerialNumber(device_sn)
                connection = PY_JAVA_ENV.get("sshManager").getSshConnection(dev_node)
                self.modify_prompts(connection)
                node_info = self.get_node_info(connection)
                if node_info:
                    self.node_info_list.append(node_info)
                    self.upgrade_dir_node_list.append(dev_node.getIp())
            except JException as exp:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", dev_node.getIp())
                self.logger.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
            finally:
                PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
                self.logger.logInfo("SSH {} connection is release".format(dev_node.getDeviceSerialNumber()))
        if not self.node_info_list:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.upgrade.json.not.exist")
        elif len(self.node_info_list) > 1:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.upgrade.json.abnormal",
                                                   ",".join(self.upgrade_dir_node_list))
        if self.err_msg:
            return False, "\n".join(self.cli_ret), self.err_msg
        self.check_node_info_base_ssh(self.node_info_list[0])
        if not self.err_msg:
            return True, "\n".join(self.cli_ret), ""
        return False, "\n".join(self.cli_ret), self.err_msg

    def get_node_info(self, connection):
        cmd = "cat /opt/upgrade/CDMServerUpdate/upgrade.json"
        cli_ret = connection.execCmdNoLog(cmd)
        self.cli_ret.append(cmd)
        if NO_FILE_FLAG in cli_ret.lower():
            self.cli_ret.append("\n".join(cli_ret.splitlines()[-2:]))
            return {}
        self.cli_ret.append(cli_ret.splitlines()[-1])
        return json.loads(cli_ret.lstrip(cmd).rstrip(self.modify_prompts(connection)).encode("utf8"))

    def check_node_info_base_ssh(self, node_info_dict):
        for node in node_info_dict.get("nodes"):
            connection = None
            node_auth_type = int(node.get("auth_type"))
            node_ip = node.get("ssh_ip")
            node_port = int(node.get("ssh_port"))
            user_name = node.get("ssh_user")
            user_pwd = node.get("ssh_pwd")
            root_pwd = node.get("root_pwd")
            try:
                connection = self.get_connection_by_ip(node_ip, node_port, user_name, user_pwd)
                self.check_root_pwd(connection, node_ip, root_pwd)
                self.check_user_and_type(node_auth_type, node_ip, user_name)
                if node_info_dict.get("nodes").index(node) != 0:
                    continue
                self.check_master_meet_json(connection)
            except JException as exp:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", node_ip)
                self.logger.logNoPass("Create SSH {} connection catch exception:{}".format(node_ip, exp))
            finally:
                if connection:
                    PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
                node["root_pwd"] = PASSWORD_PROTECTION
                node["ssh_pwd"] = PASSWORD_PROTECTION
        self.cli_ret.append(str(node_info_dict))

    def check_master_meet_json(self, connection):
        cmd = "cat /opt/CDMServer/install.cfg"
        install_cfg_ret = connection.execCmd(cmd)
        self.cli_ret.append(install_cfg_ret)
        if "node-type=MASTER" not in install_cfg_ret:
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.json.format.abnormal")

    def check_user_and_type(self, node_auth_type, node_ip, user_name):
        if any([node_auth_type == 2 and user_name == "root", node_auth_type == 1 and user_name != "root"]):
            self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.json.context.abnormal", node_ip)

    def check_root_pwd(self, connection, node_ip, root_pwd):
        cmd = "su - root"
        change_user_ret = connection.execCmd(cmd)
        self.cli_ret.append(change_user_ret)
        if "password" in change_user_ret.lower():
            check_pwd_cli_ret = connection.execCmdNoLog(root_pwd)
            self.cli_ret.append(check_pwd_cli_ret)
            if "permission denied" in check_pwd_cli_ret.lower():
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.root.pwd.abnormal", node_ip)
