# -*- coding: UTF-8 -*-
import common
import cliUtil
import re
from cbb.frame.cli import cliUtil as cliUtils
from com.huawei.ism.tool.obase.exception import ToolException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

"""
@summary: 第三方存储（HDS）加入华为8G FC卡存储导致OOPS问题，Dorado V6和5x10新硬件不涉及。
"""

def execute(cli):
    allCliRet = ""
    
    try:
        # 获取异构链路信息中的lwwpn。
        lwwpnList, cliRet = getFCLinkLwwpns(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not lwwpnList:
            return True, allCliRet, ""

        # 获取存在于链路信息中的8G FC端口。
        fcPortListInLwwpnList, cliRet, nonexistent_port = \
            get8GFCPortInLwwpnList(cli, lwwpnList)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if not fcPortListInLwwpnList:
            return True, allCliRet, ""
        
        # 检查8G FC端口的MOR状态。
        flag, cliRet, errMsg = checkFcPortMorState(cli, fcPortListInLwwpnList)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag is True and nonexistent_port:
            errMsg = common.getMsg(LANG, "not_match_port",
                                   ",".join(nonexistent_port))
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        return (flag, allCliRet, errMsg)

    except common.UnCheckException as unCheckException:
        errMsg = unCheckException.errorMsg
        allCliRet = common.joinLines(allCliRet, unCheckException.cliRet)
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, allCliRet, common. \
            getMsg(LANG, "query.result.abnormal")
    finally:
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s".format(ret))
        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def getFCLinkLwwpns(cli):
    """
    @summary: 获取链路信息中的lwwpn。
    """
    lwwpnList = []
    
    cmd = "adm show link"
    flag, cliRet, errMsg = common.executeDebugCommand(cli, cmd, py_java_env, LOGGER)
    if flag is not True:
        raise common.UnCheckException(errMsg, cliRet)
    
    for line in cliRet.split("linkid"):
        lwwpn = getLwwpnFromLine(line)
        if lwwpn:
            lwwpnList.append(lwwpn)
    
    return lwwpnList, cliRet


def getLwwpnFromLine(line):
    """
    @summary:从一行字符串中获取lwwpn。 
          如果链路信息中存在“linktype(FC)”，说明设备有使用FC做异构。
    """
    if "linktype(FC)" not in line:
        return ""
    
    if "lwwpn(" not in line or ")" not in line:
        return ""

    pattern_lwwpn = re.compile(r'lwwpn\((.*?)\)',
                               flags=re.IGNORECASE)
    lwwpn = pattern_lwwpn.search(line)
    if not lwwpn:
        return ""
    return lwwpn.group(1)

def get8GFCIntf(cli):
    '''
    获取8G FC接口卡信息
    :param cli:
    :return:
    '''
    allCliRet = ""

    # 获取8G FC接口卡
    cmd = "show interface_module"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if flag is not True:
        raise common.UnCheckException(errMsg, cliRet)

    if cliUtil.queryResultWithNoRecord(cliRet):
        return [], allCliRet

    intfIdList = []
    infoDictList = cliUtils.getHorizontalNostandardCliRet(cliRet)
    for infoDict in infoDictList:

        status_Model = infoDict.get("Model", "")
        id = infoDict.get("ID", "")
        if status_Model.strip() == "4x8G FC Optical Interface Module":
            if id:
                intfIdList.append(id)
    return intfIdList, allCliRet

def get8GFCPortInLwwpnList(cli, lwwpnList):
    """
    @summary: 获取存在于链路信息中的8G FC端口。
    """
    allCliRet = ""
    portIdList = list()
    nonexistent_port = list()
    fc_port_wwn_list = list()


    # 获取8G FC 接口卡
    intfIdList, cliRet = get8GFCIntf(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)

    # 获取FC端口
    cmd = "show port general physical_type=FC"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if flag is not True:
        raise common.UnCheckException(errMsg, cliRet)

    if cliUtil.queryResultWithNoRecord(cliRet):
        return [], allCliRet
    # 8G FC接口卡上的FC端口即为8G FC端口
    infoDictList = cliUtils.getHorizontalNostandardCliRet(cliRet)
    for infoDict in infoDictList:
        portid = infoDict.get("ID", "")
        wwn = infoDict.get("WWN", "")
        fc_port_wwn_list.append(wwn)
        if wwn not in lwwpnList:
            continue

        for port_id in intfIdList:
            if portid.startswith(port_id):
                portIdList.append(portid)
        if portid in portIdList:
            continue

        if "H" not in portid:
            continue
        rate = infoDict.get("Max Speed(Mbps)", "")
        if rate.isdigit():
            if int(rate) == 8000:
                portIdList.append(portid)
                continue
        else:
            working_rate = infoDict.get("Working Rate(Mbps)", "")
            if working_rate.isdigit() and int(working_rate) <= 8000:
                portIdList.append(portid)
    # 检查adm show link命令提取到的lwwpn未在FC端口的结果中，说明端口被拔出
    for lwwpn in lwwpnList:
        if lwwpn not in fc_port_wwn_list:
            nonexistent_port.append(lwwpn)
    LOGGER.logInfo("portIdList:{}".format(portIdList))
    return portIdList, allCliRet, nonexistent_port
            
def getPortMorState(cli, port):
    """
    @summary: 获取端口的MOR状态。
    """
    cmd = "show port general port_id=%s" % port
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        return "", flag, cliRet, errMsg

    infoList = cliUtil.getVerticalCliRet(cliRet)

    for line in infoList:
        state = line.get("FC MOR State", "")
        return state, flag, cliRet, errMsg
    
    return "", flag, cliRet, errMsg

def checkFcPortMorState(cli, fcPortList):
    """
    @summary: 检查8G FC端口的MOR状态。
    """
    allMsg = ''
    allFlag = True
    allCliRet = ""
    errFcPortList = []
    abnormalFcPortList = []
    abnormalFcPortListNoOpen = []

    for fcPort in fcPortList:
        fcMorState, flag, cliRet, errMsg = getPortMorState(cli, fcPort)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag in [False,cliUtil.RESULT_NOCHECK]:
            allFlag = common.getResult(allFlag, flag)
            errFcPortList.append(fcPort)
            continue

        if fcMorState == "Open":
            abnormalFcPortList.append(fcPort)
        if not fcMorState:
            abnormalFcPortListNoOpen.append(fcPort)
    if abnormalFcPortList:
        allMsg += common.getMsg(LANG, "8GFC.port.mor.abnormal",
                                ",".join(list(set(abnormalFcPortList))))
    if abnormalFcPortListNoOpen:
        allMsg += common.getMsg(LANG, "8GFC.port.mor.abnormal.no.open",
                                ",".join(list(set(abnormalFcPortListNoOpen))))
    if errFcPortList:
        allMsg += common.getMsg(LANG, "8GFC.port.mor.err",
                                ",".join(list(set(errFcPortList))))
        return cliUtil.RESULT_NOCHECK, allCliRet, allMsg
    if allMsg:
        return False, allCliRet, allMsg

    return allFlag, allCliRet, allMsg

