# -*- coding: UTF-8 -*-

import cliUtil
import common
import config_1

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """ check smart IOC multi bond(PSA, France)
    :param cli:
    :return:
    """
    all_cli_ret = ""
    try:
        # 检查型号
        common.refreshProcess(PY_JAVA_ENV, 1, LOGGER)
        flag, cli_ret, err_msg = check_product_mode(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag in [cliUtil.RESULT_NOSUPPORT, cliUtil.RESULT_NOCHECK]:
            return flag, all_cli_ret, err_msg

        # 检查版本
        common.refreshProcess(PY_JAVA_ENV, 20, LOGGER)
        flag, cli_ret, err_msg = check_version_range(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag in [cliUtil.RESULT_NOSUPPORT, cliUtil.RESULT_NOCHECK]:
            return flag, all_cli_ret, err_msg

        # 检查绑定端口
        common.refreshProcess(PY_JAVA_ENV, 50, LOGGER)
        flag, cli_ret, err_msg = check_bond_port(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag in [cliUtil.RESULT_NOSUPPORT, cliUtil.RESULT_NOCHECK,
                    cliUtil.RESULT_WARNING]:
            return flag, all_cli_ret, err_msg

        return True, all_cli_ret, ""
    except Exception as exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, all_cli_ret,
                common.getMsg(LANG, "query.result.abnormal"))
    finally:
        common.refreshProcess(py_java_env, 100, LOGGER)


def check_product_mode(cli):
    """ 检查是否为涉及问题的型号
    :param cli:
    :return: 返回检查结果，cli回显，错误信息
    """

    # 只做显示使用
    cmd = "show system general"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    flag, product_mode, msg = cliUtil.getProductModel(cli, LANG)
    if flag is not True:
        return cliUtil.RESULT_NOCHECK, cli_ret, msg

    if common.isDorado(product_mode):
        return cliUtil.RESULT_NOSUPPORT, cli_ret, ""

    return True, cli_ret, ""


def check_version_range(cli):
    """ 检查是否为涉及问题的型号
    :param cli: ssh连接
    :return: 返回检查结果，cli回显，错误信息
    """
    cli_ret, sys_ver, patch_ver = common.\
        getHotPatchVersionAndCurrentVersion(cli, LANG)
    if not (sys_ver and patch_ver):
        err_msg = common.getMsg(LANG, "cannot.get.product.version.info")
        return cliUtil.RESULT_NOCHECK, cli_ret, err_msg

    # Dorado NAS V3系列，避免后续V6影响
    product_mode = common.getProductModeFromContext(PY_JAVA_ENV)
    if product_mode == "Dorado NAS" and sys_ver.startswith("V3"):
        return False, cli_ret, ""

    patch_info = config_1.BOND_VERSION_MAP.get(sys_ver, "")
    if patch_info:
        start_version = patch_info.get("start_version")
        if patch_ver == "--" or patch_ver < start_version:
            return False,  cli_ret, ""
        return cliUtil.RESULT_NOSUPPORT, cli_ret, ""

    if sys_ver.startswith(
            "V3") and "V300R003C10" <= sys_ver <= "V300R006C60":
        return False, cli_ret, ""

    if sys_ver.startswith(
            "V5") and "V500R007C00" <= sys_ver <= "V500R007C60SPC200":
        return False, cli_ret, ""

    return cliUtil.RESULT_NOSUPPORT, cli_ret, ""


def is_same_intf_port(port_list):
    """ 判断所有端口是否是同一接口模块的端口
    :param port_list: 端口位置列表
    :return: 返回结果
    """

    intf_set = set()
    for port in port_list:
        intf = port[:port.rfind(".")]
        intf_set.add(intf)
    if len(intf_set) > 1:
        return False
    return True


def get_bond_port(cli):
    """ 获取绑定端口信息
    :param cli: ssh连接
    :return:
    """
    checked_bond = {}
    cmd = "show bond_port"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        return False, checked_bond, cli_ret, err_msg

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, checked_bond, cli_ret, ""

    checked_bond = {}
    bond_port_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    for bond in bond_port_list:
        p_list = bond.get("Port ID List", "").split(",")
        if not is_same_intf_port(p_list):
            p_name = bond.get("Name")
            checked_bond[p_name] = p_list
    return True, checked_bond, cli_ret, ""


def get_vlan_port(cli):
    """ 获取vlan端口信息
    :param cli: ssh连接
    :return:
    """

    vlan_port_list = set()
    cmd = "show vlan general"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        return False, vlan_port_list, cli_ret, err_msg

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, vlan_port_list, cli_ret, ""

    vlan_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
    for vlan in vlan_list:
        vlan_port_list.add(vlan.get("Port ID"))

    return True, vlan_port_list, cli_ret, ""


def check_bond_port(cli):
    """ 检查绑定端口是否配置vlan
    :param cli: ssh连接
    :return:
    """

    all_ret = ""

    # 获取绑定端口信息
    flag, checked_bond, cli_ret, err_msg = get_bond_port(cli)
    all_ret = common.joinLines(all_ret, cli_ret)
    if not flag:
        return cliUtil.RESULT_NOCHECK, all_ret, err_msg

    if not checked_bond:
        return True, all_ret, ""

    # 获取vlan端口信息
    common.refreshProcess(py_java_env, 60, LOGGER)
    flag, vlan_port_list, cli_ret, err_msg = get_vlan_port(cli)
    all_ret = common.joinLines(all_ret, cli_ret)
    if not flag:
        return cliUtil.RESULT_NOCHECK, all_ret, err_msg

    if not vlan_port_list:
        return True, all_ret, ""

    # 检查绑定端口是否配置vlan
    flag = True
    err_bond = []
    for bond in checked_bond:
        if bond in vlan_port_list:
            flag = False
            err_bond.append(common.getMsg(LANG, "multi.bond.info",
                                          (bond, ",".join(checked_bond.get(
                                              bond)))))
    if not flag:
        msg = common.getMsg(LANG, "multi.bond.risk", "".join(err_bond))
        return cliUtil.RESULT_WARNING, all_ret, msg + get_sugg_msg()

    return True, all_ret, ""


def get_sugg_msg():
    p_version = str(PY_JAVA_ENV.get("devInfo").getProductVersion())
    if config_1.BOND_VERSION_MAP.get(p_version, ""):
        return common.getMsg(LANG, "multi.bond.check.sugg.install.patch",
                             (p_version, config_1.BOND_VERSION_MAP.get(p_version, "")))
    return common.getMsg(LANG, "multi.bond.check.sugg.no.patch")
