# -*- coding: UTF-8 -*-
import math
import re
import traceback
from cbb.business.operate.expansion.common import (
    get_internal_product_model_with_ret,
)
from cbb.frame.context import contextUtil
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from cbb.frame.cli import cliUtil as framecliUtil

import cliUtil
import common
import expandconfig
from common_utils import get_err_msg
from cli_util_cache import (
    get_all_enclosure_list_cache,
    get_license_cache,
)

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
SMART_TYPE_I = "_I"
SMART_TYPE_N = "_N"
SMART_ENCLOSURE_MAX_NUM = 20
ENCLOSURE_NVME_DISK_NUM = 36
ENCLOSURE_SSD_DISK_NUM = 25
EXPANSION_ENCLOSURE = "Expansion Enclosure"
ENGINE_ENCLOSURE = "Engine"
INNER_HYPER_LICENSE_ID = "103"
INNER_HYPER_ENGINE_NUM = 2
PATCH_VERSION_NUM = 2


def execute(cli):
    """
    检查新增智能硬盘框数量
    :param cli:
    :return:
    """
    enclosure_num_check = EnclosureNumCheck(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = enclosure_num_check.execute_check()
    return flag, "\n".join(enclosure_num_check.all_ret_list), msg


class EnclosureNumCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.err_msg = ""
        self.internal_product_mode = None
        self.enc_engine_num = 0
        self.enc_exp_num = 0
        self.current_disk_num = 0
        self.new_enclosure_num = 0
        self.product_version = str(env.get("devInfo").getProductVersion())
        self.patch_version = str(env.get("devInfo").getHotPatchVersion())

    def execute_check(self):
        try:
            flag, msg = self.get_all_information()
            if flag is not True:
                return flag, msg

            # 巡检场景
            if not self.is_expansion_inspection_scene():
                return self.check_enclosure_num_inspection()

            # 扩容控制器场景和扩容硬盘、硬盘框、硬盘柜场景。
            if not self.is_expansion_disk_or_ctrl_scene():
                return True, ""

            # 扩容评估场景
            ctrl_number = PY_JAVA_ENV.get("expCtrl")
            self.logger.logInfo(
                "ctrl_number:{}".format(ctrl_number))
            if ctrl_number and int(ctrl_number) > 4:
                if self.is_inner_hyper():
                    return cliUtil.RESULT_NOSUPPORT, ""
            prompt_version = self.patch_version or self.product_version
            # 若待扩智能硬盘框数量大于20；
            if self.check_enclosure_num_expansion():
                err_key = "exp.smart.enclosure.num.expansion.601.has.patch"
                return False, get_err_msg(self.lang, err_key, prompt_version)
            # 若引擎为2，
            if self.enc_engine_num == INNER_HYPER_ENGINE_NUM:
                return self.check_enclosure_num_on_2_engine()
            # 若引擎为1
            # 扩后引擎为2,当前或者扩容后智能硬盘框数量大于12，且版本小于6.0.1.SPH6，则检查不通过
            if ctrl_number and int(ctrl_number) > 4:
                return self.check_enclosure_num_on_2_engine()

            else:
                # 扩容后引擎数为1，当前或者扩容后智能硬盘框数量大于20，且版本小于6.0.1.SPH2，则检查不通过
                return self.check_enclosure_num_on_1_engine()
        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def get_all_information(self):
        """
        巡公共流程，获取信息
        :return:
        """
        if not self.is_risk_version():
            self.logger.logInfo("not risk version!")
            return cliUtil.RESULT_NOSUPPORT, ""

        if not self.is_risk_model():
            self.logger.logInfo("not risk mode!")
            return cliUtil.RESULT_NOSUPPORT, ""

        if not self.is_smart_enclosure():
            self.logger.logInfo("not smart enclosure!")
            return cliUtil.RESULT_NOSUPPORT, ""

        # 获取控制框和硬盘框数量
        self.enc_engine_num, self.enc_exp_num = self.get_all_enclosure_info()
        self.logger.logInfo(
            "engine num is:{}, expansion num is {}".format(
                self.enc_engine_num, self.enc_exp_num
            )
        )
        # 获取系统硬盘数量
        self.current_disk_num = self.get_disk_info()
        self.logger.logInfo("disk num is:{}".format(self.current_disk_num))

        # 引擎数为2，存在内双活，则不涉及
        if (
                self.is_inner_hyper()
                and self.enc_engine_num == INNER_HYPER_ENGINE_NUM
        ):
            return cliUtil.RESULT_NOSUPPORT, ""

        return True, ""

    def check_enclosure_num_on_1_engine(self):
        """
        当前或者扩容后智能硬盘框数量大于20，且版本小于6.0.1.SPH2，则检查不通过，否则继续检查；
        :return:
        """
        prompt_version = (
            self.patch_version if self.patch_version else self.product_version
        )
        if all([any([self.enc_exp_num > SMART_ENCLOSURE_MAX_NUM,
                     (self.new_enclosure_num + self.enc_exp_num) >
                     SMART_ENCLOSURE_MAX_NUM]),
                self.check_patch_bigger_than_target(PATCH_VERSION_NUM)]):
            err_key = "exp.smart.enclosure.num.600"
            return False, get_err_msg(self.lang, err_key, prompt_version)
        return True, ""

    def check_enclosure_num_on_2_engine(self):
        """
        当前或者扩容后智能硬盘框数量大于12，且版本小于6.0.1.SPH6，则检查不通过，否则继续检查；
        :return:
        """
        prompt_version = (
            self.patch_version if self.patch_version else self.product_version
        )
        if all([any([self.enc_exp_num > 12,
                     (self.new_enclosure_num + self.enc_exp_num) > 12]),
                self.check_patch_bigger_than_target(6)]):
            err_key = "exp.smart.enclosure.num.12"
            return False, get_err_msg(self.lang, err_key, prompt_version)
        return True, ""

    def is_expansion_inspection_scene(self):
        # 先确定是否是扩容场景巡检--扩容前评估，区分巡检和扩容评估
        # is_expansion_disk_or_ctrl_scence再确定子场景是否为扩控和框场景。
        scene_data = self.env.get("sceneData")
        self.logger.logInfo("scene data :{}".format(scene_data))
        if not scene_data:
            return False
        return (
                scene_data.get("mainScene") == "Expansion"
                and scene_data.get("toolScene") == "perInspector"
        )

    def is_expansion_disk_or_ctrl_scene(self):
        """
        扩容控制器场景和扩容硬盘、硬盘框、硬盘柜场景。
        :return:
        """
        scene_data = self.env.get("sceneData")
        if not scene_data:
            return False

        self.logger.logInfo("scene data :{}".format(scene_data))
        return all(
            [
                scene_data.get("mainScene") == "Expansion",
                scene_data.get("toolScene") == "perInspector",
                any(
                    [
                        scene_data.get("subScene") == "Expansion Controller",
                        self.env.get("expMode") == "EXTEND_CTRL",
                        scene_data.get("subScene") == "Expansion Disk",
                        self.env.get("expMode") == "EXTEND_DISK",
                    ]
                ),
            ]
        )

    def check_enclosure_num_inspection(self):
        """
        巡检场景
        :return:
        """
        prompt_version = (
            self.patch_version if self.patch_version else self.product_version
        )
        # 引擎数为2，智能硬盘框数量大于12，且版本小于6.0.1.SPH6，则检查不通过，否则继续检查；
        if self.enc_engine_num == INNER_HYPER_ENGINE_NUM:
            err_key = "exp.smart.enclosure.num.12.inspect"
            if self.enc_exp_num > 12 and self.check_patch_bigger_than_target(6):
                return False, get_err_msg(self.lang, err_key, prompt_version)

        # 引擎数为1，智能硬盘框数量大于20，且版本小于6.0.1.SPH2，则检查不通过，否则继续检查；
        if self.enc_engine_num == 1:
            err_key = "exp.smart.enclosure.num.20.inspect"
            if self.enc_exp_num > 20 and \
                    self.check_patch_bigger_than_target(PATCH_VERSION_NUM):
                return False, get_err_msg(self.lang, err_key, prompt_version)

        return True, ""

    def check_patch_bigger_than_target(self, target_ver):
        """
        检查补丁是否比目标版本大
        :param target_ver:
        :return:
        """
        if self.product_version.startswith("6.0.0"):
            return True
        if not self.patch_version:
            return True
        p_patch_reg = re.compile(r"SPH(\d+)")
        res = p_patch_reg.search(self.patch_version)
        if res:
            p_patch = res.group(1)
        else:
            p_patch = 0

        return int(p_patch) < target_ver

    def is_inner_hyper(self):
        """
        是否存在内双活license
        :return:True: 内双活，False: 非内双活
        """
        __, ret, msg = get_license_cache(self.cli, self.env, self.logger)
        self.all_ret_list.append(ret)
        regx = re.compile(r"Feature ID\s*:\s*(\d*)")
        res_list = regx.findall(ret)
        return INNER_HYPER_LICENSE_ID in res_list

    def is_smart_enclosure(self):
        """
        判断是否是智能框 D18800V6_I
        :return:True: 智能框，False: 非智能框
        """
        context = contextUtil.getContext(PY_JAVA_ENV)
        (
            internal_product_mode,
            cli_ret,
        ) = get_internal_product_model_with_ret(context)
        self.all_ret_list.append(cli_ret)
        rest = contextUtil.getRest(context)
        rest.close()
        self.internal_product_mode = internal_product_mode
        if (
                SMART_TYPE_I in internal_product_mode
                or SMART_TYPE_N in internal_product_mode
        ):
            return True

        return False

    def is_risk_model(self):
        """
        判断是否只风险型号
        :return: True: 风险，False: 无风险
        """
        flag, product_model, cli_ret, msg = cliUtil.getProductModelWithCliRet(
            self.cli, self.lang
        )
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(msg, cli_ret)
        self.logger.logInfo("product_model {}".format(product_model))
        return product_model in DORADO_DEVS_V6_HIGH

    def is_risk_version(self):
        """
        判断是否为风险版本
        :return: True: 风险，False: 无风险
        """
        flag, software_version, hotpatch_version = \
            framecliUtil.getSystemVersion(self.cli, self.lang)
        self.logger.logInfo("product_version {}".format(software_version))
        self.logger.logInfo("hotpatch_version {}".format(hotpatch_version))
        return software_version in ["6.0.0", "6.0.1"]

    def get_disk_info(self):
        """
        获取系统硬盘数量
        :return:
        """
        flag, cli_ret, disk_info_list, msg = common.getDiskList(self.cli, LANG)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(msg, cli_ret)

        return len(disk_info_list)

    def get_all_enclosure_info(self):
        """
        获取控制框和硬盘框
        :return:控制框和硬盘框
        """
        flag, ret, enclosure_list, msg = get_all_enclosure_list_cache(
            self.cli, self.env, self.logger
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)

        enc_engine_list = list(
            filter(
                lambda enc: enc.get("Logic Type") == ENGINE_ENCLOSURE,
                enclosure_list,
            )
        )
        enc_exp_list = list(
            filter(
                lambda enc: enc.get("Logic Type") == EXPANSION_ENCLOSURE,
                enclosure_list,
            )
        )
        return len(enc_engine_list), len(enc_exp_list)

    def check_enclosure_num_expansion(self):
        """
        计算待扩硬盘框数量是否超过20
        :return: True: 超过20， False: 未超过20
        """
        enclosure_disk_num = (
            ENCLOSURE_NVME_DISK_NUM
            if SMART_TYPE_N in self.internal_product_mode
            else ENCLOSURE_SSD_DISK_NUM
        )
        self.logger.logInfo("enclosure disk num {}".format(enclosure_disk_num))
        exp_disk_list = expandconfig.ExpandConfig(
            PY_JAVA_ENV
        ).getAllExpDiskList()
        # 计算扩容的硬盘总数
        exp_disk_num = sum(
            [disk_info.getDiskNum() for disk_info in exp_disk_list]
        )
        self.logger.logInfo(
            "ready disk:{}, cur disk:{}, enc exp num:{}".format(
                exp_disk_num, self.current_disk_num, self.enc_exp_num
            )
        )
        enclosure_free_slot = (
                enclosure_disk_num * self.enc_exp_num - self.current_disk_num
        )
        # 计算出的新增扩容硬盘框
        self.new_enclosure_num = math.ceil(
            float(exp_disk_num - enclosure_free_slot) / enclosure_disk_num
        )
        self.logger.logInfo(
            "enc free slot is :{}, new enc num is:{}".format(
                enclosure_free_slot, self.new_enclosure_num
            )
        )
        return int(self.new_enclosure_num) > SMART_ENCLOSURE_MAX_NUM
