# -*- coding: UTF-8 -*-
import cliUtil
import common
import traceback
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
    远距离双活fast write优化检查： 
    1、若步骤3与步骤5中存在相同端口ID，则检查结果为建议优化。
    2、若步骤6中存在“Fast Write Enabled”字段值为“NO”的iSCSI链路，则检查结果为建议优化。
    3、其他情况，检查结果为通过。

    '''
    fcResult = True
    iscsiResult = True
    errMsg = ""
    allCliRet = "" 
    
    try:
        #步骤2，获取回显中"FC Link"与"ISCSI Link"的 Remote Device ID
        checkResult, fcLinkIdList, iscsiLinkIdList = getFcAndIscsiLinkId(cli)
        allCliRet = common.joinLines(allCliRet, checkResult[1])
        if checkResult[0] != True:
            return checkResult
        
        LOGGER.logInfo("fcLinkIdList = %s, iscsiLinkIdList = %s" % (fcLinkIdList, iscsiLinkIdList))
        #检查FC链路fast write 开关是否打开
        if fcLinkIdList:
            LOGGER.logInfo("check FC Link fast Write Switch start")
            fcResult, cliRet, errInfo = checkFastWriteForFcLink(cli, fcLinkIdList)
            allCliRet = common.joinLines(allCliRet, cliRet)
            if fcResult != True:
                errMsg += errInfo
        
        #检查FC链路fast write 开关是否打开
        if iscsiLinkIdList:
            LOGGER.logInfo("check ISCSI Link fast Write Switch start")
            iscsiResult, cliRet, errInfo = checkFastWriteForIscsiLink(cli, iscsiLinkIdList)
            allCliRet = common.joinLines(allCliRet, cliRet)
            if iscsiResult != True:
                errMsg += errInfo 
        
        #FC链路或ISCSI链路均为未检查，结果为未检查        
        if fcResult == cliUtil.RESULT_NOCHECK or iscsiResult == cliUtil.RESULT_NOCHECK:
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        #FC链路或ISCSI链路均为建议优化，结果为建议优化 
        if fcResult == cliUtil.RESULT_WARNING or iscsiResult == cliUtil.RESULT_WARNING:
            return (cliUtil.RESULT_WARNING, allCliRet, errMsg)
        
        return (True, allCliRet, errMsg)
    except Exception:
        LOGGER.logError("check_item_service_fc_fastwrite_check traceback:" + str(traceback.format_exc()))
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))


def getFcAndIscsiLinkId(cli):
    '''
    @summary: 获取风险FC Link和ISCSI Link的ID
    @param cli: cli对象
    '''
    fc_link_id_list = []
    iscsi_link_id_list = []
    cmd = "show remote_device link"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True,
                                                        LANG)
    # 检查命令权限，若无权限 返回不涉及
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return (cliUtil.RESULT_NOSUPPORT, cli_ret,
                err_msg), fc_link_id_list, iscsi_link_id_list

    if flag is not True:
        return cliUtil.RESULT_NOCHECK, cli_ret, err_msg, \
               fc_link_id_list, iscsi_link_id_list

    # 提取出FC及ISCSI的相应回显
    fc_link_index = cli_ret.find("FC Link")
    iscsi_link_index = cli_ret.find("ISCSI Link")
    ip_link_index = cli_ret.find("IP Link")
    if fc_link_index == -1:
        err_msg = common.getMsg(LANG, "remote.device.idinfomation")
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                err_msg), fc_link_id_list, iscsi_link_id_list
    fc_link_info_dict_list = []
    iscsi_link_info_dict_list = []
    if iscsi_link_index != -1 and ip_link_index == -1:
        fc_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[fc_link_index:iscsi_link_index])
        iscsi_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[iscsi_link_index:])
    elif iscsi_link_index != -1 and ip_link_index != -1:
        fc_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[fc_link_index:iscsi_link_index])
        iscsi_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[iscsi_link_index:ip_link_index])
    elif iscsi_link_index == -1 and ip_link_index != -1:
        fc_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[fc_link_index:ip_link_index])
    elif iscsi_link_index == -1 and ip_link_index == -1:
        fc_link_info_dict_list = cliUtil.getHorizontalNostandardCliRet(
            cli_ret[fc_link_index:])
    else:
        return (cliUtil.RESULT_NOCHECK,
                cli_ret), err_msg, fc_link_id_list, iscsi_link_id_list

    # 获取风险FC链路ID
    for fc_link_info_dict in fc_link_info_dict_list:
        fc_link_id = fc_link_info_dict.get("ID", "")
        remote_device = fc_link_info_dict.get("In Remote Device", "")
        if remote_device.upper() == "YES":
            fc_link_id_list.append(fc_link_id)

    # 获取风险ISCSI链路ID
    for iscsi_link_info_dict in iscsi_link_info_dict_list:
        iscsi_link_id = iscsi_link_info_dict.get("ID", "")
        remote_device = iscsi_link_info_dict.get("In Remote Device", "")
        if remote_device.upper() == "YES":
            iscsi_link_id_list.append(iscsi_link_id)

    return (True, cli_ret, err_msg), fc_link_id_list, iscsi_link_id_list

def checkFastWriteForFcLink(cli, fcLinkIdList):
    '''
    @summary: 检查FC链路的Fast Write开关是否打开
    @param cli: cli对象
    @param fcLinkIdList: 风险FC链路ID列表
    '''
    flag = True
    errMsg = ""
    allCliRet = ""
    
    #获取 “Fast Write Enable”字段值为“NO”时的“Local Port ID”列表
    localPortIdList = []
    for fcLinkId in fcLinkIdList:
        cmd = "show remote_device link link_type=FC link_id=%s" % fcLinkId
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        allCliRet = common.joinLines(allCliRet, cliRet) 
        if flag != True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        fcLinkInfoDictList = cliUtil.getVerticalCliRet(cliRet)
        for fcLinkInfoDict in fcLinkInfoDictList:
            fastWriteEnable = fcLinkInfoDict.get("Fast Write Enable", "")
            localPortId = fcLinkInfoDict.get("Local Port ID", "")
            if fastWriteEnable.upper() == "NO":
                localPortIdList.append(localPortId)
                
    #获取FC端口的ID列表
    cmd = "show port general physical_type=FC"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if flag != True:
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
    
    fcPortIdList = []
    fcPortInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    for fcPortInfoDict in fcPortInfoDictList:
        fcPortId = fcPortInfoDict.get("ID", "")
        if fcPortId:
            fcPortIdList.append(fcPortId)
    
    portIdList = []
    for fcPortId in fcPortIdList:
        cmd = "show port general port_id=%s" % fcPortId
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        portInfoDictList = cliUtil.getVerticalCliRet(cliRet)            
        for portInfoDict in portInfoDictList:
            fastWriteSupported = portInfoDict.get("Fast Write Supported", "")
            id = portInfoDict.get("ID", "")
            if fastWriteSupported.upper() == "YES":
                portIdList.append(id)
    
    #判断是否有Fast Write开关未打开：取支持fast write开关端口列表和fast write开关关闭的端口列表交集
    riskPortIdList = list(set(localPortIdList)&set(portIdList))
    
    if riskPortIdList:
        flag = cliUtil.RESULT_WARNING
        err_msg = common.getMsg(
            LANG, "fc.link.fast.write.error", ",".join(riskPortIdList)
        ) + common.getMsg(LANG, "fc.link.fast.write.error.sugg")
        return flag, allCliRet, err_msg
    return True, allCliRet, ""
          
def checkFastWriteForIscsiLink(cli, iscsiLinkIdList):
    '''
    @summary: 检查ISCSI链路的Fast Write开关是否打开
    @param cli: cli对象
    @param iscsiLinkIdList: 风险ISCSI链路ID列表
    '''
    flag = True
    err_msg = ""
    allCliRet = ""
    iscsiNotLinkIdList = []
    
    for iscsiLinkId in iscsiLinkIdList:
        cmd = "show remote_device link link_type=iSCSI link_id=%s" % iscsiLinkId
        flag, cliRet, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        allCliRet = common.joinLines(allCliRet, cliRet) 
        if flag is not True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, err_msg)
        
        iscsiLinkInfoDictList = cliUtil.getVerticalCliRet(cliRet)
        for iscsiLinkInfoDict in iscsiLinkInfoDictList:
            fastWriteEnable = iscsiLinkInfoDict.get("Fast Write Enabled", "")
            if fastWriteEnable.upper() == "NO":
                iscsiNotLinkIdList.append(iscsiLinkId)
    
    if iscsiNotLinkIdList:
        flag = cliUtil.RESULT_WARNING
        err_msg = common.getMsg(LANG, "iscsi.link.fast.write.error", ",".join(iscsiNotLinkIdList))
        err_msg += common.getMsg(LANG, "iscsi.link.fast.write.error.sugg")
    LOGGER.logInfo("iscsiNotLinkIdList = %s" %  iscsiNotLinkIdList)    
    return flag, allCliRet, err_msg
    