# -*- coding:utf-8 -*-
import traceback
import re
import time

import common
import common_utils
import cliUtil
from common import UnCheckException
from common import AsynProgress

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ENV = py_java_env
LOGIC_TYPE = 'Expansion Enclosure'
TYPE = '2U 25 Slot 2.5 SSD Disks Enclosure'


def execute(cli):
    """
    检查appdata进程D状态
    """
    check_appdata_status = CheckAppdataStatus(
        cli, LANG, LOGGER, ENV)
    flag, err_msg = check_appdata_status.execute_check()
    return flag, "\n".join(check_appdata_status.all_cli_ret), err_msg


class CheckAppdataStatus:

    def __init__(self, cli, lang, logger, env):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.env = env
        self.all_cli_ret = []
        self.error_msg = ''

    def execute_check(self):
        count = 0
        max_checks_number = 7
        my_pthread = AsynProgress(self.env, self.logger)
        my_pthread.start_thread()
        try:
            flag, err_msg = self.get_enclosure()
            if flag is True:
                return (flag, '')
            if flag is False:
                return (flag, err_msg)

            while count < max_checks_number:
                appdata_flag = self.get_app_data()
                if appdata_flag is True:
                    return (True, '')
                count += 1
                if count < max_checks_number:
                    time.sleep(10)
            return (cliUtil.RESULT_WARNING, self.error_msg)

        except UnCheckException as unCheckException:
            self.logger.logInfo("UnCheckException, err_msg: %s" % unCheckException.errorMsg)
            # unCheckException.cliRet
            if not unCheckException.flag:
                return cliUtil.RESULT_NOCHECK, unCheckException.errorMsg
            else:
                return unCheckException.flag, unCheckException.errorMsg
        except Exception as exception:
            self.logger.logException(exception)
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, "query.result.abnormal")
        finally:
            # 退出到cli模式
            my_pthread.setStopFlag(True)
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.logger.logInfo("enter cli mode from some model ret is {}".format(ret))

            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)

    def get_enclosure(self):
        '''
        执行命令：show enclosure
        :return:是否存在框设备信息，若存在框设备信息，返回框设备信息
        '''

        self.logger.logInfo("Prepare to view the device box.")
        cmd = "show enclosure"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.lang)
        self.all_cli_ret.append(cli_ret)

        # 判断是否含异常场景
        if not flag:
            self.logger.logInfo("Failed to get enclosure info, errMsg:%s" % err_msg)
            raise UnCheckException(common.getMsg(self.lang, "query.result.abnormal"),
                                   cli_ret, flag)

        # 不存在硬盘框通过
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return (True, '')

        enclosure_info_dcts = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for enclosure_info_dct in enclosure_info_dcts:
            enclosure_id = enclosure_info_dct.get('ID', '')
            logicType = enclosure_info_dct.get('Logic Type', '')
            enclosure_type = enclosure_info_dct.get('Type', '')
            # 硬盘框为问题框继续检查(如果存在Logic Type字段是“Expansion Enclosure”，且Type字段是“2U 25 Slot 2.5 SSD Disks Enclosure”)
            if logicType == LOGIC_TYPE and enclosure_type == TYPE:
                self.logger.logInfo(
                    "need check whether the appdata process is in the D state."
                    "not pass disk enclosure,ID:{},Logic Type:{},Type:{}.".format(enclosure_id, logicType,
                                                                                  enclosure_type))
                return (cliUtil.RESULT_WARNING, err_msg)
        return (True, '')

    # 执行命令：ps -aux，查看系统当前进程信息。(检查appdata进程是否处于D状态,STAT字段为"D",COMMAND字段包含"app_data")
    def get_app_data(self):
        """
        执行命令：ps -aux

        :return:是否处于d状态，及cmd回显信息
        """
        self.logger.logInfo("Prepare to check whether the appdata process is in the D state.")
        cmd = "ps -aux"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInMinisystemModel(self.cli, cmd, self.lang)
        self.all_cli_ret.append(cli_ret)
        # 判断是否含异常场景
        if not flag:
            self.logger.logInfo("Failed to get system processes, err_msg:%s" % err_msg)
            raise UnCheckException(common.getMsg(self.lang, "cannot.get.info", "processes"),
                                   cli_ret, flag)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True

        processes_info_dcts = getHorizontalNostandardCliRet(cli_ret)
        for processes_info_dct in processes_info_dcts:
            stat = processes_info_dct.get('STAT', '')
            command = processes_info_dct.get('COMMAND', '')
            if stat == 'D' and 'app_data' in command:
                self.logger.logInfo(str(processes_info_dct))
                self.error_msg = common_utils.get_err_msg(self.lang, "check.appdata.status.is.d")
                return cliUtil.RESULT_WARNING
        return True


def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,此方法用来解析CLI回显未对其情况
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    emptylist = list()
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            i += 1
            reg_headline = re.compile("^\s*.+(\s+.+)*\s*$")
            match_headline = reg_headline.search(line)
            if i == 2:
                headline = match_headline.group()
                break
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        field_words = cliRetList[(i):]
        reg_split = re.compile("\s*\S+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = get_keys(cliRetList[i - 1], tuple_idxs)
        lenkeys = len(keys)
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0:
                break
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            if len(line.strip()) == 0:
                continue
            valueSpaceList, vals = get_lenvalue(line)
            lenvalue = len(valueSpaceList)
            if lenvalue == lenkeys:
                dictList.append(dict(zip(keys, vals)))
        return dictList
    except UnCheckException as unCheckException:
        LOGGER.logError(str(traceback.format_exc()))
        LOGGER.logInfo("UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        return emptylist
    except Exception as exception:
        LOGGER.logException(exception)
        return emptylist


def get_lenvalue(line):
    vals = []
    count = 0
    command_value = ""
    valueList = line.strip().split(" ")
    valueSpaceList = []
    for value in valueList:
        if value != "":
            count += 1
            if count < 11:
                vals.append(value.strip().decode("utf8"))
                valueSpaceList.append(value)
            else:
                command_value += value
    vals.append(command_value.strip().decode("utf8"))
    valueSpaceList.append(command_value)
    return (valueSpaceList, vals)


def get_keys(title, tuple_idxs):
    keys = []
    for item in tuple_idxs:
        key = title[item[0]:item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key.decode("utf8"))
    return keys
